/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "callback_func.h"
#include "detect.h"
#include "display.h"
#include "interface.h"
#include "shared.h"
#include "utils.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../Hermes2/interface.h"
#include "../rsid/rsid_modes.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//------------------------------------------------------------------------

// All callback functions of the Hellschreiber client

  void
on_hell_window_destroy(
    GObject  *object,
    gpointer  user_data )
{
  Hell_Cleanup();
}

//------------------------------------------------------------------------

  gboolean
on_hell_window_delete(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  Save_Window_Position( hell_gui.window, hell_rc_data.rc_fpath );
  Guest_Quit_Activate( hell_gui.window );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_hell_quit_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  Save_Window_Position( hell_gui.window, hell_rc_data.rc_fpath );
  Guest_Quit_Activate( hell_gui.window );
}

//------------------------------------------------------------------------

  gboolean
on_hell_tx_textview_key_press(
    GtkWidget   *widget,
    const GdkEventKey *event,
    gpointer     user_data )
{
  if( Flag[GUEST_TRANSMIT_MODE] )
    return( Hell_Tx_Textview_Key_Press(event) );
  else return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_hell_tx_textview_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data )
{
  return( Hell_Gui_Button_Press(event) );
}

//------------------------------------------------------------------------

  void
on_hell_transmit_clicked(
    GtkButton *button,
    gpointer   user_data )
{

  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  const Transceiver_t *TRx       = Transceiver[Indices.TRx_Index];

 // Default config parameters
  Strlcpy( hermes2_rc.pcm_device, "hw:1,0", sizeof(hermes2_rc.pcm_device) );
  if( !TRx->receive_active || !ddv->tx_pa_enable )
    return;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );
  Hell_Change_Modes();
}

//------------------------------------------------------------------------

  void
on_hell_receive_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus( Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );
  Hell_Receive_Clicked();
}

//------------------------------------------------------------------------

  void
on_hell_rx_drawingarea_configure(
    GtkWidget *widget,
    GdkEventConfigure *event,
    gpointer user_data )
{
  hell_rc_data.rxarea_width  = (uint16_t)event->width;
  hell_rc_data.rxarea_height = (uint16_t)event->height;
  if( !Hell_Create_Rx_Pixbuf() )
  {
    Error_Dialog( _("Failed to create pixbuf for Rx window"), HIDE_OK );
    return;
  }

  Flag[GUEST_CLEAR_RX_WINDOW] = True;
  Flag[HELL_DAREA_CONFIG]     = True;
}

//------------------------------------------------------------------------

  gboolean
on_hell_rx_drawingarea_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  if( hell_receive_pixbuf.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, hell_receive_pixbuf.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
    return( TRUE );
  }
  return( FALSE );
}

//------------------------------------------------------------------------

  gboolean
on_hell_rx_drawingarea_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data )
{
  Hell_Gui_Button_Press( event );
  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_hell_deskew_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  Hell_Save_Pixbuf();
  Hell_Clear_Pixbuf( &hell_receive_pixbuf, 0xd0d0d0ff );
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    Flag[HELL_AUTO_DESKEW] = True;
    hell_rc_data.num_rows = 1;
  }
  else
  {
    Flag[HELL_AUTO_DESKEW] = False;
    hell_rc_data.num_rows = 2;
  }

  Flag[GUEST_CLEAR_RX_WINDOW] = True;
  g_idle_add( Hell_Set_TxRx_Labels, NULL );
}

//------------------------------------------------------------------------

  void
on_hell_height_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( (hell_gui.drawingarea == NULL) ||  Flag[GUEST_RECEIVING] || Flag[GUEST_TRANSMITTING] )
    return;

  Hell_Save_Pixbuf();
  Hell_Clear_Pixbuf( &hell_receive_pixbuf, 0xd0d0d0ff );
  Flag[GUEST_CLEAR_RX_WINDOW] = True;
  Flag[HELL_DAREA_CONFIG]     = False;
  if( gtk_toggle_button_get_active(togglebutton) )
    gtk_widget_set_size_request(
        hell_gui.drawingarea, hell_rc_data.rxarea_width, SHORT_HEIGHT );
  else
    gtk_widget_set_size_request(
        hell_gui.drawingarea, hell_rc_data.rxarea_width, NORMAL_HEIGHT );
}

//------------------------------------------------------------------------

  void
on_hell_contrast_value_changed(
    GtkRange *range,
    gpointer  user_data )
{
  GtkAdjustment *adjust;

  // Read and enter contrast value
  adjust = gtk_range_get_adjustment( range );
  hell_rc_data.contrast = (uint8_t)( ELEMENT_MAX + 1 -
      gtk_adjustment_get_value(adjust) * (ELEMENT_MAX / 10.0) );
}

//------------------------------------------------------------------------

  gboolean
on_hell_scope_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  Hell_Display_Signal( cr );
  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_hell_wfall_draw(
    GtkWidget *widget,
    cairo_t   *cr,
    gpointer   user_data )
{
  if( hell_wfall.pixbuf != NULL )
  {
    gdk_cairo_set_source_pixbuf( cr, hell_wfall.pixbuf, 0.0, 0.0 );
    cairo_paint( cr );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  gboolean
on_hell_scope_button_press(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        user_data )
{
  if( event->button == 1 )
  {
    if( Flag[GUEST_RECEIVING] )
      Tune_to_Monitor(
          event->x, CLICK_TUNE_RANGE, hell_wfall.width,
          HELL_AUDIO_FREQUENCY, &hell_ifft_data );
  }

  return( TRUE );
}

//------------------------------------------------------------------------

  void
on_hell_callsign_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Callsign_Changed( editable, &qso_record, "HELL" );
}

//------------------------------------------------------------------------

  void
on_hell_rst_out_changed(
    GtkEditable *editable,
    gpointer    user_data )
{
  Process_RST( editable );
  Strlcpy( qso_record.dx_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.dx_rst) );
}

//------------------------------------------------------------------------

  void
on_hell_rst_in_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Process_RST( editable );
  Strlcpy( qso_record.my_rst,
      gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(qso_record.my_rst) );
}

//------------------------------------------------------------------------

  void
on_hell_op_name_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field QSO record structure
  Strlcpy( qso_record.dx_name, buff, sizeof(qso_record.dx_name) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_name );
}

//------------------------------------------------------------------------

  void
on_hell_qth_name_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  char buff[13];

  if( !Get_Record_Field(buff, sizeof(buff), editable) )
    return;

  // Enter data to field and QSO record structure
  Strlcpy( qso_record.dx_qth, buff, sizeof(qso_record.dx_qth) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_record.dx_qth );
}

//------------------------------------------------------------------------

  void
on_hell_locator_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Locator_Changed( editable, &qso_record, "PSK31" );
}

//------------------------------------------------------------------------

  void
on_hell_band_changed(
    GtkEditable *editable,
    gpointer     user_data )
{
  Band_Changed( editable, &qso_record, "PSK31" );
}

//------------------------------------------------------------------------

  void
on_hell_new_record_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  New_Record( &qso_record, hell_gui.window_builder, "HELL" );
  Strlcpy( qso_record.mode_dir, "hell", sizeof(qso_record.mode_dir) );
}

//------------------------------------------------------------------------

  void
on_hell_save_record_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  // Save record if needed(and valid)
  if( Read_QSO_Record(&qso_record, hell_gui.window_builder, "HELL") )
    Guest_Save_Dialog( _("Save QSO Record to file?") );
  else Guest_Save_Dialog(
      _("QSO Record is incomplete\n"\
        "Do you still want to save?") );
}

//------------------------------------------------------------------------

  void
on_hell_clear_fields_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  Clear_Record_Fields( True, hell_gui.window_builder, "HELL" );
}

//------------------------------------------------------------------------

  void
on_hell_clear_windows_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  Hell_Save_Pixbuf();
  Hell_Clear_Pixbuf( &hell_receive_pixbuf, 0xd0d0d0ff );
  Flag[GUEST_CLEAR_RX_WINDOW] = True;
  gtk_text_buffer_set_text(hell_gui.tx_text_buffer, "", -1);
  gtk_text_buffer_get_iter_at_offset(
      hell_gui.tx_text_buffer, &hell_gui.tx_text_buffer_iter,
      gtk_text_buffer_get_char_count(hell_gui.tx_text_buffer) );
  gtk_widget_grab_focus(
      Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );
}

//------------------------------------------------------------------------

  void
on_hell_macro_clicked(
    GtkButton *button,
    gpointer   user_data )
{
  Hell_Select_Macro( button );
}

//------------------------------------------------------------------------

  void
on_hell_feldhell_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  Flag[HELL_MODE_FELDHELL] = True;
  Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_FELDHELL;
  New_Parameters();
}

//------------------------------------------------------------------------

  void
on_hell_fmhell_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  Flag[HELL_MODE_FELDHELL] = False;
  Transceiver[Indices.TRx_Index]->RSID_Mode = MODE_FSKH105;
  New_Parameters();
}

//------------------------------------------------------------------------

  void
on_hell_bdrate_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    New_Parameters();
}

//------------------------------------------------------------------------

static GtkFileChooser *filechooser = NULL;

  void
on_hell_font_selector_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  GtkBuilder *builder = NULL;
  if( !filechooser )
  {
    filechooser = GTK_FILE_CHOOSER( Create_Filechooser( &builder ) );
    gtk_widget_show( GTK_WIDGET(filechooser) );
    gtk_file_chooser_set_action( filechooser, GTK_FILE_CHOOSER_ACTION_OPEN );
    char folder[FILE_NAME_SIZE];
    Strlcpy( folder, getenv("HOME"),   sizeof(folder) );
    Strlcat( folder, "/.hermes2/hell/fonts/", sizeof(folder) );
    gtk_file_chooser_set_current_folder( filechooser, folder );
    g_object_unref( builder );
  }
}

//------------------------------------------------------------------------

  void
on_hell_filechooser_response(
    GtkDialog *dialog,
    gint       response_id,
    gpointer   user_data )
{
  if( response_id == GTK_RESPONSE_OK )
  {
    if( strlen(gtk_file_chooser_get_filename(filechooser)) > 80 )
    {
      Error_Dialog( _("Font file name to long"), HIDE_OK );
      return;
    }

    Strlcpy( hell_rc_data.font_file,
        gtk_file_chooser_get_filename(filechooser),
        sizeof(hell_rc_data.font_file) );

    if( strstr(hell_rc_data.font_file, ".bdf") == NULL )
    {
      Error_Dialog( _("This is not a .bdf font file"), HIDE_OK );
      return;
    }

    New_Parameters();
  } // if( response_id == GTK_RESPONSE_OK )

  gtk_widget_destroy( GTK_WIDGET(filechooser) );
}

//------------------------------------------------------------------------

  void
on_hell_filechooser_destroy(
    GObject *object,
    gpointer user_data )
{
  filechooser = NULL;
}

//------------------------------------------------------------------------

  void
on_hell_show_font_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  int idx;

  for( idx = 0; idx < hell_rc_data.font_data.num_glyphs; idx++ )
    if( !Render_Font(&hell_rc_data.font_data, idx) )
      return;
}

//------------------------------------------------------------------------

  void
on_hell_reverse_video_toggled(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[HELL_REVERSE_VIDEO] = True;
  else
    Flag[HELL_REVERSE_VIDEO] = False;
}

//------------------------------------------------------------------------

  void
on_hell_loopback_toggled(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[HELL_LOOPBACK] = True;
  else
    Flag[HELL_LOOPBACK] = False;
}

//------------------------------------------------------------------------

  void
on_hell_md2x2_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  hell_rc_data.dot_size = 2;
  New_Parameters();
}

//------------------------------------------------------------------------

  void
on_hell_md3x3_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  hell_rc_data.dot_size = 3;
  New_Parameters();
}

//------------------------------------------------------------------------

  void
on_hell_md4x4_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  hell_rc_data.dot_size = 4;
  New_Parameters();
}

//------------------------------------------------------------------------

  void
on_hell_capitalize_letters_toggled(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
    Flag[GUEST_CAPITALIZE] = True;
  else Flag[GUEST_CAPITALIZE] = False;
}

//------------------------------------------------------------------------

  void
on_hell_record_qsos_toggled(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  if( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) )
  {
    Flag[HELL_SAVE_PIXBUF] = True;
    Flag[GUEST_RECORD_QSO] = True;
  }
  else
  {
    Flag[GUEST_RECORD_QSO] = False;
    Flag[HELL_SAVE_PIXBUF] = False;
    if( qso_record.qso_record_fp != NULL )
    {
      Close_File( &qso_record.qso_record_fp );
      qso_record.qso_record_fp = NULL;
    }
  }
}

//------------------------------------------------------------------------

  void
on_hell_identify_in_cw_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  Flag[GUEST_TRANSMIT_ID] = True;
}

//------------------------------------------------------------------------

