/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callback_func.h"
#include "display.h"
#include "operation.h"
#include "shared.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include <ctype.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>

//------------------------------------------------------------------------

/* Hell_Change_Modes()
 *
 * Handles the on_hell_transmit_clicked() callback
 */
  void
Hell_Change_Modes( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Toggle Tx/Rx/Macro modes
  if( Flag[GUEST_TRANSMIT_MACRO] )
  {
    Flag[GUEST_TRANSMIT_MACRO] = False;
  }
  else if( Flag[GUEST_TRANSMIT_KEYBD] )
  {
    Flag[GUEST_TRANSMIT_KEYBD] = False;
    Flag[GUEST_KEYBD_BUSY]     = False;
  }
  else if( !ddv->transmit_on ) // MOX or Tune not on
  {
    // Set up control flags for Keyboard transmit
    Flag[GUEST_TRANSMIT_KEYBD] = True;

    // Wait for Rx mode to exit
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Clear keyboard buffer data
    keybd_buf.key_count    = 0;
    keybd_buf.char_pointer = 0;
    keybd_buf.space_count  = 0;
    keybd_buf.char_count   = 0;

    // Create a thread to run keyboard transmit mode
    if( !Hell_Initialize_Transmit() ) return;
    if( !Pthread_Create(
          &hermes2_rc.guest_tx_thread, NULL, Hell_Transmit_Keybd, NULL,
          _("Failed to create Transmit Keyboard thread")) )
    {
      Flag[GUEST_KEYBD_BUSY] = False;
    }
  }
} // Hell_Change_Modes()

//------------------------------------------------------------------------

/* Hell_Select_Macro()
 *
 *  Selects a macro to be transmitted
 */
  void
Hell_Select_Macro( GtkButton *button )
{
  if( !Transceiver[Indices.TRx_Index]->receive_active )
    return;

  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Delay macro till keybd not busy
  if( Flag[GUEST_KEYBD_BUSY] ) return;

  // Clear keyboard buffer data
  keybd_buf.key_count    = 0;
  keybd_buf.char_pointer = 0;
  keybd_buf.space_count  = 0;
  keybd_buf.char_count   = 0;

  // Direct keyboard entries to Tx textview
  gtk_widget_grab_focus(
      Builder_Get_Object(hell_gui.window_builder, "hell_tx_textview") );

  // Transmit a macro
  if( !Flag[GUEST_TRANSMIT_MACRO] )
  {
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      pthread_join( hermes2_rc.guest_rx_thread, NULL );
    }

    // Enable macro transmission if we are not
    // in transmit mode (MOX on or Tune active)
    if( !ddv->transmit_on || Flag[GUEST_TRANSMIT_KEYBD] )
    {
      if( !Flag[GUEST_TRANSMIT_KEYBD] &&
          !Hell_Initialize_Transmit() )
        return;

      Flag[GUEST_TRANSMIT_MACRO] = True;
      g_idle_add( Hell_Set_TxRx_Labels, NULL );

      // Get macro index from the name of the macro button
      hell_macro_idx =
        (uint8_t)( atoi(gtk_widget_get_name(GTK_WIDGET(button))) - 1 );

      // Create a thread to run Macro Transmit if not in Keyboard mode
      if( !Flag[GUEST_TRANSMIT_KEYBD] )
      {
        if( !Pthread_Create(
              &hermes2_rc.guest_tx_thread, NULL, Hell_Transmit_Macro, NULL,
              _("Failed to create Transmit_Macro thread")) )
          return;
      }
      Flag[GUEST_TRANSMIT_KEYBD] = False;

    } // if( !hermes2_rc.transmit_status
  } // if( !Flag[GUEST_TRANSMIT_MACRO] )
  else
   {
    // Exit Macro mode
    Flag[GUEST_TRANSMIT_MACRO] = False;
  }

} // Select_Macro()

//------------------------------------------------------------------------

/* Hell_Receive_Clicked()
 *
 * Handles the on_hell_receive_clicked() callback
 */
  void
Hell_Receive_Clicked( void )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // If in Rx mode, terminated it
  if( Flag[GUEST_RECEIVING] )
  {
    Flag[GUEST_RECEIVE_MODE] = False;
    pthread_join( hermes2_rc.guest_rx_thread, NULL );
  }
  else
  { // If not in Tx mode, start Rx mode
    if( !Flag[GUEST_TRANSMIT_MACRO] &&
        !Flag[GUEST_TRANSMIT_KEYBD] &&
        !Flag[GUEST_KEYBD_BUSY]     &&
        !ddv->transmit_on )
    {
      Flag[GUEST_RECEIVE_MODE] = True;
      if( !Pthread_Create(
            &hermes2_rc.guest_rx_thread, NULL, Hell_Receive_Mode, NULL,
            _("Failed to create Receive thread")) )
      {
        Flag[GUEST_RECEIVE_MODE] = False;
        g_idle_add( Hell_Set_TxRx_Labels, NULL );
        return;
      }
    } // Terminate TX mode if no outstanding characters in keyboard buffer
    else if( !Flag[GUEST_KEYBD_BUSY] )
    {
      Flag[GUEST_TRANSMIT_MACRO] = False;
      Flag[GUEST_TRANSMIT_KEYBD] = False;
      if( Flag[GUEST_TRANSMITTING] )
        pthread_join( hermes2_rc.guest_tx_thread, NULL );
    }
  } // if( Flag[RECEIVE_MODE] )

  g_idle_add( Hell_Set_TxRx_Labels, NULL );
} // Hell_Receive_Clicked()

//------------------------------------------------------------------------

/* Hell_Gui_Button_Press()
 *
 * Handles button_press_events on the Tx/Rx view ports
 */
  gboolean
Hell_Gui_Button_Press( const GdkEventButton *event )
{
  switch( event->button )
  {
    // Focus to widget that received the button press
    case 1:
      return( FALSE );

    case 2: // Toggle Tx/Rx/Macro modes
      Hell_Change_Modes();
      break;

    case 3: // Popup menu. "Sensitize Identify in CW" in keyboard mode
      if( Flag[GUEST_TRANSMIT_KEYBD] )
        gtk_widget_set_sensitive( Builder_Get_Object(
              hell_gui.popup_menu_builder, "hell_identify_in_cw"), TRUE );
      else
        gtk_widget_set_sensitive( Builder_Get_Object(
              hell_gui.popup_menu_builder, "hell_identify_in_cw"), FALSE );

      gtk_menu_popup_at_pointer( GTK_MENU(hell_gui.popup_menu), NULL );
      break;
  } // switch( event->button )

  return( TRUE );
} // Hell_Gui_Button_Press()

//------------------------------------------------------------------------

/* Hell_Tx_Textview_Key_Press()
 *
 * Handles the on_hell_tx_textview_key_press_event() callback
 */
  gboolean
Hell_Tx_Textview_Key_Press( const GdkEventKey *event )
{
  guint keyval = event->keyval;

  if( (keyval == GDK_KEY_Shift_L) ||
      (keyval == GDK_KEY_Shift_R) )
    return( FALSE );

  // Do nothing if not in keybd mode
  if( !Flag[GUEST_TRANSMIT_KEYBD] && !Flag[GUEST_RECEIVE_MODE] )
    return( FALSE );

  // Deal with Backspace
  if( keyval == GDK_KEY_BackSpace )
  {
    // Reduce character count for word wrap
    keybd_buf.char_count--;
    if( keybd_buf.char_count < 0 )
      keybd_buf.char_count += hell_rc_data.word_wrap;

    // Track xmit char pointer with new char pointer
    if( keybd_buf.char_pointer == keybd_buf.key_count )
    {
      keybd_buf.char_pointer--;
      if( keybd_buf.char_pointer < 0 )
        keybd_buf.char_pointer += KEY_BUF_SIZE;
    }

    // Backspace new character pointer
    keybd_buf.key_count--;
    if( keybd_buf.key_count < 0 )
      keybd_buf.key_count += KEY_BUF_SIZE;

    // Count down spaces if deleted
    if( (keybd_buf.key_buf[keybd_buf.key_count] == ' ') &&
        keybd_buf.space_count )
      keybd_buf.space_count--;

    // Re-terminate buffer
    keybd_buf.key_buf[keybd_buf.key_count] = '\0';

    // Print Backspace to text view
    tx_print_chr.printchr = GDK_KEY_BackSpace;
    Queue_Character( &tx_print_chr );

    return( TRUE );
  } // if( keyval == GDK_KEY_BackSpace )

  // Don't process non-ASCII characters
  if( (event->keyval & 0xFF) > 0x7F )
    return( TRUE );

  // Capitalize letters if enabled
  if( Flag[GUEST_CAPITALIZE] ) keyval = (guint)toupper( keyval );
  keybd_buf.char_count++;

  // Print ascii characters to screen
  if( keyval <= 0xFF ) tx_print_chr.printchr = keyval;

  // Send a space in place of Return
  if( keyval == GDK_KEY_Return )
  {
    keyval = ' ';

    // Reset word wrap counter
    keybd_buf.char_count  = 0;
    tx_print_chr.printchr = GDK_KEY_Return;
  }

  // Set keybd busy flag since a space
  // will start xmission of the buffer
  if( keyval == ' ' )
  {
    if( !Flag[GUEST_RECEIVE_MODE] ) Flag[GUEST_KEYBD_BUSY] = True;

    /* If zero typed-in spaces left in keyboard string, increment
     * space count to allow for the default end space added below */
    if( !keybd_buf.space_count ) keybd_buf.space_count++;
    keybd_buf.space_count++;

    // Do word wrapping
    if( keybd_buf.char_count >= hell_rc_data.word_wrap )
    {
      // Reset word wrap counter
      keybd_buf.char_count  = 0;
      tx_print_chr.printchr = GDK_KEY_Return;
    }
  } // if( keybd_buff[keybd_buf.key_count] == ' ' )

  // Enter keyboard character to buffer
  keybd_buf.key_buf[keybd_buf.key_count] = keyval;

  // Increment keystroke pointer & enter a default space
  keybd_buf.key_count++;
  if( keybd_buf.key_count >= KEY_BUF_SIZE )
    keybd_buf.key_count = 0;
  keybd_buf.key_buf[keybd_buf.key_count] = ' ';

  // Increment keystroke pointer & terminate buffer
  keybd_buf.key_count++;
  if( keybd_buf.key_count >= KEY_BUF_SIZE )
    keybd_buf.key_count = 0;
  keybd_buf.key_buf[keybd_buf.key_count] = '\0';

  // Backstep keystroke pointer for next keystroke by user
  keybd_buf.key_count--;
  if( keybd_buf.key_count < 0 )
    keybd_buf.key_count += KEY_BUF_SIZE;

  // Print final character to text view
  Queue_Character( &tx_print_chr );

  return( FALSE );
} // Hell_Tx_Textview_Key_Press()

//------------------------------------------------------------------------

