/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "interface.h"
#include "shared.h"
#include "../Hermes2/callback_func.h"
#include "../common/guest_utils.h"
#include "../../config.h"

//------------------------------------------------------------------------

/* FTx_Read_Config()
 *
 * Loads the ftxrc configuration file
 */
  gboolean
FTx_Read_Config( gpointer data )
{
   // Buffer for Read_Line
  char line[81];

  // Config file pointer
  FILE *ftxrc;

  // Setup file path to ftxrc
  // Create the ftx config file name
  Strlcpy( ftx_rc.rc_fpath,
      getenv("HOME"), sizeof(ftx_rc.rc_fpath) );
  Strlcat( ftx_rc.rc_fpath,
      "/.hermes2/ftx/ftx.config", sizeof(ftx_rc.rc_fpath) );

  // Open ftx.config file
  ftxrc = fopen( ftx_rc.rc_fpath, "r" );
  if( ftxrc == NULL )
  {
    perror( ftx_rc.rc_fpath );
    Error_Dialog(
        _("Failed to open ftx.config file\n"\
          "Quit FTx and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read default FTx mode
  if( Read_Line(line, ftxrc, _("Default Mode") ) != SUCCESS )
    return( FALSE );
  ftx_rc.proto = (uint8_t)atoi( line );
  if( ftx_rc.proto >= FTX_NUM_OF_PROTO )
  {
    Close_File( &ftxrc );
    Error_Dialog(
        _("Error reading ftx.config\n"\
          "Incorrect FTx Mode\n"\
          "Quit ftx and correct"), HIDE_OK );
    return( FALSE );
  }

  // Read Station Callsign
  if( Read_Line(line, ftxrc, _("Station Callsign") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.own_call, line, sizeof(ftx_rc.own_call) );

  // Read Station Locator
  if( Read_Line(line, ftxrc, _("Station Locator") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.own_locator, line, sizeof(ftx_rc.own_locator) );

  // Read Station QTH name
  if( Read_Line(line, ftxrc, _("Station QTH") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.home_name, line, sizeof(ftx_rc.home_name) );

  // Calculate "Home" position Longitude and Latitude
  Gridloc_to_Position(
      ftx_rc.own_locator, &ftx_rc.home_latitude, &ftx_rc.home_longitude );

  // Read Xplanet display DX enable flag
  if( Read_Line(line, ftxrc, _("Enable DX in Xplanet") ) != SUCCESS )
    return( FALSE );
  ftx_rc.dx_xplanet = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.inqso_xplanet), ftx_rc.dx_xplanet );

  // Read Xplanet display Band enable flag
  if( Read_Line(line, ftxrc, _("Enable Band in Xplanet") ) != SUCCESS )
    return( FALSE );
  ftx_rc.band_xplanet = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.band_xplanet), ftx_rc.band_xplanet );

  // Read Colorize Enable flag
  if( Read_Line(line, ftxrc, _("Colorize Enable") ) != SUCCESS )
    return( FALSE );
  ftx_rc.colorize = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.colorize), ftx_rc.colorize );

  // Read Auto Clear flag
  if( Read_Line(line, ftxrc, _("Read Auto Clear flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.auto_clear = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.auto_clear), ftx_rc.auto_clear );

  // Read Display of CQ calls flag
  if( Read_Line(line, ftxrc, _("Read Display of CQ calls flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.with_cq = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.with_cq), ftx_rc.with_cq );

  // Read Display of Own Callsign flag
  if( Read_Line(line, ftxrc, _("Read Display of Own Callsign flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.calling_me = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.calling_me), ftx_rc.calling_me );

  // Read Display of DX Callsign flag
  if( Read_Line(line, ftxrc, _("Read Display of DX Callsign flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.dx_call_show = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.dx_call), ftx_rc.dx_call_show );

  // Read Highlight Fields flag 
  if( Read_Line(line, ftxrc, _("Read Highlight Fields flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.highlighted = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.highlighted), ftx_rc.highlighted );

  // Read Auto Sequence flag
  if( Read_Line(line, ftxrc, _("Read Auto Sequence flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.auto_sequence = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.auto_sequence), ftx_rc.auto_sequence );

  // Read Auto CQ flag
  if( Read_Line(line, ftxrc, _("Read Auto CQ flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.auto_qso = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.auto_qso), ftx_rc.auto_qso );

  // Read TX Even flag
  if( Read_Line(line, ftxrc, _("Read TX Even flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.tx_even = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.tx_even), ftx_rc.tx_even );

  // Read Auto Save flag
  if( Read_Line(line, ftxrc, _("Read Auto Save flag") ) != SUCCESS )
    return( FALSE );
  ftx_rc.auto_save = (BOOLEAN)( atoi(line) );
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(ftx_gui.auto_save), ftx_rc.auto_save );

  // Read the Index for the Reply to CQ combobox
  if( Read_Line(line, ftxrc, _("Read Index for Reply to CQ") ) != SUCCESS )
    return( FALSE );
  ftx_rc.reply_cq_idx = (uint8_t)( atoi(line) );
  gtk_combo_box_set_active( GTK_COMBO_BOX(ftx_gui.reply_cq_combo), ftx_rc.reply_cq_idx );

  // Read the Index for the Select CQ combobox
  if( Read_Line(line, ftxrc, _("Read Index for Select CQ") ) != SUCCESS )
    return( FALSE );
  ftx_rc.select_cq_idx = (uint8_t)( atoi(line) );
  gtk_combo_box_set_active( GTK_COMBO_BOX(ftx_gui.select_cq_combo), ftx_rc.select_cq_idx );

  // Read the Transmit Audio Frequency
  if( Read_Line(line, ftxrc, _("Read Transmit Audio Frequency") ) != SUCCESS )
    return( FALSE );
  ftx_rc.tx_frequency = (uint16_t)( atoi(line) );
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(ftx_gui.tx_frequency), ftx_rc.tx_frequency );

  // Read Xplanet command options
  if( Read_Line(line, ftxrc, _("Xplanet Options") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.xplanet_options, line, sizeof(ftx_rc.xplanet_options) );

  // Read Xplanet config file name
  if( Read_Line(line, ftxrc, _("Xplanet Config file") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.xplanet_config_file, line, sizeof(ftx_rc.xplanet_config_file) );

  // Read Xplanet Marker file
  if( Read_Line(line, ftxrc, _("Xplanet Marker file") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.xplanet_marker_file, line, sizeof(ftx_rc.xplanet_marker_file) );

  // Read Xplanet Arc file
  if( Read_Line(line, ftxrc, _("Xplanet Arc file") ) != SUCCESS )
    return( FALSE );
  Strlcpy( ftx_rc.xplanet_arc_file, line, sizeof(ftx_rc.xplanet_arc_file) );

  // Create xplanet command
  snprintf( ftx_rc.xplanet_command, sizeof(ftx_rc.xplanet_command),
      "xplanet -window -projection rectangular %s", ftx_rc.xplanet_options );

  // Read main Window's position
  if( Read_Line(line, ftxrc, _("Window Position") ) != SUCCESS )
    return( FALSE );
  gint x = (gint)atoi( line );
  gint y = (gint)atoi( &line[5] );

  // Read main Window's size
  if( Read_Line(line, ftxrc, _("Window Size") ) != SUCCESS )
    return( FALSE );
  gint w = (gint)atoi( line );
  gint h = (gint)atoi( &line[5] );

  // Place window and resize it
  Set_Window_Geometry( ftx_gui.window, x, y, w, h );

  // Wait for GTK to complete its tasks
  while( g_main_context_iteration(NULL, FALSE) );

  Close_File( &ftxrc );
  return( FALSE );
} // End of FTx_Read_Config()

//------------------------------------------------------------------

#define FTX_CONFIG_FORMAT \
  _("### FTx Configuration File ###\n"\
      "#\n"\
      "# FTx Protocol\n"\
      "%d\n"\
      "# Station Callsign\n"\
      "%s\n"\
      "# Station Locator\n"\
      "%s\n"\
      "# Station QTH\n"\
      "%s\n"\
      "# Enable display DX in Xplanet\n"\
      "%u\n"\
      "# Enable display Band in Xplanet\n"\
      "%u\n"\
      "# Enable Colorization\n"\
      "%u\n"\
      "# Enable Auto Clear of Treeview \n"\
      "%u\n"\
      "# Enable Display of CQ calls\n"\
      "%u\n"\
      "# Enable Display of Own Callsign\n"\
      "%u\n"\
      "# Enable Display of DX Callsign\n"\
      "%u\n"\
      "# Highlight Fields with user-defined pattern\n"\
      "%u\n"\
      "# Auto Sequence Flag\n"\
      "%u\n"\
      "# Auto CQ Flag\n"\
      "%u\n"\
      "# TX Even Flag\n"\
      "%u\n"\
      "# Auto Save Flag\n"\
      "%u\n"\
      "# Reply to CQ Index\n"\
      "%u\n"\
      "# Select CQ Index\n"\
      "%u\n"\
      "# Transmit Audio Frequency\n"\
      "%u\n"\
      "# Xplanet Options\n"\
      "%s\n"\
      "# Xplanet Config\n"\
      "%s\n"\
      "# Xplanet Marker file\n"\
      "%s\n"\
      "# Xplanet Arc file\n"\
      "%s\n"\
      "# Window Position\n"\
      "%4u,%4u\n"\
      "# Window Size\n"\
      "%4u,%4u\n")

/* FTx_Save_Config()
 *
 * Loads the ftxrc configuration file
 */
  BOOLEAN
FTx_Save_Config( void )
{
  // Config file pointer
  FILE *ftxrc;

  // Setup file path to ftxrc
  // Create the ftx config file name
  Strlcpy( ftx_rc.rc_fpath,
      getenv("HOME"), sizeof(ftx_rc.rc_fpath) );
  Strlcat( ftx_rc.rc_fpath,
      "/.hermes2/ftx/ftx.config", sizeof(ftx_rc.rc_fpath) );

  // Open ftx.config file
  ftxrc = fopen( ftx_rc.rc_fpath, "w" );
  if( ftxrc == NULL )
  {
    perror( ftx_rc.rc_fpath );
    Error_Dialog(
        _("Failed to open ftx.config file\n"\
          "Quit FTx and correct"), HIDE_OK );
    return( FALSE );
  }

  // Get Window geometry
  gint x, y, w, h;
  Get_Window_Geometry( ftx_gui.window, &x, &y, &w, &h );

  int ret = fprintf( ftxrc,
      FTX_CONFIG_FORMAT,
      ftx_rc.proto,
      ftx_rc.own_call,
      ftx_rc.own_locator,
      ftx_rc.home_name,
      ftx_rc.dx_xplanet,
      ftx_rc.band_xplanet,
      ftx_rc.colorize,
      ftx_rc.auto_clear,
      ftx_rc.with_cq,
      ftx_rc.calling_me,
      ftx_rc.dx_call_show,
      ftx_rc.highlighted,
      ftx_rc.auto_sequence,
      ftx_rc.auto_qso,
      ftx_rc.tx_even,
      ftx_rc.auto_save,
      ftx_rc.reply_cq_idx,
      ftx_rc.select_cq_idx,
      ftx_rc.tx_frequency,
      ftx_rc.xplanet_options,
      ftx_rc.xplanet_config_file,
      ftx_rc.xplanet_marker_file,
      ftx_rc.xplanet_arc_file,
      x, y, w, h );

  if( ret < 0 )
  {
    Fprintf_Error( ftxrc, ftx_rc.rc_fpath );
    Close_File( &ftxrc );
    return( False );
  }

  // Terminate file
  ret = fprintf( ftxrc, "#\n[END]\n#" );
  if( ret < 0 )
  {
    Fprintf_Error( ftxrc, ftx_rc.rc_fpath );
    Close_File( &ftxrc );
    return( False );
  }

  Close_File( &ftxrc );
  return( True );
} // FTx_Save_Config()

//------------------------------------------------------------------------

/* Is_CQ()
 *
 * Checks if the supplied string is a valid CQ string
 */
  BOOLEAN
Is_CQ( const char *cq )
{
  // It is a CQ if the field is just CQ or CQ_....
  if( (strcmp(cq, "CQ") == 0) || (strncmp(cq, "CQ_", 3) == 0) )
    return( True );
  else
    return( False );
} // Is_CQ()

//------------------------------------------------------------------------

/* Is_Locator()
 *
 * Checks if the supplied string is a valid Maidenhead Locator
 */
  BOOLEAN
Is_Locator( const char *locator )
{
  BOOLEAN loc_ok = False;
  size_t siz = strlen( locator );

  // Test if it is a 4 character locator, the usual
  if( (siz == 4) &&
      (strcmp(locator, "RR73") != 0) &&
      (locator[0] >= 'A') && (locator[0] <= 'R') &&
      (locator[1] >= 'A') && (locator[1] <= 'R') &&
      (locator[2] >= '0') && (locator[2] <= '9') &&
      (locator[3] >= '0') && (locator[3] <= '9') )
  {
    loc_ok = True;
  }
  // Test if it is a 6-char Locator (if such is ever used)
  else if( (siz == 6) &&
      (locator[0] >= 'A') && (locator[0] <= 'R') &&
      (locator[1] >= 'A') && (locator[1] <= 'R') &&
      (locator[2] >= '0') && (locator[2] <= '9') &&
      (locator[3] >= '0') && (locator[3] <= '9') &&
      (locator[4] >= 'A') && (locator[4] <= 'X') &&
      (locator[5] >= 'A') && (locator[5] <= 'X') )
  {
    loc_ok = True;
  }

  return( loc_ok );
} // Is_Locator()

//------------------------------------------------------------------------

/* Get_Callsign()
 *
 * Looks for a valid callsign in 'field' and returns it in 'call'.
 * A valid callsign is taken as having minimum 3 characters,
 * beginning with one or two letters [A-Z] followed by a number
 * [0-9], beginning with one number [0-9] followed by one letter
 * [A-Z], then followed by a number [0-9]. Hashed callsigns are
 * processed to remove angle brackets or rejected if <...>
 */
  BOOLEAN
Get_Callsign( char *call, char *field )
{
  // In case of invalid callsign
  BOOLEAN call_ok = False;
  call[0] = '\0';

  // Unknown hashed callsign
  if( strcmp(field, "<...>") == 0 ) return( call_ok );

  // Field too short to be a callsign
  size_t len = strlen( field );
  if( len < 3 ) return( call_ok );

  /* Remove the leading and trailing spaces of highlighted labels.
   * String length should be min 5: space + 3 char call + space */
  if( (field[0] == ' ') && (len >= 5) )
  {
    len -= 2;
    memmove( field, &field[1], len );
    field[len] = '\0';
  }

  // Prefix is a single letter followed by a number
  if( ((field[0] >= 'A') && (field[0] <= 'Z')) &&
      ((field[1] >= '0') && (field[1] <= '9')) )
  {
    call_ok = True;
  }
  else  // Prefix is two letters followed by a number
    if( ((field[0] >= 'A') && (field[0] <= 'Z')) &&
        ((field[1] >= 'A') && (field[1] <= 'Z')) &&
        ((field[2] >= '0') && (field[2] <= '9')) )
    {
      call_ok = True;
    }
    else // Prefix is a number followed by letter
      if( ((field[0] >= '0') && (field[0] <= '9')) &&
          ((field[1] >= 'A') && (field[1] <= 'Z')) )
      {
        call_ok = True;
      }
      else // Field is a hashed callsign, eliminate angle brackets
        if( field[0] == '<' )
        {
          len = strlen( field );
          field[len - 1] = '\0';  // Replace '>' with terminator
          field++;  // Bypass '<'
          call_ok = True;
        }
        else // Field is callsign preceded by country prefix (AB1/CA1LL ...)
          if( strstr(field, "/") != NULL )
          {
            call_ok = True;
          }

  // Copy field into return call string
  if( call_ok ) Strlcpy( call, field, FTX_CALL_SIZE );
  return( call_ok );

} // Get_Callsign()

//------------------------------------------------------------------------

/* Get_SNR()
 *
 * Looks for a valid SNR in 'field' and returns it in 'snr'.
 * A valid SNR string should start with +|- or a number
 * [0-9] or with R followed by +|- and a number [0-9].
 */
  BOOLEAN
Get_SNR( char *snr, char *field )
{
  // In case of invalid SNR field
  BOOLEAN snr_ok = False;
  snr[0] = '\0';

  if( ((field[0] == '+') || (field[0] == '-')) &&
      ((field[1] >= '0') && (field[1] <= '9')) )
    snr_ok = True;

  if( (field[0] == 'R') &&
      ( (field[1] == '+') || (field[1] == '-') ||
       ((field[1] >= '0') && (field[1] <= '9')) ) )
  {
    field++;   // Reject thr 'R'
    snr_ok = True;
  }

  if( snr_ok ) Strlcpy( snr, field, FTX_SNR_SIZE );

  return( snr_ok );
} // Get_SNR()

//------------------------------------------------------------------------

/* Is_Final()
 *
 * Identifies and gets the "final" greeting in a QSO from 'field' and
 * returns it in 'final'. This is usually RRR or RR73 or R73 or just 73.
 */
  BOOLEAN
Is_Final( const char *field )
{
  // In case of unrecognised "final"
  BOOLEAN final_ok = False;

  if( (strcmp(field, "73")   == 0) ||
      (strcmp(field, "R73")  == 0) ||
      (strcmp(field, "RR73") == 0) ||
      (strcmp(field, "RRR")  == 0) )
    final_ok = True;

  return( final_ok );
} // Get_Final()

//------------------------------------------------------------------------

/* FTx_Cleanup()
 *
 * Cleans up before quitting
 */
  void
FTx_Cleanup( void )
{
  // Stop FTx decoder thread
  ftx_rc.decoder_exit = True;

  // Save outstanding records
  if( Flag[GUEST_SAVE_RECORD] ) FTx_Save_QSO( &ftx_qso_record );

  // Close open files
  Close_File( &(ftx_qso_record.adif_log_fp) );
  Close_File( &(ftx_qso_record.station_log_fp) );

  // Free Digimode semaphores
  Init_Semaphore( &digimode_semaphore, False );
  Init_Semaphore( &duc_send_semaphore, False );

  // Close some GUI objects
  //g_object_unref( ftx_gui.popup_menu_builder );
  g_object_unref( ftx_gui.window_builder );
  ftx_gui.window_builder     = NULL;
  ftx_gui.popup_menu_builder = NULL;
  ftx_gui.window             = NULL;
  hermes2_gui.guest_window   = NULL;
  ftx_gui.mesg_store         = NULL;

  // Clear general flags
  Flag[GUEST_RECORD_QSO]    = False;
  Flag[GUEST_RECEIVE_MODE]  = False;
  Flag[GUEST_TRANSMIT_MODE] = False;
  Flag[GUEST_SAVE_RECORD]   = False;
  Flag[GUEST_QUIT]          = False;
  Flag[GUEST_RECEIVING]     = False;
  Flag[GUEST_TRANSMITTING]  = False;

  // Disable the common Guest modulation mode
  if( Transceiver[Indices.TRx_Index] != NULL )
    Transceiver[Indices.TRx_Index]->guest_modulation_mode = RX_MODE_ITEMS;

  // Kill xplanet and remove files
  if( ftx_rc.dx_xplanet || ftx_rc.band_xplanet )
    Kill_Xplanet(
        ftx_rc.xplanet_config_file,
        ftx_rc.xplanet_marker_file,
        ftx_rc.xplanet_arc_file );
  ftx_rc.dx_xplanet   = False;
  ftx_rc.band_xplanet = False;

} // FTx_Cleanup()

//------------------------------------------------------------------------

/* Save_QSO_Record()
 *
 * Saves QSO record to .adif and .txt files
 */
  BOOLEAN
FTx_Save_QSO( ftx_qso_record_t *ftx_qso_rec )
{
  // File name to save QSO record to
  char file_name[FILE_NAME_SIZE];

  // Buffer for reading and printing to files
  char buff[REC_BUF_SIZE];

  // Guest_Open_File() return value
  uint8_t fopen_ret;

  // Create a file for QSO ADIF log if not open
  Strlcpy( file_name, ftx_rc.ftx_dir, sizeof(file_name) );
  Strlcat( file_name, "/log.adif",    sizeof(file_name) );

  // Open ADIF log file to append
  fopen_ret = Guest_Open_File( &ftx_qso_rec->adif_log_fp, file_name, "a" );
  if( fopen_ret == FILE_NEW )
  {
    snprintf( buff, sizeof(buff),
        "QSO Log file created in ADIF v1.0 format by %s <EOH>\n\n",
        PACKAGE_STRING );
    if( !File_Print(&ftx_qso_rec->adif_log_fp, buff) )
      return( False );
  }
  else if( fopen_ret == FILE_FAIL )
    return( False );

  // Make an ADIF format frequency string
  double freq = atof( ftx_qso_rec->frequency );
  freq /= 1000000;  //Change from Hz to MHz
  char freq_str[FTX_FREQ_SIZE];
  snprintf( freq_str, FTX_FREQ_SIZE, "%FTX_FREQ_SIZE.3f", freq );

  // Print QSO record to adif format file
  snprintf( buff, sizeof(buff),
      "<CALL:%u>%s<QSO_DATE:8>%s<TIME_ON:4>%s\n"
      "<FREQ:%u>%s<MODE:%u>%s<RST_SENT:3>%3s<EOR>\n\n",
      (uint8_t)strlen(ftx_qso_rec->dx_call), ftx_qso_rec->dx_call,
      ftx_qso_rec->date, ftx_qso_rec->time,
      (uint8_t)strlen(freq_str), freq_str,
      (uint8_t)strlen(ftx_qso_rec->mode), ftx_qso_rec->mode,
      ftx_qso_rec->dx_snr );

  if( !File_Print(&ftx_qso_rec->adif_log_fp, buff) )
    return( False );

  // Create a file for station log if not already open
  Strlcpy( file_name, ftx_rc.ftx_dir, sizeof(file_name) );
  Strlcat( file_name, "/log.txt",     sizeof(file_name) );

  // Open station log for read/write, create new if failure
  fopen_ret = Guest_Open_File( &ftx_qso_rec->station_log_fp, file_name, "r+" );
  if( fopen_ret == FILE_FAIL ) return( False );

  // *** Print record in text log file ***
  if( fseek(ftx_qso_rec->station_log_fp, 0, SEEK_END) != 0 )
    return( False );

  snprintf( buff, sizeof(buff), "%-13s %-9s %-3s %-3s %-6s %-8s %-4s\n",
      ftx_qso_rec->dx_call, ftx_qso_rec->frequency,
      ftx_qso_rec->dx_snr,  ftx_qso_rec->mode,
      ftx_qso_rec->dx_loc,  ftx_qso_rec->date,
      ftx_qso_rec->time );

  if( !File_Print(&ftx_qso_rec->station_log_fp, buff) )
    return( False );

  fflush( ftx_qso_rec->station_log_fp );
  fflush( ftx_qso_rec->adif_log_fp );
  //Close_File( ftx_qso_rec->station_log_fp );
  //Close_File( ftx_qso_rec->adif_log_fp );

  Flag[GUEST_SAVE_RECORD] = False;

  return( True );

} // Save_QSO_Record()

//------------------------------------------------------------------------


