/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef _INCLUDE_MONITOR_H_
#define _INCLUDE_MONITOR_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include "decode.h"
#include "kiss_fftr.h"

  /// Configuration options for FT4/FT8 monitor
  typedef struct
  {
    double f_min;            ///< Lower frequency bound for analysis
    double f_max;            ///< Upper frequency bound for analysis
    int sample_rate;         ///< Sample rate in Hertz
    int time_osr;            ///< Number of time subdivisions
    int freq_osr;            ///< Number of frequency subdivisions
    ftx_protocol_t protocol; ///< Protocol: FT4 or FT8
  } monitor_config_t;

  /// FT4/FT8 monitor object that manages DSP processing of incoming audio data
  /// and prepares a waterfall object
  typedef struct
  {
    double symbol_period; ///< FT4/FT8 symbol period in seconds
    int min_bin;          ///< First FFT bin in the frequency range (begin)
    int max_bin;          ///< First FFT bin outside the frequency range (end)
    int block_size;       ///< Number of samples per symbol (block)
    int subblock_size;    ///< Analysis shift size (number of samples)
    int nfft;             ///< FFT size
    double fft_norm;      ///< FFT normalization factor
    double *window;       ///< Window function for STFT analysis (nfft samples)
    double *last_frame;   ///< Current STFT analysis frame (nfft samples)
    ftx_waterfall_t wf;   ///< Waterfall object
    double max_mag;       ///< Maximum detected magnitude (debug stats)

    // KISS FFT housekeeping variables
    void *fft_work;        ///< Work area required by Kiss FFT
    kiss_fftr_cfg fft_cfg; ///< Kiss FFT housekeeping object
#ifdef WATERFALL_USE_PHASE
    int nifft;             ///< iFFT size
    void *ifft_work;       ///< Work area required by inverse Kiss FFT
    kiss_fft_cfg ifft_cfg; ///< Inverse Kiss FFT housekeeping object
#endif
  } monitor_t;

  void monitor_init(monitor_t *me, const monitor_config_t *cfg);
  void monitor_reset(monitor_t *me);
  void monitor_process(monitor_t *me, const double *frame);
  void monitor_free(monitor_t *me);

#ifdef WATERFALL_USE_PHASE
  void monitor_resynth(const monitor_t *me, const candidate_t *candidate, double *signal);
#endif

#ifdef __cplusplus
}
#endif

#endif // _INCLUDE_MONITOR_H_

