/*
 *  Copyright (c) 2003-2004, Mark Borgerding. All rights reserved.
 *  This file is part of KISS FFT - https://github.com/mborgerding/kissfft
 *
 *  SPDX-License-Identifier: BSD-3-Clause
 *  See COPYING file for more information.
 */

#include "kiss_fftr.h"
#include "_kiss_fft_guts.h"

//-----------------------------------------------------------------------

struct kiss_fftr_state
{
  kiss_fft_cfg substate;
  kiss_fft_cpx *tmpbuf;
  kiss_fft_cpx *super_twiddles;
};

//-----------------------------------------------------------------------

  kiss_fftr_cfg
kiss_fftr_alloc( int nfft, int inverse_fft, void *mem, size_t *lenmem )
{
  int i;
  kiss_fftr_cfg st = NULL;
  size_t subsize = 0, memneeded;

  if( nfft & 1 )
  {
    fprintf(stderr,"Real FFT optimization must be even.\n");
    return NULL;
  }
  nfft >>= 1;

  kiss_fft_alloc( nfft, inverse_fft, NULL, &subsize );
  memneeded =
    sizeof(struct kiss_fftr_state) + subsize + sizeof(kiss_fft_cpx) * (size_t)(nfft * 3 / 2);

  if (lenmem == NULL)
  {
    st = (kiss_fftr_cfg)KISS_FFT_MALLOC( memneeded );
  }
  else
  {
    if( *lenmem >= memneeded ) st = (kiss_fftr_cfg)mem;
    *lenmem = memneeded;
  }
  if( !st ) return NULL;

  st->substate = (kiss_fft_cfg)(st + 1); /* just beyond kiss_fftr_state struct */
  st->tmpbuf   = (kiss_fft_cpx *)( ((char *)st->substate) + subsize );
  st->super_twiddles = st->tmpbuf + nfft;
  kiss_fft_alloc( nfft, inverse_fft, st->substate, &subsize );

  for( i = 0; i < nfft / 2; ++i )
  {
    double phase = -M_PI * ( (double)(i + 1) / nfft + 0.5 );
    if( inverse_fft ) phase *= -1;
    kf_cexp( *(st->super_twiddles + i), phase );
  }
  return st;
}

//-----------------------------------------------------------------------

  void
kiss_fftr(
    kiss_fftr_cfg st,
    const kiss_fft_scalar *timedata,
    kiss_fft_cpx *freqdata )
{
  /* input buffer timedata is stored row-wise */
  int k, ncfft;
  kiss_fft_cpx fpnk, fpk, f1k, f2k, tw, tdc;

  if( st->substate->inverse )
  {
    fprintf( stderr, "kiss fft usage error: improper alloc\n" );
    exit( 1 );
  }

  ncfft = st->substate->nfft;

  /*perform the parallel fft of two real signals packed in real,imag*/
  kiss_fft( st->substate, (const kiss_fft_cpx *)timedata, st->tmpbuf );

  /* The real part of the DC element of the frequency spectrum in st->tmpbuf
   * contains the sum of the even-numbered elements of the input time sequence
   * The imag part is the sum of the odd-numbered elements
   *
   * The sum of tdc.r and tdc.i is the sum of the input time sequence.
   *      yielding DC of input time sequence
   * The difference of tdc.r - tdc.i is the sum of the input (dot product) [1,-1,1,-1...
   *      yielding Nyquist bin of input time sequence
   */

  tdc = st->tmpbuf[0];

  freqdata[0]     = creal( tdc ) + cimag( tdc );
  freqdata[ncfft] = creal( tdc ) - cimag( tdc );

  for( k = 1; k <= ncfft / 2; ++k )
  {
    fpk    = st->tmpbuf[k];
    fpnk = creal( st->tmpbuf[ncfft-k] ) - II * cimag( st->tmpbuf[ncfft-k] );

    C_ADD( f1k, fpk, fpnk );
    C_SUB( f2k, fpk, fpnk );
    C_MUL(  tw, f2k, st->super_twiddles[k-1] );

    freqdata[k] =
      HALF_OF( creal(f1k) + creal(tw) ) +
      II * HALF_OF( cimag(f1k) + cimag(tw) );
    freqdata[ncfft-k] =
      HALF_OF( creal(f1k) - creal(tw) ) +
      II * HALF_OF( cimag(tw) - cimag(f1k) );
  }
}

//-----------------------------------------------------------------------

  void
kiss_fftri(
    kiss_fftr_cfg st,
    const kiss_fft_cpx *freqdata,
    kiss_fft_scalar *timedata )
{
  /* input buffer timedata is stored row-wise */
  int k, ncfft;

  if( st->substate->inverse == 0 )
  {
    fprintf( stderr, "kiss fft usage error: improper alloc\n" );
    exit( 1 );
  }

  ncfft = st->substate->nfft;

  st->tmpbuf[0] =
    creal( freqdata[0] ) + creal( freqdata[ncfft] ) +
    II * ( creal(freqdata[0]) - creal(freqdata[ncfft]) );

  for( k = 1; k <= ncfft / 2; ++k )
  {
    kiss_fft_cpx fk, fnkc, fek, fok, tmp;
    fk   = freqdata[k];
    fnkc = creal( freqdata[ncfft - k] ) - II * cimag( freqdata[ncfft - k] );

    C_ADD( fek, fk, fnkc );
    C_SUB( tmp, fk, fnkc );
    C_MUL( fok, tmp, st->super_twiddles[k-1] );
    C_ADD( st->tmpbuf[k], fek, fok );
    C_SUB( st->tmpbuf[ncfft - k], fek, fok );

    st->tmpbuf[ncfft - k] =
      creal( st->tmpbuf[ncfft - k] ) +
      II * ( cimag(st->tmpbuf[ncfft - k]) * (-1) );
  }

  kiss_fft( st->substate, st->tmpbuf, (kiss_fft_cpx *)timedata );
}

//-----------------------------------------------------------------------

