/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "encode_ftx.h"
#include "constants.h"
#include "encode.h"
#include "message.h"
#include "shared.h"
#include "../common/utils.h"

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>

#define FT8_SYMBOL_BT 2.0  ///< symbol smoothing filter bandwidth factor (BT)
#define FT4_SYMBOL_BT 1.0  ///< symbol smoothing filter bandwidth factor (BT)

#define GFSK_CONST_K 5.336446  ///< == pi * sqrt(2 / ln(2))

//-----------------------------------------------------------------------

/// Computes a GFSK smoothing pulse.
/// The pulse is theoretically infinitely long, however,
/// here it's truncated at 3 times the symbol length.
/// This means the pulse array has to have space for 3*n_spsym elements.
/// @param[in] n_spsym Number of samples per symbol
/// @param[in] b Shape parameter (values defined for FT8/FT4)
/// @param[out] pulse Output array of pulse samples
///
  static void
gfsk_pulse( int n_spsym, double symbol_bt, double *pulse )
{
  for( int i = 0; i < 3 * n_spsym; ++i )
  {
    double t = i / (double)n_spsym - 1.5;
    double arg1 = GFSK_CONST_K * symbol_bt * ( t + 0.5 );
    double arg2 = GFSK_CONST_K * symbol_bt * ( t - 0.5 );
    pulse[i] = ( erf(arg1) - erf(arg2) ) / 2.0;
  }
}

//-----------------------------------------------------------------------

/// Synthesize waveform data using GFSK phase shaping.
/// The output waveform will contain n_sym symbols.
/// @param[in] symbols Array of symbols (tones) (0-7 for FT8)
/// @param[in] n_sym Number of symbols in the symbol array
/// @param[in] f0 Audio frequency in Hertz for the symbol 0 (base frequency)
/// @param[in] symbol_bt Symbol smoothing filter bandwidth (2 for FT8, 1 for FT4)
/// @param[in] symbol_period Symbol period (duration), seconds
/// @param[in] signal_rate Sample rate of synthesized signal, Hertz
/// @param[out] signal Output array of signal waveform samples
/// (should have space for n_sym*n_spsym samples)
///
  static void
synth_gfsk(
    const uint8_t *symbols,
    int n_sym,
    double f0,
    double symbol_bt,
    double symbol_period,
    int signal_rate,
    double *signal )
{
  int n_spsym = (int)( 0.5 + signal_rate * symbol_period ); // Samples per symbol
  int n_wave  = n_sym * n_spsym;                            // Number of output samples
  double hmod = 1.0;

  // Compute the smoothed frequency waveform.
  // Length = (nsym+2)*n_spsym samples, first and last symbols extended
  double dphi_peak = 2 * M_PI * hmod / n_spsym;

  double *dphi = NULL;
  Mem_Alloc( (void **)&dphi, (size_t)(n_wave + 2 * n_spsym) * sizeof(double) );

  // Shift frequency up by f0
  for( int i = 0; i < n_wave + 2 * n_spsym; ++i )
  {
    dphi[i] = 2 * M_PI * f0 / signal_rate;
  }

  double *pulse = NULL;
  Mem_Alloc( (void **)&pulse, (size_t)(3 * n_spsym) * sizeof(double) );
  gfsk_pulse( n_spsym, symbol_bt, pulse );

  for( int i = 0; i < n_sym; ++i )
  {
    int ib = i * n_spsym;
    for( int j = 0; j < 3 * n_spsym; ++j )
    {
      dphi[j + ib] += dphi_peak * symbols[i] * pulse[j];
    }
  }

  // Add dummy symbols at beginning and end with tone
  // values equal to 1st and last symbol, respectively
  for( int j = 0; j < 2 * n_spsym; ++j )
  {
    dphi[j] += dphi_peak * pulse[j + n_spsym] * symbols[0];
    dphi[j + n_sym * n_spsym] += dphi_peak * pulse[j] * symbols[n_sym - 1];
  }

  // Calculate and insert the audio waveform. Signal max amplitude empirical
#define FTX_SIGNAL_MAX   28000
  double phi = 0;
  int k;
  for( k = 0; k < n_wave; ++k )
  {
    // Don't include dummy symbols
    signal[k] = FTX_SIGNAL_MAX * sin( phi );
    phi = fmod( phi + dphi[k + n_spsym], M_2PI );
  }

  // My addition, clear the end of the buffer
  for( ; k < (int)ftx_signal.num_samples; ++k )
    signal[k] = 0;

  // Apply envelope shaping to the first and last symbols
  int n_ramp = n_spsym / 8;
  for( int i = 0; i < n_ramp; ++i )
  {
    double env = ( 1 - cos( 2 * M_PI * i / (2 * n_ramp) ) ) / 2;
    signal[i] *= env;
    signal[n_wave - 1 - i] *= env;
  }

  Mem_Free( (void **)&dphi );
  Mem_Free( (void **)&pulse );
} // synth_gfsk()

//-----------------------------------------------------------------------

/* Save signal in floating point format (-1 .. +1)
 * as a WAVE file using 16-bit signed integers.
static int save_wav(
    const double* signal, uint32_t num_samples, uint32_t sample_rate, const char* path)
{
    char subChunk1ID[4] = { 'f', 'm', 't', ' ' };
    uint32_t subChunk1Size = 16; // 16 for PCM
    uint16_t audioFormat = 1;    // PCM = 1
    uint16_t numChannels = 1;
    uint16_t bitsPerSample = 16;
    uint32_t sampleRate = sample_rate;
    uint16_t blockAlign = numChannels * bitsPerSample / 8;
    uint32_t byteRate = sampleRate * blockAlign;

    char subChunk2ID[4] = { 'd', 'a', 't', 'a' };
    uint32_t subChunk2Size = num_samples * blockAlign;

    char chunkID[4] = { 'R', 'I', 'F', 'F' };
    uint32_t chunkSize = 4 + (8 + subChunk1Size) + (8 + subChunk2Size);
    char format[4] = { 'W', 'A', 'V', 'E' };

    int16_t* raw_data = (int16_t*)malloc(num_samples * blockAlign);
    for (uint32_t i = 0; i < num_samples; i++)
    {
        double x = signal[i];
        if (x > 1.0)
            x = 1.0;
        else if (x < -1.0)
            x = -1.0;
        raw_data[i] = (int16_t)(0.5 + (x * 32767.0));
    }

    FILE* f = fopen(path, "wb");
    if (f == NULL)
        return -1;

    // NOTE: works only on little-endian architecture
    fwrite(chunkID, sizeof(chunkID), 1, f);
    fwrite(&chunkSize, sizeof(chunkSize), 1, f);
    fwrite(format, sizeof(format), 1, f);

    fwrite(subChunk1ID, sizeof(subChunk1ID), 1, f);
    fwrite(&subChunk1Size, sizeof(subChunk1Size), 1, f);
    fwrite(&audioFormat, sizeof(audioFormat), 1, f);
    fwrite(&numChannels, sizeof(numChannels), 1, f);
    fwrite(&sampleRate, sizeof(sampleRate), 1, f);
    fwrite(&byteRate, sizeof(byteRate), 1, f);
    fwrite(&blockAlign, sizeof(blockAlign), 1, f);
    fwrite(&bitsPerSample, sizeof(bitsPerSample), 1, f);

    fwrite(subChunk2ID, sizeof(subChunk2ID), 1, f);
    fwrite(&subChunk2Size, sizeof(subChunk2Size), 1, f);

    fwrite(raw_data, blockAlign, num_samples, f);

    fclose(f);

    free(raw_data);
    return 0;
} */

//-----------------------------------------------------------------------

/* Encode_Message()
 *
 * Encodes a text Message into a stream of 48kS/s sound
 * samples for transmission through the SSB modulator
 */
  BOOLEAN
Encode_Message( const int8_t mesg_idx )
{
  const char *message;

  // Read the message from the message entries
  message = gtk_entry_get_text( GTK_ENTRY(ftx_gui.message_entry[mesg_idx] ) );

  /*** First, pack the text data into binary message ***/
  ftx_message_t msg;
  ftx_message_rc_t rc = ftx_message_encode( &msg, NULL, message );
  if( rc != FTX_MESSAGE_RC_OK )
  {
    char text[FTX_TXT_SIZE];
    snprintf( text, FTX_TXT_SIZE, " Encoder: Parse Error: RC = %d ", (int)rc );
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label), MARKUP_RED, text );
    return( False );
  }
  else
  {
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
        MARKUP_GREEN, " Encoder: Message Parsed OK " );
  }

  // Set up parameters for FT4 | FT8
  BOOLEAN is_ft4 = ( ftx_rc.proto == FTX_PROTOCOL_FT4 );
  uint8_t num_tones    = ( is_ft4 ) ? FT4_NN            : FT8_NN;
  double symbol_period = ( is_ft4 ) ? FT4_SYMBOL_PERIOD : FT8_SYMBOL_PERIOD;
  double symbol_bt     = ( is_ft4 ) ? FT4_SYMBOL_BT     : FT8_SYMBOL_BT;

  /*** Second, encode the binary message as a sequence of FSK tones ***/
  uint8_t *tones = NULL; // Array of 79 tones (symbols)
  Mem_Alloc( (void **)&tones, (size_t)num_tones * sizeof(uint8_t) );

  // Encode tones into message payload
  if( is_ft4 )
    ft4_encode( msg.payload, tones );
  else
    ft8_encode( msg.payload, tones );

  /*** Third, convert the FSK tones into an audio signal ***/
  // Number of samples in the data signal
  ftx_signal.num_samples = (uint32_t)( num_tones * symbol_period * FTX_TX_SAMPLE_RATE + 0.5 );

  // Make number of samples a mutiple of DUC buffer size
  ftx_signal.num_samples =
    ( (ftx_signal.num_samples / DUC_NUM_IQ_SAMPLES) + 1 ) * DUC_NUM_IQ_SAMPLES;
  Mem_Alloc( (void **)&ftx_signal.signal, (size_t)ftx_signal.num_samples * sizeof(double) );

  /*** Synthesize waveform data (signal at 48kS/s SDR TX rate) ***/
  synth_gfsk(
      tones, num_tones, ftx_rc.tx_frequency, symbol_bt,
      symbol_period, FTX_TX_SAMPLE_RATE, ftx_signal.signal );

  // Free resources
  Mem_Free( (void **)&tones );

  return( True );
} // Encode_Message()

//-----------------------------------------------------------------------

