/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "callbacks.h"
#include "decode_ftx.h"
#include "shared.h"
#include "utils.h"
#include "xplanet.h"
#include "../common/utils.h"
#include <gtk/gtk.h>

//----------------------------------------------------------------------

#define FTX_DATA_COLS   4  // Number of FTx data columns (time, snr, dt, freq)
#define FTX_OPER_COLS   3  // Number of FTx columns with operational data (CQ/call/grid etc)

/* FTx_List_Store()
 *
 * Create store needed for the treeview
 */
  void
FTx_List_Store( GtkListStore **list_store, GtkTreeView *tree_view )
{
  GtkTreeModel *model;
  GtkTreeIter iter;  // Treeview parent iteration
  gchar *titles[FTX_TREEVIEW_COLS] =
  {"  TIME   ", " SNR ", "  DT ", " FREQ ", " FIELD 1        ", " FIELD 2        ", " FIELD 3 "};

  // Create list stores only if needed
  if( *list_store != NULL ) return;

  // Create bookmarks tree store
  *list_store = gtk_list_store_new(
      FTX_TREEVIEW_COLS,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING );

  // Create cell renderers for data elements (time, snr, dt, freq)
  for( uint8_t idx = 0; idx < FTX_DATA_COLS; idx++ )
  {
    GtkTreeViewColumn *column;
    GtkCellRenderer   *renderer;

    // Cell render type "text"
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer), "editable", FALSE, NULL );
    g_object_set_data( G_OBJECT(renderer), "column", GUINT_TO_POINTER(idx) );
    gtk_cell_renderer_set_alignment( renderer, 0.0, 0.0 );

    /* Create a column */
    column = gtk_tree_view_column_new_with_attributes(
        titles[idx], renderer, "text", idx, NULL );

    // Add the column to the view
    gtk_tree_view_append_column( GTK_TREE_VIEW(tree_view), column );
  } // for( uint8_t idx = 0; idx < FTX_DATA_COLS; idx++ )

  // Create cell renderers for message elements (call, cq, report, other)
  for( uint8_t idx = FTX_DATA_COLS; idx < FTX_TREEVIEW_COLS; idx++ )
  {
    GtkTreeViewColumn *column;
    GtkCellRenderer   *renderer;

    // Cell render type "text"
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer), "editable", FALSE, NULL );
    g_object_set_data( G_OBJECT(renderer), "column", GUINT_TO_POINTER(idx) );
    gtk_cell_renderer_set_alignment( renderer, 0.0, 0.0 );

    /* Create a column */
    column = gtk_tree_view_column_new_with_attributes(
        titles[idx], renderer, "markup", idx, NULL );

    // Add the column to the view
    gtk_tree_view_append_column( GTK_TREE_VIEW(tree_view), column );
  } // for( uint8_t idx = 0; idx < FTX_DATA_COLS; idx++ )

  // Create a model to insert tree store
  Flag[HERMES2_BLOCK_GTK_SIGNALS] = True;
  model = GTK_TREE_MODEL( *list_store );
  gtk_tree_view_set_model( tree_view, model );

  // Set the selection mode to single
  GtkTreeSelection *selection = gtk_tree_view_get_selection( tree_view );
  gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

  // Get new row if available
  gtk_tree_model_get_iter_first( GTK_TREE_MODEL(*list_store), &iter );

  // Create the required number of rows
  for( uint8_t idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
  {
    gtk_list_store_append( *list_store, &iter );
  }

  // Destroy model automatically with treeview
  g_object_unref( model );
  Flag[HERMES2_BLOCK_GTK_SIGNALS] = False;

} // FTx_List_Store()

//----------------------------------------------------------------------

/* FTx_Set_Markup()
 *
 * Sets a marked-up text in "mkup", depending on contents of "word"
 */
  static void
FTx_Set_Markup( gchar *mkup, size_t mkup_siz, char *word )
{
  // Local temporary string to hold final text to markup
  char text[HERMES2_MARKUP_SIZE];

  // Escape hashed callsigns
  if( strstr(word, "<" ) != NULL )
  {
    gchar *temp = g_markup_escape_text( word, -1 );
    Strlcpy( text, temp, HERMES2_MARKUP_SIZE );
    g_free( temp );
  }
  else
  {
    Strlcpy( text, word, HERMES2_MARKUP_SIZE );
  }

  // Markup "CQ" or "CQ_...."
  if( Is_CQ(text) )
  {
    snprintf( mkup, mkup_siz, MKUP_GREEN, text );
    return;
  }

  // Markup own Callsign
  if( strstr(text, ftx_rc.own_call) != NULL )
  {
    snprintf( mkup, mkup_siz, MKUP_RED, text );
    return;
  }

  // Markup DX Callsign
  if( strlen(ftx_qso_record.dx_call) &&
    strstr(text, ftx_qso_record.dx_call) != NULL )
  {
    snprintf( mkup, mkup_siz, MKUP_ORANGE, text );
    return;
  }

  // Highlight user-defined string
  if( strlen(ftx_rc.highlight) &&
      (strstr(text, ftx_rc.highlight) != NULL) )
  {
    snprintf( mkup, mkup_siz, MKUP_CYAN, text );
    return;
  }

  // Print regular "words"
  snprintf( mkup, mkup_siz, MKUP_TEXT, text );

} // FTx_Set_Markup()

//----------------------------------------------------------------------

/* FTx_Bypass_Line()
 *
 * Indicate that a Message line need not be displayed if
 * it does not contain text that the user is interested in
 */
  static BOOLEAN
FTx_Bypass_Line( char *line )
{
  BOOLEAN bypass = True, req = False;

  // If user is interested in lines with a CQ call
  if( ftx_rc.with_cq )
  {
    char *cq = strstr(line, " CQ");
    if( cq != NULL )
    {
      cq++;
      if( !Is_CQ(cq) ) bypass = False;
    }
    req = True;
  }

  // If user is interested in lines with his own call included
  if( ftx_rc.calling_me )
  {
    // Create a copy of the callsign with a space before and after it
    char temp[FTX_CALL_STR_SIZE];
    Strlcpy( temp, " ", sizeof(temp) );
    Strlcat( temp, ftx_rc.own_call, sizeof(temp) );
    Strlcat( temp, " ", sizeof(temp) );

    // Bypass line if own call is not included
    if( strstr(line, temp) != NULL )
      bypass = False;
    req = True;
  }

  // If user is interested in lines with his own call included
  if( ftx_rc.dx_call_show )
  {
    // Create a copy of the callsign with a space before and after it
    char temp[FTX_CALL_STR_SIZE];
    Strlcpy( temp, " ", sizeof(temp) );
    Strlcat( temp, ftx_qso_record.dx_call, sizeof(temp) );
    Strlcat( temp, " ", sizeof(temp) );

    // Bypass line if own call is not included
    if( strstr(line, temp) != NULL )
      bypass = False;
    req = True;
  }

  // If user is interested in lines with highlighted text
  if( ftx_rc.highlighted )
  {
    if( strstr(line, ftx_rc.highlight) != NULL )
      bypass = False;
    req = True;
  }

  // If user has not chosen specific contents in Messages
  if( !req ) bypass = False;

  return( bypass );
} // FTx_Bypass_Line()

//----------------------------------------------------------------------

typedef struct _FTX_DATA
{
  char  **mesg_data;
  int16_t data_idx;
  int     num_decoded;
  int     hashtable_size;
  double  max_magn;
  GtkListStore *list_store;
} decoder_data_t;

/* _Display_FTx_Data()
 *
 * Idle Add function to display Decoder Data
 */
  static gboolean
_Display_FTx_Data( gpointer data )
{
  GtkListStore *list_store = ((decoder_data_t *)data)->list_store;
  char  **message_data     = ((decoder_data_t *)data)->mesg_data;
  int16_t data_idx         = ((decoder_data_t *)data)->data_idx;
  int     num_decoded      = ((decoder_data_t *)data)->num_decoded;
  int     hashtable_size   = ((decoder_data_t *)data)->hashtable_size;
  double  max_magn         = ((decoder_data_t *)data)->max_magn;
  static GtkTreeIter iter;  // Treeview parent iteration


  // Clear the Treeview rows if enabled
  if( ftx_rc.auto_clear )
  {
    gtk_tree_model_get_iter_first( GTK_TREE_MODEL(ftx_gui.mesg_store), &iter );
    for( uint8_t row = 0; row < FTX_TREEVIEW_ROWS; row++ )
    {
      for( uint8_t col = 0; col < FTX_TREEVIEW_COLS; col++ )
      {
        gtk_list_store_set( ftx_gui.mesg_store, &iter, col, "", -1 );
      }
      gtk_tree_model_iter_next( GTK_TREE_MODEL(ftx_gui.mesg_store), &iter );
    }
  }

  // Go to the top of the treeview model
  gtk_tree_model_get_iter_first( GTK_TREE_MODEL(list_store), &iter );

  // Signal the Xplanet display
  BOOLEAN xplanet_ok = False;
  if( ftx_rc.band_xplanet && FTx_Open_Xplanet_Files() )
    xplanet_ok = True;

  // Fill in all rows of the treeview with the decoded messages
  uint8_t cnt = 0; // Count number of messages displayed in xplanet
  for( uint8_t row = 0; row < FTX_TREEVIEW_ROWS; row++ )
  {
    // Set space-separated individual words in the message to the tree store
    uint8_t msg_idx = 0;
    char temp[FTX_MESG_DATA_SIZE];

    // Moving backwards in list to maintain chronological order in display
    data_idx--;
    if( data_idx < 0 ) data_idx += FTX_TREEVIEW_ROWS;
    cnt++;

     // Current row's Message data ("words")
    memcpy( temp, message_data[data_idx], FTX_MESG_DATA_SIZE );

    /* Bypass Message line if it does not contain any of the user's choices
     * regarding what data to display (CQ, Own Callsign, Highlighted text etc) */
    if( FTx_Bypass_Line(temp) ) continue;

    /* Fill in all rows of the treeview with the decoded messages.
     * First four columns are plain text (time, snr, dt, freq) */
    for( uint8_t col = 0; col < FTX_DATA_COLS; col++ )
    {
      char *word;

      // Point to the start of a word in the message
      word = &temp[msg_idx];  // A word in the temp row above

      // Look for the next space in the message or end of string
      while( (temp[msg_idx] != '\0') && (temp[msg_idx] != ' ') )
        msg_idx++;

      // Replace a space with string terminator
      if( temp[msg_idx] == ' ' )
      {
        temp[msg_idx] = '\0';
        msg_idx++;
      }

      // Set the word to the selected column
      gtk_list_store_set( list_store, &iter, col, word, -1 );

    } // for( col = 0, col < FTX_DATA_COLS; col++ )

    char *word[FTX_OPER_COLS];
    uint8_t col = FTX_TREEVIEW_COLS - FTX_OPER_COLS;
    for( uint8_t word_idx = 0; word_idx < FTX_OPER_COLS; word_idx++ )
    {
      // Point to the start of a word in the message
      word[word_idx] = &temp[msg_idx];  // A word in the temp row above

      // Look for the next space in the message or end of string
      while( (temp[msg_idx] != '\0') && (temp[msg_idx] != ' ') )
        msg_idx++;

      // Replace a space with string terminator
      if( temp[msg_idx] == ' ' )
      {
        temp[msg_idx] = '\0';
        msg_idx++;
      }

      // Markup word if enabled and set to the selected column
      gchar mkup[HERMES2_MARKUP_SIZE];
      if( ftx_rc.colorize )
      {
        FTx_Set_Markup( mkup, sizeof(mkup), word[word_idx] );
        gtk_list_store_set( list_store, &iter, col, mkup, -1 );
      }
      else if( word[word_idx][0] == '<' )
      {
        // Local temporary string to hold final text to markup
        char text[HERMES2_MARKUP_SIZE];
        gchar *tmp = g_markup_escape_text( word[word_idx], -1 );
        Strlcpy( text, tmp, sizeof(text) );
        gtk_list_store_set( list_store, &iter, col, text, -1 );
        g_free( tmp );
      }
      else gtk_list_store_set( list_store, &iter, col, word[word_idx], -1 );

      // Next column
      col++;

    } // for( uint8_t word_idx = 0; word_idx < FTX_OPER_COLS; word_idx++ )
    gtk_tree_model_iter_next( GTK_TREE_MODEL(list_store), &iter );

    // If Xplanet display is enabled
    if( xplanet_ok && (cnt <= num_decoded) )
    {
      // Test that Grid locator is valid. It should be in last word[].
      if( Is_Locator(word[FTX_OPER_COLS-1]) )
      {
        // Prepare xplanet files. Call should be in second from last word[].
        FTx_Prepare_Xplanet( word[FTX_OPER_COLS-1], word[FTX_OPER_COLS-2] );
      }
    } // if( ftx_rc.band_xplanet )

  } // for( uint8_t row = 0; row < FTX_TREEVIEW_ROWS; row++ )

  // If Xplanet display is enabled
  if( xplanet_ok )
  {
    FTx_Close_Xplanet_Files();
    Launch_Xplanet( ftx_rc.xplanet_command );
  }

  // Display Decoder status
  gchar text[24];

  snprintf( text, sizeof(text), "%2d", num_decoded );
  gtk_label_set_text( GTK_LABEL(ftx_gui.decoded_label), text );

  snprintf( text, sizeof(text), "%3d", hashtable_size );
  gtk_label_set_text( GTK_LABEL(ftx_gui.hashtable_label), text );

  snprintf( text, sizeof(text), "%+4.1f", max_magn );
  gtk_label_set_text( GTK_LABEL(ftx_gui.magn_label), text );

  // Clear the Decoder messages array if enabled
  if( ftx_rc.auto_clear )
  {
    for( uint8_t idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
      message_data[idx][0] = '\0';
  }

  return( FALSE );
} // _Display_FTx_Data()

/* Display_FTx_Data()
 *
 * Enters Ftx Message data (time, snr, dT, freq, message)
 * into one row of the given Message or Receive GtkListStore
 */

static decoder_data_t decoder_data;

  void
Display_FTx_Data(
    char  **msg_data,
    int16_t data_idx,
    GtkListStore *list_store )
{
  decoder_data.mesg_data  = msg_data;
  decoder_data.data_idx   = data_idx;
  decoder_data.list_store = list_store;

  Flag[FTX_DEC_DISPLAY_CB] = True;
  g_idle_add_full(
      G_PRIORITY_DEFAULT_IDLE,
      _Display_FTx_Data,
      (gpointer)&decoder_data,
      FTx_g_Idle_cb );

} // Display_FTx_Data()

//----------------------------------------------------------------------

/* Decoder_Status()
 *
 * Enters the FTX Decoder status (Number of decoded
 * messages, maximum signal strength, hash table size )
 */
  void
Decoder_Status( int num_decoded, int hashtable_size, double max_magn )
{
  decoder_data.num_decoded    = num_decoded;
  decoder_data.hashtable_size = hashtable_size;
  decoder_data.max_magn       = max_magn;
} // Decoder_Status()

//----------------------------------------------------------------------

