/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "decode_ftx.h"
#include "callback_func.h"
#include "decode.h"
#include "display.h"
#include "encode_ftx.h"
#include "interface.h"
#include "message.h"
#include "monitor.h"
#include "sequence.h"
#include "shared.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/display.h"
#include "../Hermes2/sound.h"

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <sys/time.h>
#include <unistd.h>

#define KMIN_SCORE               10   // Minimum sync score threshold for candidates
#define KFREQ_OSR                2    // Frequency oversampling rate (bin subdivision)
#define KTIME_OSR                2    // Time oversampling rate (symbol subdivision)
#define KMAX_DECODED_MESSAGES    50
#define KMAX_CANDIDATES          140
#define KLDPC_ITERATIONS         25
#define CALLSIGN_HASHTABLE_SIZE  256

//-----------------------------------------------------------------------

static struct
{
  char callsign[12]; ///> Up to 11 symbols of callsign + trailing zeros (always filled)
  uint32_t hash;     ///> 8 MSBs contain the age of callsign; 22 LSBs contain hash value
} callsign_hashtable[CALLSIGN_HASHTABLE_SIZE];

//-----------------------------------------------------------------------

static int callsign_hashtable_size;

static void hashtable_init( void )
{
  callsign_hashtable_size = 0;
  memset( callsign_hashtable, 0, sizeof(callsign_hashtable) );
}

//-----------------------------------------------------------------------

  static void
hashtable_cleanup( uint8_t max_age )
{
  for( int idx_hash = 0; idx_hash < CALLSIGN_HASHTABLE_SIZE; ++idx_hash )
  {
    if( callsign_hashtable[idx_hash].callsign[0] != '\0' )
    {
      uint8_t age = (uint8_t)( callsign_hashtable[idx_hash].hash >> 24 );
      if( age > max_age )
      {
        // free the hash entry
        callsign_hashtable[idx_hash].callsign[0] = '\0';
        callsign_hashtable[idx_hash].hash = 0;
        callsign_hashtable_size--;
      }
      else
      {
        // increase callsign age
        callsign_hashtable[idx_hash].hash =
          ( ((uint32_t)age + 1u) << 24 ) | ( callsign_hashtable[idx_hash].hash & 0x3FFFFFu );
      }
    }
  }
} // hashtable_cleanup()

//-----------------------------------------------------------------------

  static void
hashtable_add( const char *callsign, uint32_t hash )
{
  uint16_t hash10 = ( hash >> 12 ) & 0x3FFu;
  int idx_hash = ( hash10 * 23 ) % CALLSIGN_HASHTABLE_SIZE;
  while( callsign_hashtable[idx_hash].callsign[0] != '\0' )
  {
    if( ((callsign_hashtable[idx_hash].hash & 0x3FFFFFu) == hash) &&
        (strcmp(callsign_hashtable[idx_hash].callsign, callsign) == 0) )
    {
      // reset age
      callsign_hashtable[idx_hash].hash &= 0x3FFFFFu;
      return;
    }
    else
    {
      // Move on to check the next entry in hash table
      idx_hash = ( idx_hash + 1 ) % CALLSIGN_HASHTABLE_SIZE;
    }
  }

  callsign_hashtable_size++;
  strncpy( callsign_hashtable[idx_hash].callsign, callsign, 11 );
  callsign_hashtable[idx_hash].callsign[11] = '\0';
  callsign_hashtable[idx_hash].hash = hash;

} // hashtable_add()

//-----------------------------------------------------------------------

  static BOOLEAN
hashtable_lookup( ftx_callsign_hash_type_t hash_type, uint32_t hash, char *callsign )
{
  uint8_t hash_shift =
    ( hash_type == FTX_CALLSIGN_HASH_10_BITS ) ? 12 :
    ( (hash_type == FTX_CALLSIGN_HASH_12_BITS) ? 10 : 0 );

  uint16_t hash10 = ( hash >> (12 - hash_shift) ) & 0x3FFu;
  int idx_hash    = ( hash10 * 23 ) % CALLSIGN_HASHTABLE_SIZE;

  while( callsign_hashtable[idx_hash].callsign[0] != '\0' )
  {
    if( ((callsign_hashtable[idx_hash].hash & 0x3FFFFFu) >> hash_shift) == hash )
    {
      stpcpy( callsign, callsign_hashtable[idx_hash].callsign );
      return True;
    }
    // Move on to check the next entry in hash table
    idx_hash = ( idx_hash + 1 ) % CALLSIGN_HASHTABLE_SIZE;
  }

  callsign[0] = '\0';
  return False;
} // hashtable_lookup()

//-----------------------------------------------------------------------

static ftx_callsign_hash_interface_t hash_if =
{
  .lookup_hash = hashtable_lookup,
  .save_hash   = hashtable_add
};

//-----------------------------------------------------------------------

// Audio signal samples buffers
#define SIGNAL_BUFS   2     // Number of signal buffers
static double    *signal_bufs[SIGNAL_BUFS] = { NULL, NULL };
static int        signal_idx;      // Index to current signal buffer
static uint8_t    signal_buf;      // Current signal buffer
static size_t     signal_siz;      // Size of signal buffers
static int        num_samples;     // Number of samples per time slot
static struct tm  slot_start;      // Start Time of a time slot
static int64_t    ave_dt;          // Average timing error (Dt) of stations
static monitor_t  mon;
static monitor_config_t mon_cfg;

#define TIMESHIFT_INIT  900000     // Initial time shift in uS, empirical.

// Shift in start time of signal collecting for the monitor
static int64_t time_shift = TIMESHIFT_INIT;

// New average timing error calculated
static BOOLEAN new_time_shift;

//-----------------------------------------------------------------------

/* Decode_FTx()
 *
 * Controls the FTx Message decoding process
 */
  static void
Decode_FTx( void )
{
  const   ftx_waterfall_t *wf = &mon.wf;
  int64_t dts_summed = 0;  // Number of station Dt's summed for the average

  // Messages buffer index
  static int16_t data_idx = 0;


  // Allocate Decoded messages buffer
  if( mesg_data == NULL )
  {
    Mem_Alloc( (void **)&mesg_data, sizeof(char *) * FTX_TREEVIEW_ROWS );
    for( uint8_t idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
    {
      mesg_data[idx] = NULL;
      Mem_Alloc( (void **)&mesg_data[idx], sizeof(char *) * FTX_MESG_DATA_SIZE );
    }
  }

  // Find top candidates by Costas sync score and localize them in time and frequency
  ftx_candidate_t *candidate_list = NULL;
  Mem_Alloc( (void **)&candidate_list, KMAX_CANDIDATES * sizeof(ftx_candidate_t) );
  int num_candidates = ftx_find_candidates( wf, KMAX_CANDIDATES, candidate_list, KMIN_SCORE );

  // Hash table for decoded messages (to check for duplicates)
  int num_decoded = 0;
  size_t siz = KMAX_DECODED_MESSAGES * sizeof(ftx_message_t);

  ftx_message_t *decoded = NULL;
  Mem_Alloc( (void **)&decoded, siz );

  ftx_message_t **decoded_hashtable = NULL;
  Mem_Alloc( (void **)&decoded_hashtable, siz );

  // Initialize hash table pointers
  for( int i = 0; i < KMAX_DECODED_MESSAGES; ++i )
    decoded_hashtable[i] = NULL;

  // Indicate start of FTx messages decode
  if( ftx_rc.proto == FTX_PROTOCOL_FT8 )
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
        MARKUP_GREEN, " Decoder: Decoding FT8 Messages " );
  else
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
        MARKUP_GREEN, " Decoder: Decoding FT4 Messages " );

  // Go over candidates and attempt to decode messages
  ave_dt = 0;
  new_time_shift = False;
  for( int idx = 0; idx < num_candidates; ++idx )
  {
    const ftx_candidate_t *cand = &candidate_list[idx];
    double freq_hz = ( mon.min_bin + cand->freq_offset +
        (double)cand->freq_sub / wf->freq_osr ) / mon.symbol_period;
    double time_sec =
      ( cand->time_offset + (double)cand->time_sub / wf->time_osr ) * mon.symbol_period;

#ifdef WATERFALL_USE_PHASE
    // int resynth_len = 12000 * 16;
    // double resynth_signal[resynth_len];
    // for( int pos = 0; pos < resynth_len; ++pos)
    // {
    //     resynth_signal[pos] = 0;
    // }
    // monitor_resynth(mon, cand, resynth_signal);
    // char resynth_path[80];
    // sprintf(resynth_path, "resynth_%04f_%02.1f.wav", freq_hz, time_sec);
    // save_wav(resynth_signal, resynth_len, 12000, resynth_path);
#endif

    ftx_message_t       message;
    ftx_decode_status_t status;
    if( !ftx_decode_candidate(wf, cand, KLDPC_ITERATIONS, &message, &status) )
    {
      continue;
    }

    int idx_hash = message.hash % KMAX_DECODED_MESSAGES;
    BOOLEAN found_empty_slot = False;
    BOOLEAN found_duplicate  = False;

    do
    {
      if( decoded_hashtable[idx_hash] == NULL )
      {
        found_empty_slot = True;
      }
      else if(
          (decoded_hashtable[idx_hash]->hash == message.hash) &&
          (memcmp( decoded_hashtable[idx_hash]->payload,
                   message.payload, sizeof(message.payload)) == 0) )
      {
        found_duplicate = True;
      }
      else
      {
        // Move on to check the next entry in hash table
        idx_hash = ( idx_hash + 1 ) % KMAX_DECODED_MESSAGES;
      }
    }
    while( !found_empty_slot && !found_duplicate );

    if( found_empty_slot )
    {
      // Fill the empty hashtable slot
      memcpy( &decoded[idx_hash], &message, sizeof(message) );
      decoded_hashtable[idx_hash] = &decoded[idx_hash];
      ++num_decoded;

      char text[FTX_TXT_SIZE];
      ftx_message_rc_t unpack_status = ftx_message_decode( &message, &hash_if, text );
      if( unpack_status != FTX_MESSAGE_RC_OK )
      {
        snprintf( text, FTX_TXT_SIZE,
            " Decoder: Error [%d] while unpacking ", (int)unpack_status );
        Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label), MARKUP_RED, text );
      }
      else
      {
        // Fake WSJT-X-like output for now
        double snr = cand->score * 0.5; // TODO: compute better approximation of SNR

        // Print Decoder output to the message buffer
        snprintf( mesg_data[data_idx], FTX_MESG_DATA_SIZE,
            "%02d:%02d:%02d %+05.1f %+05.2f %d %s",
            slot_start.tm_hour, slot_start.tm_min, slot_start.tm_sec,
            snr, time_sec, (uint16_t)freq_hz, text );

        // Summate station DT's (timing errors)
        ave_dt += (int64_t)( time_sec * 1000000 );
        dts_summed++;

        // Advance data index and keep in range
        data_idx++;
        data_idx %= FTX_TREEVIEW_ROWS;

      } // if( unpack_status != FTX_MESSAGE_RC_OK )
    } // if( found_empty_slot )
  } // for( int idx = 0; idx < num_candidates; ++idx )

  // Calculate average timing error and signal its use.
  if( dts_summed )
  {
    ave_dt /= dts_summed;
    new_time_shift = True;
  }

  /* Display Decoder results and status in the Messages Treeview.
   * The functions must be called in the order below */
  Decoder_Status( num_decoded, callsign_hashtable_size, mon.max_mag );
  Display_FTx_Data( mesg_data, data_idx, ftx_gui.mesg_store );
  if( !num_decoded ) time_shift = TIMESHIFT_INIT;

  hashtable_cleanup( 10 );
  Mem_Free( (void **)&candidate_list );
  Mem_Free( (void **)&decoded );
  Mem_Free( (void **)&decoded_hashtable );

  return;
} // Decode_FTx()

//-----------------------------------------------------------------------

  static void *
Decoder_Thrd( void *arg )
{
  ftx_rc.decoder_thrd_running = True;

  // Process and accumulate audio data in a monitor/waterfall instance
  for( int frame_pos = 0;
      frame_pos + mon.block_size <= num_samples;
      frame_pos += mon.block_size )
  {
    // Process the waveform data frame by frame
    monitor_process( &mon, signal_bufs[signal_buf] + frame_pos );
  }

  // Decode FTx messages
  Decode_FTx();

  ftx_rc.decoder_thrd_running = False;
  return( NULL );
} // Decoder_Thrd()

//-----------------------------------------------------------------------

#define SCALE_SIGNAL     10000  // Down scale magnitude of signal samples
#define FTX_DOWNSAMPLE   4      // Number of SDR samples to average

/* FTx_Decoder_Run()
 *
 * A thread function that runs the FTx Decoder code while FTx reception
 * and decoding is enabled. This function also contains the timing code
 * that controls the start and stop of audio samples buffering for the
 * FTx monitor/waterfall but additionally controls the timing of the
 * Message Encoder and the Transmit functions if a message is to be sent
 */
  void *
FTx_Decoder_Run( void *protocol )
{
  ftx_protocol_t proto = *( (ftx_protocol_t *)protocol );
  double slot_period   = (proto == FTX_PROTOCOL_FT8) ? FT8_SLOT_TIME : FT4_SLOT_TIME;
  num_samples          = (int)( slot_period * FTX_RX_SAMPLE_RATE );

  /* Index to digimode samples buffer currently
   * in use and sample rate downsample counter */
  static uint8_t digi_buf_cnt = 0;
  uint8_t downsample = 0;

  // Digimode buffer index
  static uint32_t digi_buf_idx = 0;

  // Wait for sample buffers full
  static BOOLEAN wait;

  // Used in slot time synchronizations
  static int64_t
    dec_uwait_prev  = -1,  // Used to time the FTx decoder
    rcve_uwait_prev = -1,  // Used to time received DSP samples buffering 
    trx_uwait_prev  = -1,  // Used to time the FTx transmit process 
    slot_period_usec;      // FTx slot period in microsec


  // Initialize program
  mon_cfg.f_min       = F_MIN;
  mon_cfg.f_max       = F_MAX;
  mon_cfg.sample_rate = FTX_RX_SAMPLE_RATE;
  mon_cfg.time_osr    = KTIME_OSR;
  mon_cfg.freq_osr    = KFREQ_OSR;
  mon_cfg.protocol    = proto;
  signal_buf = 0;
  signal_siz = 0;

  monitor_init( &mon, &mon_cfg );
  hashtable_init();

  slot_period_usec = (int64_t)( slot_period * 1000000 ); // Slot period in uSec
  Flag[GUEST_RECEIVING] = True;

  // Allocate samples buffer of the Decoder
  signal_siz = (size_t)num_samples * sizeof(double);
  Mem_Alloc( (void **)&signal_bufs[0], signal_siz );
  Mem_Alloc( (void **)&signal_bufs[1], signal_siz );
  memset( signal_bufs[0], 0, signal_siz );
  memset( signal_bufs[1], 0, signal_siz );

  // Keep looping till reception is stopped
  pthread_t decoder_thrd = 0;
  while( Transceiver[Indices.TRx_Index]->receive_active &&
         !ftx_rc.decoder_exit && !Flag[GUEST_QUIT] )
  {
    // Wait for digimode buffer to fill with audio samples from demodulator
    if( wait && !ftx_rc.decoder_exit )
    {
      sem_wait( &digimode_semaphore );
      wait = False;
    }

    // Store audio samples in ftx decoder buffer, downsampled (48 to 12 kS/s)
    for( digi_buf_idx = 0; digi_buf_idx < DIGIMODE_BUFFER_SIZE; digi_buf_idx++ )
    {
      // Scale magnitude of signals to suit the decoder and save in buffer
      if( signal_idx < num_samples )
      {
        signal_bufs[signal_buf][signal_idx] +=
          (double)digimode_buffer[digi_buf_cnt][digi_buf_idx] / SCALE_SIGNAL;

        // Count number of samples averaged for downsampling
        downsample++;
        if( downsample >= FTX_DOWNSAMPLE )
        {
          signal_idx++;
          downsample = 0;
        }
      }
    } // for( digi_buf_idx = 0; digi_buf_idx < DIGIMODE_BUFFER_SIZE; digi_buf_idx++ )

    // Increment current audio buffer output index
    digi_buf_cnt++;
    digi_buf_cnt %= NUM_DIGIMODE_BUFFERS;

    // If sync between ring buffer input and output indices is lost,
    // sem_wait is not enabled, to allow the input index to catch up
    if( digi_buf_input == digi_buf_cnt )
      wait = True;

    // Get Real Time in sec and nsec
    struct timespec spec;
    clock_gettime( CLOCK_REALTIME, &spec );

    // Real Time and wait time in micro seconds
    int64_t rcve_utime, rcve_uwait, dec_utime, dec_uwait, trx_utime, trx_uwait;
    rcve_utime = (int64_t)( (double)spec.tv_sec * 1000000 + (double)spec.tv_nsec / 1000 );

    // Transmitter time is kept nominal like receive time
    trx_utime = rcve_utime;

    // Decoder time is set forward 1 sec to produce a decode early
    dec_utime = rcve_utime + 1000000;

    // Gradually summate DT, for the timing error average
    if( new_time_shift )
    {
      time_shift     += ave_dt / 4; // Asymptotically summate the timing error average
      rcve_uwait_prev = 0;          // Clear previous wait time to avoid false decodes
      new_time_shift  = False;
    }

    // Wait time is the modulus of time relative to slot time
    rcve_utime -= time_shift;
    rcve_uwait  = rcve_utime % slot_period_usec;

    /*** Here one time slot, corrected for the average DT, is completed.  ***
     *** This time slot is used for buffering DSP samples for the decoder ***/
    if( rcve_uwait_prev >= rcve_uwait )
    {
      // Calculate the slot start time
      time_t start = (time_t)( (double)spec.tv_sec + (double)spec.tv_nsec / 1.0E09 );
      gmtime_r( &start, &slot_start );

      // Reset internal variables for the next time slot
      monitor_reset( &mon );
      signal_idx = 0;
      signal_buf++;
      signal_buf %= SIGNAL_BUFS;
      memset( signal_bufs[0], 0, signal_siz );
      memset( signal_bufs[1], 0, signal_siz );

    } // if( rcve_uwait_prev <= rcve_uwait )
    rcve_uwait_prev = rcve_uwait;
 
    /*** Here we start the FTx Decoder about 1 sec before the end ***
     *** of the time slot, in order to have a decode in time for  ***
     *** the next time slot, for the auto-sequencer to do its job ***/
    dec_uwait = dec_utime % slot_period_usec;
    if( dec_uwait_prev >= dec_uwait )
    {
      // Create a thread for separate decoding
      if( !Pthread_Create(&decoder_thrd, NULL, Decoder_Thrd, NULL,
            _("Failed to create FTx Decoder thread")) )
        return( FALSE );

      /* Even slots are those whose start time is an even multiple of slot period.
       * The test below detects an _odd_ time slot (boolean true), but it will be
       * used in the next time slot to determine if its even or odd, for the Tx */
      ftx_rc.slot_for_tx = (BOOLEAN)( (trx_utime / slot_period_usec + 1) & 0x01 );
      ftx_rc.slot_for_tx = ftx_rc.slot_for_tx ^ ftx_rc.tx_even;

    } // if( dec_uwait_prev <= dec_uwait )
    dec_uwait_prev = dec_uwait;

    /*** Here we transmit a message if one is ready and queued and if ****
    **** the slot is right, e.g. even or odd, as selected by the user ***/
    trx_uwait = trx_utime % slot_period_usec;
    if( (trx_uwait_prev >= trx_uwait) &&  // Start of time slot
        ftx_rc.tx_enabled &&              // Transmitter is enabled
        ftx_rc.slot_for_tx &&             // Slot is as user's selection (odd or even)
        ACTIVE_MESSAGE )                  // A message has been queued (CQ or std message)
    {
      // Encode the selected message and produce samples for the Modulator in 'signal'
      if( Encode_Message(ftx_rc.active_mesg_idx) )
      {
        // Markup selected message in red, if encoded OK
        if( CQ_MESSAGE )
          Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_RED, " Send CQ " );
        else
          Set_Label_Markup( GTK_LABEL(ftx_gui.send_label[ftx_rc.active_mesg_idx]),
              MARKUP_RED, label_numb[ftx_rc.active_mesg_idx] );

        // Designate the FTx Modulator as the active Tx modulator
        Flag[HERMES2_SEND_DUC_PACKET] = True;
        Modulator = Modulate_FTx;
        MOX_Control( MOX_ON );
      }
      else
      {
        // Markup selected message in grey, if encoding failed
        if( CQ_MESSAGE )
          Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_GREY, " Send CQ " );
        else
          Set_Label_Markup( GTK_LABEL(ftx_gui.send_label[ftx_rc.active_mesg_idx]),
              MARKUP_GREY, label_numb[ftx_rc.active_mesg_idx] );

      } // if( Encode_Message(ftx_rc.active_mesg_idx, ftx_signal.signal) )

    } // if( dec_uwait_prev <= dec_uwait )
    trx_uwait_prev = trx_uwait;

  } // while( Transceiver[Indices.TRx_Index]->receive_active )

  // Wait for decoder thread to exit
  if( ftx_rc.decoder_thrd_running && decoder_thrd )
    pthread_join( decoder_thrd, NULL );

  // Clear Guest Receive flags
  Flag[GUEST_RECEIVE_MODE] = False;
  Flag[GUEST_RECEIVING]    = False;
  Init_Semaphore( &digimode_semaphore, False );

  monitor_free( &mon );
  Mem_Free( (void **)&signal_bufs[0] );
  Mem_Free( (void **)&signal_bufs[1] );

  if( !Flag[FTX_DEC_DISPLAY_CB] )
    Free_Mesg_Buffer();

  return( FALSE );
} // FTx_Decoder_Run()

//-----------------------------------------------------------------------

// Frees the Messages buffer
  void
Free_Mesg_Buffer( void )
{
  if( mesg_data != NULL )
  {
    for( uint8_t idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
    {
      Mem_Free( (void **)&mesg_data[idx] );
      mesg_data[idx] = NULL;
    }

    Mem_Free( (void **)&mesg_data );
    mesg_data = NULL;
  }
} // Free_Mesg_Buffer()

//-----------------------------------------------------------------------

// Clears the Messages buffer
  void
Clear_Mesg_Buffer( void )
{
  // Clear the Messages buffer
  if( !Flag[FTX_DEC_DISPLAY_CB] && (mesg_data != NULL) )
  {
    for( uint8_t idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
      memset( mesg_data[idx], 0, FTX_MESG_DATA_SIZE );
  }
} // Clear_Mesg_Buffer()

//-----------------------------------------------------------------------

