/*
 *  This program is free software; you can redistribute it and/or
*  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "callback_func.h"
#include "decode_ftx.h"
#include "utils.h"
#include "xplanet.h"
#include "../Hermes2/callback_func.h"
#include <gtk/gtk.h>
#include <stdio.h>
#include <time.h>

//----------------------------------------------------------------------

/* FTx_Std_Messages()
 *
 * Creates apropriate Standard Messages when a row
 * is activated in the Messages Treeview.
 */
  void
FTx_Std_Messages( void )
{
  gchar text[FTX_TXT_SIZE];

  // Abort if no valid DX callsign received
  if( !ftx_qso_record.dx_call_received )
  {
    // Clear Message Entries
    for( uint8_t idx = 0; idx < NUM_MESG_ENTRIES - 1; idx++ )
      gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[idx]), "" );

    // Make the Send Message button labels grey to indicate not ready
    for( uint8_t idx = 0; idx < NUM_SEND_LABELS; idx++ )
    {
      Set_Label_Markup(GTK_LABEL(ftx_gui.send_label[idx]), MARKUP_GREY, label_numb[idx] );
    }

    ftx_rc.active_mesg_idx = FTX_NO_ACTIVE_MESG;
    return;
  }

  // Respond to a CQ call by DX
  snprintf( text, FTX_TXT_SIZE, "%s %s ", ftx_qso_record.dx_call, ftx_rc.own_call );
  Strlcat( text, ftx_rc.own_locator, 5 ); // cat's first 4 locator chars
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_REPLY_CQ_ENTRY]), text );

  // Give the DX station its SNR report
  snprintf( text, FTX_TXT_SIZE, "%s %s %3s",
      ftx_qso_record.dx_call, ftx_rc.own_call, ftx_qso_record.dx_snr );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_SNR_ENTRY]), text );

  // Confirm reception of own SNR and give DX it SNR report
  snprintf( text, FTX_TXT_SIZE, "%s %s R%3s",
      ftx_qso_record.dx_call, ftx_rc.own_call, ftx_qso_record.dx_snr );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_R_SNR_ENTRY]), text );

  // Acknowledge reception of SNR and completion of QSO
  snprintf( text, FTX_TXT_SIZE, "%s %s %s",
      ftx_qso_record.dx_call, ftx_rc.own_call, "RRR" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_RRR_ENTRY]), text );

  // Acknowledge reception of SNR and completion of QSO and 73's
  snprintf( text, FTX_TXT_SIZE, "%s %s %s",
      ftx_qso_record.dx_call, ftx_rc.own_call, "RR73" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_RR73_ENTRY]), text );

  // Give the DX station 73's
  snprintf( text, FTX_TXT_SIZE, "%s %s %s",
      ftx_qso_record.dx_call, ftx_rc.own_call, "73" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_73_ENTRY]), text );

  // Make the Send Message button labels green to indicate ready and set flag
  for( uint8_t idx = 0; idx < NUM_SEND_LABELS; idx++ )
  {
    Set_Label_Markup( GTK_LABEL(ftx_gui.send_label[idx]), MARKUP_GREEN, label_numb[idx] );
  }
  ftx_rc.messages_ready = True;

} // FTx_Std_Messages()

//----------------------------------------------------------------------

/* Ftx_QSO_Details()
 *
 * Displays relevant data in the DX Station Details labels
 */
  static void
Ftx_QSO_Details( void )
{
  // Temporary text string
  gchar text[FTX_TXT_SIZE];

  // Display the DX Callsign if a valid one was found in Field2
  if( ftx_qso_record.dx_call_received )
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_call_entry), ftx_qso_record.dx_call );
  else
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_call_entry), " - - - " );

  // DX locator is valid but not RR73
  if( ftx_qso_record.dx_loc_received )
  {
    // Display DX in xplanet if enabled
    if( ftx_rc.dx_xplanet && FTx_Open_Xplanet_Files() )
    {
      FTx_Prepare_Xplanet( ftx_qso_record.dx_loc, ftx_qso_record.dx_call );
      FTx_Close_Xplanet_Files();
      Launch_Xplanet( ftx_rc.xplanet_command );
    }
    else  // Calculate "Remote" station's position, done above if Xplanet enabled
      Gridloc_to_Position(
          ftx_qso_record.dx_loc, &ftx_rc.dx_latitude, &ftx_rc.dx_longitude );

    // Calculate Bearing and Distance of DX station
    double bearing, distance;
    Bearing_Distance(
        &bearing, &distance,
        ftx_rc.home_latitude,
        ftx_rc.home_longitude,
        ftx_rc.dx_latitude,
        ftx_rc.dx_longitude );

    // Display DX Grid, Bearing and Distance in DX Details display
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_grid_entry), ftx_qso_record.dx_loc );
    snprintf( text, FTX_TXT_SIZE, "%d", (int)bearing );
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_azim_entry), text );
    snprintf( text, FTX_TXT_SIZE, "%d", (int)distance );
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_dist_entry), text );

  } // if( loc_ok )
  else // Clear display
  {
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_grid_entry), "- - -" );
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_azim_entry), "- - -" );
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_dist_entry), "- - -" );
  }

  // Display SNR report in Own SNR label if valid value received in Field3
  if( ftx_qso_record.own_snr_received )
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.own_snr_entry), ftx_qso_record.own_snr );
  else
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.own_snr_entry), "- - -" );

  // Get an incoming "final" (73, RR73, R73 or RRR) and display in FINAL
  if( ftx_qso_record.dx_final_received )
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_final_entry), ftx_qso_record.dx_final );
  else
    gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_final_entry), "- - -" );

  // Display DX measured SNR in DX SNR label
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_snr_entry), ftx_qso_record.dx_snr );

  // Actual DX carrier (receiver) frequency
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_freq_entry), ftx_qso_record.frequency );

} // Ftx_QSO_Details()

//----------------------------------------------------------------------

/* Ftx_Mesg_Row_Activated()
 *
 * Handles the row_activated signal of the Messages treeview
 */
  void
Ftx_Mesg_Row_Activated( GtkTreeView *treeview )
{
  gchar
    call[FTX_CALL_SIZE],
    field[FTX_CALL_STR_SIZE],
    snr[FTX_SNR_SIZE],
    *value;

  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Saves a callsign for comparisons
  static char prev_dx_call[FTX_CALL_SIZE] = { '\0' };
  BOOLEAN new_dx_call = False;

  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  // Get the selected row in the Messages treeview
  selection = gtk_tree_view_get_selection( treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter );

  /*** Enter values of Selected Message row into the Selected Message display ***/
  // Get the Time value
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_TIME, &value, -1 );
  if( value != NULL )
  {
    gtk_label_set_text(GTK_LABEL(ftx_gui.time_label), value );
    // Make time ADIF format compatible
    memmove( &value[2], &value[3], 2 );
    Strlcpy( ftx_qso_record.time, value, FTX_TIME_SIZE );
    g_free( value );
  }

  // Get the Date in ADIF format
  time_t tim = time( NULL );
  strftime( ftx_qso_record.date, FTX_DATE_SIZE, "%Y%m%d", gmtime(&tim) );

  // Get the DX SNR value
  Strlcpy( ftx_qso_record.dx_snr, "- - -", FTX_SNR_SIZE );
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_SNR, &value, -1 );
  if( value != NULL )
  {
    Strlcpy( ftx_qso_record.dx_snr, value, FTX_SNR_SIZE );
    gtk_label_set_text(GTK_LABEL(ftx_gui.snr_label), value );
    g_free( value );
  }

  // Get the Dt timing error
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_DT, &value, -1 );
  if( value != NULL )
  {
    gtk_label_set_text(GTK_LABEL(ftx_gui.dt_label), value );
    g_free( value );
  }

  // Get the DX Frequency value
  Strlcpy( ftx_qso_record.frequency, "- - -", FTX_FREQ_SIZE );
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_FREQ, &value, -1 );
  if( value != NULL )
  {
    gtk_label_set_text(GTK_LABEL(ftx_gui.freq_label), value );
    uint32_t frq = TRx->rx_frequency + (uint32_t)( atoi(value) );
    snprintf( ftx_qso_record.frequency, FTX_FREQ_SIZE, "%u", frq );
    g_free( value );
  }

  // Enter the current mode
  if( ftx_rc.proto == FTX_PROTOCOL_FT8 )
    Strlcpy( ftx_qso_record.mode, "FT8", FTX_MODE_SIZE );
  else
    Strlcpy( ftx_qso_record.mode, "FT4", FTX_MODE_SIZE );

  // Get the Message Field1 value. It can be a CQ, a DX called or Own Call
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_COL1, &value, -1 );
  ftx_qso_record.own_call_received = False;
  ftx_qso_record.cq_call_received  = False;
  if( value != NULL )
  {
    gtk_label_set_markup(GTK_LABEL(ftx_gui.field1_label), value );
    Strlcpy( field, gtk_label_get_text(GTK_LABEL(ftx_gui.field1_label)), FTX_CALL_STR_SIZE );

    if( Is_CQ(field) )  // Check if Field1 is a CQ....
    {
      ftx_qso_record.cq_call_received = True;
    }
    else if( Get_Callsign(call, field) )  // Check if Field1 matches own call
    {
      if( strcmp(call, ftx_rc.own_call) == 0 )
        ftx_qso_record.own_call_received = True;
    }

    g_free( value );
  }

  // Get the Message Field2 value. It is normally the DX Call
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_COL2, &value, -1 );
  ftx_qso_record.dx_call_received = False;
  if( value != NULL )
  {
    gtk_label_set_markup(GTK_LABEL(ftx_gui.field2_label), value );
    Strlcpy( field, gtk_label_get_text(GTK_LABEL(ftx_gui.field2_label)), FTX_CALL_STR_SIZE );

    if( Get_Callsign(call, field) )
    {
      Strlcpy( ftx_qso_record.dx_call, call, FTX_CALL_SIZE );
      ftx_qso_record.dx_call_received = True;
    }
    else
    {
      ftx_qso_record.dx_loc_received   = False;
      ftx_qso_record.dx_final_received = False;
      ftx_qso_record.own_snr_received  = False;
      ftx_qso_record.dx_call[0]        = '\0';
    }

    g_free( value );
  }

  // Check if this is a new DX call
  if( ftx_qso_record.dx_call_received &&
      (strcmp(ftx_qso_record.dx_call, prev_dx_call) != 0) )
  {
    new_dx_call = True;
    Strlcpy( prev_dx_call, ftx_qso_record.dx_call, FTX_CALL_SIZE );
  }

  // Get the Message Field3 value. It can be Own SNR or DX Grid or DX Final
  gtk_tree_model_get( model, &iter, FTX_TREEVIEW_COL3, &value, -1 );
  if( new_dx_call )
  {
    ftx_qso_record.own_snr_received  = False;
    ftx_qso_record.dx_loc_received   = False;
    ftx_qso_record.dx_final_received = False;
  }
  if( value != NULL )
  {
    gtk_label_set_markup(GTK_LABEL(ftx_gui.field3_label), value );
    Strlcpy( field, gtk_label_get_text(GTK_LABEL(ftx_gui.field3_label)), FTX_CALL_STR_SIZE );

    // Check if it is an SNR report
    if( Get_SNR(snr, field) )
    {
      Strlcpy( ftx_qso_record.own_snr, snr, FTX_SNR_SIZE );
      ftx_qso_record.own_snr_received = True;
    }
    else // Check if it is a DX grid locator
      if( !ftx_qso_record.dx_loc_received && Is_Locator(field) )
      {
        Strlcpy( ftx_qso_record.dx_loc, field, FTX_LOC_SIZE );
        ftx_qso_record.dx_loc_received = True;
      }
      else // Check if it is a final from DX
        if( !ftx_qso_record.dx_final_received && Is_Final(field) )
        {
          Strlcpy( ftx_qso_record.dx_final, field, FTX_FINAL_SIZE );
          ftx_qso_record.dx_final_received = True;
        }

    g_free( value );
  }

  // Display the QSO/DX station details
  Ftx_QSO_Details();

  // Create Standard messages for the selected DX call
  FTx_Std_Messages();

} // Ftx_Mesg_Row_Activated()

//----------------------------------------------------------------------

/* FTx_Monitor_Toggled()
 *
 * Handles the on_ftx_monitor_toggled() callback
 */
  void
FTx_Monitor_Toggled( GtkToggleButton *togglebutton )
{
  // Abort if Receiver is not running
  if( !Transceiver[Indices.TRx_Index]->receive_active || Flag[GUEST_QUIT] )
  {
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
        MARKUP_RED, " Monitor: Hermes2 Rx Inactive " );
    return;
  }

  // Start FTX Receive / Decode functions
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    ftx_rc.decoder_exit = False;
    if( !Pthread_Create(
          &hermes2_rc.guest_rx_thread, NULL,
          FTx_Decoder_Run, (void *)&(ftx_rc.proto),
          _("Failed to create Receive thread")) )
      return;
    Set_Label_Markup( GTK_LABEL(ftx_gui.monitor_label), MARKUP_GREEN, " MONITOR " );

    // Indicate receiving of FTx siganls
    if( ftx_rc.proto == FTX_PROTOCOL_FT8 )
      Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
          MARKUP_GREEN, " Monitor: Receiving FT8 Signals " );
    else
      Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label),
          MARKUP_GREEN, " Monitor: Receiving FT4 Signals " );
  }
  else
  {
    ftx_rc.decoder_exit = True;
    Set_Label_Markup( GTK_LABEL(ftx_gui.monitor_label), MARKUP_GREY, " MONITOR " );
    Set_Label_Markup( GTK_LABEL(ftx_gui.ftx_log_label), MARKUP_GREY, " Log/Status Messages " );
  }
} // FTx_Monitor_Toggled()

//----------------------------------------------------------------------

// Clears the labels and flags related to the Received message
  void
FTx_Clear_Received( void )
{
  gtk_label_set_text( GTK_LABEL(ftx_gui.time_label),     "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.snr_label),      "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.dt_label),       "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.freq_label),     "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.field1_label),   "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.field2_label),   "" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.field3_label),   "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_call_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_snr_entry),   "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.own_snr_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_freq_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_grid_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_azim_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_dist_entry),  "" );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.dx_final_entry), "" );
  ftx_qso_record.own_call_received = False;
  ftx_qso_record.own_snr_received  = False;
  ftx_qso_record.dx_call_received  = False;
  ftx_qso_record.dx_loc_received   = False;
  ftx_qso_record.dx_final_received = False;
  ftx_qso_record.cq_call_received  = False;
  ftx_rc.messages_ready            = False;
  ftx_qso_record.dx_call[0]        = '\0';

} // FTx_Clear_Received()

//----------------------------------------------------------------------

  void
FTx_Clear_Messages( void )
{
  // Clears the Messages display (buffer, status and treeview)
  Clear_Mesg_Buffer();

  // Clear status Decoder labels
  gtk_label_set_text( GTK_LABEL(ftx_gui.decoded_label),   "0" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.hashtable_label), "0" );
  gtk_label_set_text( GTK_LABEL(ftx_gui.magn_label),      "0" );

  // Clear the Decoder treeview
  GtkTreeIter iter;
  gtk_tree_model_get_iter_first( GTK_TREE_MODEL(ftx_gui.mesg_store), &iter );
  for( uint8_t row = 0; row < FTX_TREEVIEW_ROWS; row++ )
  {
    for( uint8_t col = 0; col < FTX_TREEVIEW_COLS; col++ )
    {
      gtk_list_store_set( ftx_gui.mesg_store, &iter, col, "", -1 );
    }
    gtk_tree_model_iter_next( GTK_TREE_MODEL(ftx_gui.mesg_store), &iter );
  }

} // FTx_Clear_Messages()

//----------------------------------------------------------------------

/* FTx_Create_CQ()
 *
 * Creates a CQ call FTx message
 */
  void
FTx_Create_CQ( void )
{
  // Print selected CQ string and own call in cq string
  gchar cq[FTX_MESSAGE_SIZE];
  gchar *text = gtk_combo_box_text_get_active_text(
      GTK_COMBO_BOX_TEXT(ftx_gui.select_cq_combo) );
  snprintf( cq, FTX_MESSAGE_SIZE, "%s %s ", text, ftx_rc.own_call );

  // cat only first 4 chars of grid to cq and set to entry
  Strlcat( cq, ftx_rc.own_locator, 5 );
  gtk_entry_set_text( GTK_ENTRY(ftx_gui.message_entry[FTX_CQ_ENTRY]), cq );
  g_free( text );

  // Set a marked up text to "Send CQ" button label
  Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_GREEN, " Send CQ " );
  ftx_rc.send_cq_ready = True;

} // FTx_Create_CQ()

//----------------------------------------------------------------------

/* FTx_Find_Tx_Freq()
 *
 * Finds a suitable frequency slot for the FTx transmitter
 * by looking for the largest gap between received stations
 */
  void
FTx_Find_Tx_Freq( void )
{
#define COUNT_SPACES    3

  uint16_t
    prev_freq,
    save_prev_freq,
    freq_gap,
    max_gap,
    found_freq;

  uint8_t idx, pos;

  // Sorted list of DX frequencies from the decoder results
  uint16_t ftx_freq[FTX_TREEVIEW_ROWS + 2];

  // Abort if mesg_data not initialized
  if( mesg_data == NULL ) return;

  // First collect all DX station frequencies in the buffer
  for( idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )
  {
    uint8_t cnt = 0;
    for( pos = 0; pos < FTX_MESG_DATA_SIZE; pos++ )
    {
      if( mesg_data[idx][pos] == ' ' ) cnt++;
      if( cnt >= COUNT_SPACES ) break; // Frequency is after 3rd space
    }

    // Frequencies should be within the min and max limits
    uint16_t frq = (uint16_t)( atoi(&mesg_data[idx][pos]) );
    if( (frq >= F_MIN) && (frq <= F_MAX) )
      ftx_freq[idx+1] = frq;
    else
      ftx_freq[idx+1] = 0;

  } // for( idx = 0; idx < FTX_TREEVIEW_ROWS; idx++ )

  // Then bubble sort frequencies from idx=1 to idx=FTX_TREEVIEW_ROWS
  for( idx = 0; idx < FTX_TREEVIEW_ROWS - 1; idx++)
  {
    BOOLEAN flag = False;
    for( pos = 1; pos < FTX_TREEVIEW_ROWS - idx; pos++)
    {
      uint8_t pp1 = pos + 1;
      if( ftx_freq[pos] > ftx_freq[pp1] )
      {
        flag = True;
        uint16_t temp = ftx_freq[pp1];
        ftx_freq[pp1] = ftx_freq[pos];
        ftx_freq[pos] = temp;
      }
    }

    // No Swapping happened, array is sorted
    if( !flag ) break;
  } // for( idx = 0; idx < FTX_TREEVIEW_ROWS - 1; idx++)

  // These should be the range of frequencies
  ftx_freq[0] = F_MIN;
  ftx_freq[FTX_TREEVIEW_ROWS + 1] = F_MAX;

  // Find the largest gap between adjacent frequencies
  max_gap        = 0;
  save_prev_freq = 0;
  prev_freq      = ftx_freq[0];
  for( idx = 1; idx < FTX_TREEVIEW_ROWS + 2; idx++ )
  {
    uint16_t new_freq = ftx_freq[idx];
    if( !new_freq ) continue;
    freq_gap = new_freq - prev_freq;
    if( max_gap < freq_gap )
    {
      max_gap        = freq_gap;
      save_prev_freq = prev_freq;
    }
    prev_freq = new_freq;
  } // for( idx = 1; idx < FTX_TREEVIEW_ROWS + 1; idx++ )

  // Make the suggested frequency half way between the two adjacent ones
  found_freq = save_prev_freq + max_gap / 2;

  // Round it to nearest 50Hz, the bandwidth of FT8
  found_freq = ( (found_freq + 25) / 50 ) * 50;

  // Set frequency to TX frequency spinbutton
  ftx_rc.tx_frequency = found_freq;
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(ftx_gui.tx_frequency), ftx_rc.tx_frequency );

} // FTx_Find_Tx_Freq()

//----------------------------------------------------------------------
  
/* FTx_Enable_Tx()
 *
 * Handles the callback from the Enable Tx toggle button
 */
  void
FTx_Enable_Tx( GtkToggleButton *togglebutton )
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    // Enable Transmit
    ftx_rc.tx_enabled = True;
    Flag[HERMES2_SEND_DUC_PACKET] = True;
    Set_Label_Markup( GTK_LABEL(ftx_gui.halt_tx_label), MARKUP_ORANGE, " TX ENABLED " );
  }
  else
  {
    // Disable Transmit
    ftx_rc.tx_enabled = False;
    MOX_Control( MOX_OFF );
    Modulator = NULL;
    Flag[GUEST_TRANSMIT_MODE]     = False;
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Set_Label_Markup( GTK_LABEL(ftx_gui.halt_tx_label), MARKUP_GREY, " ENABLE TX " );
  }
} // FTx_Enable_Tx()

//----------------------------------------------------------------------

/* FTx_Send_Message()
 *
 * Handles the on_ftx_send_mesg_clicked callback
 */
  void
FTx_Send_Message( gpointer user_data )
{
  char text[4];

  // The active message index (0 to 5 + 6 for CQ)
  int8_t idx = ftx_rc.active_mesg_idx;

  // Get the gtk label widget from button
  GtkLabel *lbl = (GtkLabel *)user_data;

  // Get the label's text
  Strlcpy( text, gtk_label_get_text(lbl), 4 );

  // Get the Number in the button's label (1-6)
  int8_t active = (int8_t)( atoi(text) );

  // Create a label text with no markup
  snprintf( text, 4, " %d ", active );

  // If the message is not the active one
  if( idx + 1 != active )
  {
    // Create an orange markup with index number and set to label
    Set_Label_Markup( lbl, MARKUP_ORANGE, text );

    // Make it the active message
    ftx_rc.active_mesg_idx = active - 1;
  }
  else // If the message is already the active one, reset it
  {
     // Create a green markup with index number and set to label
    Set_Label_Markup( lbl, MARKUP_GREEN, text );
    ftx_rc.active_mesg_idx = FTX_NO_ACTIVE_MESG;
    return;
 }

  // If there is a pre-selected message
  if( idx >= 0 )
  {
    // Reset a pre-selected Send button, if there is one
    if( idx < NUM_SEND_LABELS ) // One of 6 Send Message labels
    {
      snprintf( text, 4, " %d ", idx + 1 );
      Set_Label_Markup( GTK_LABEL(ftx_gui.send_label[idx]), MARKUP_GREEN, text );
    }
    else // Send its the CQ label
      Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_GREEN, " Send CQ " );
  }

} // FTx_Send_Message()

//----------------------------------------------------------------------

/* FTx_Send_CQ()
 *
 * Handles the on_ftx_send_cq_clicked callback
 */
  void
FTx_Send_CQ( void )
{
  // The active message index (0 to 5 + 6 for CQ)
  int8_t idx = ftx_rc.active_mesg_idx;

  // If the message is not the active one
  if( idx != FTX_CQ_ENTRY )
  {
    // Create an orange markup with index number and set to label
    Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_ORANGE, " Send CQ " );

    // Make it the active message
    ftx_rc.active_mesg_idx = FTX_CQ_ENTRY;
  }
  else // If the message is already the active one, reset it
  {
     // Create a green markup with index number and set to label
    Set_Label_Markup( GTK_LABEL(ftx_gui.send_cq_label), MARKUP_GREEN, " Send CQ " );
    ftx_rc.active_mesg_idx = FTX_NO_ACTIVE_MESG;
    return;
 }

  // If there is a pre-selected message
  if( idx >= 0 )
  {
    // Reset a pre-selected Send button, if there is one
    if( idx < NUM_SEND_LABELS ) // One of 6 Send Message labels
    {
      char text[4];
      snprintf( text, 4, " %d ", idx + 1 );
      Set_Label_Markup( GTK_LABEL(ftx_gui.send_label[idx]), MARKUP_GREEN, text );
    }
  }

} // FTx_Send_CQ()

//----------------------------------------------------------------------

