/*
 *  Copyright (c) 2003-2010, Mark Borgerding. All rights reserved.
 *  This file is part of KISS FFT - https://github.com/mborgerding/kissfft
 *
 *  SPDX-License-Identifier: BSD-3-Clause
 *  See COPYING file for more information.
 */

/* kiss_fft.h
   defines kiss_fft_scalar as either short or a double type and defines
   typedef struct { kiss_fft_scalar r; kiss_fft_scalar i; } kiss_fft_cpx;
 */

#include "kiss_fft.h"
#include <complex.h>
#include <limits.h>

#define MAXFACTORS 32
/* e.g. an fft of length 128 has 4 factors as far as kissfft is concerned
   4 * 4 * 4 * 2
 */

struct kiss_fft_state
{
  int nfft;
  int inverse;
  int factors[2 * MAXFACTORS];
  kiss_fft_cpx twiddles[1];
};

/*
   Explanation of macros dealing with complex math:

   C_MUL(m, a, b)      : m = a*b
   C_FIXDIV( c, div )  : if a fixed point impl., c /= div. noop otherwise
   C_SUB( res, a, b)   : res = a - b
   C_SUBFROM( res, a)  : res -= a
   C_ADDTO( res, a)    : res += a
 */

#define II (__extension__ 1.0i) // My addition, double imaginary extension

#define S_MUL(a,b) ( (a)*(b) )

#define C_MUL(m,a,b) ( (m) = (a)*(b) )

#define C_MULBYSCALAR( c, s ) ( (c) *= (s) )

#define C_ADD( res, a, b ) ( (res) = (a) + (b) )

#define C_SUB( res, a, b ) ( (res) = (a) - (b) )

#define C_ADDTO( res, a ) ( (res) += (a) )

#define C_SUBFROM( res, a ) ( (res) -= (a) )

#define KISS_FFT_COS(phase) (kiss_fft_scalar) cos(phase)

#define KISS_FFT_SIN(phase) (kiss_fft_scalar) sin(phase)

#define HALF_OF(x) ( (x) / 2.0 )

#define kf_cexp(x, phase) ( (x) = KISS_FFT_COS(phase) + II * KISS_FFT_SIN(phase) )

#define  KISS_FFT_TMP_ALLOC(nbytes) KISS_FFT_MALLOC(nbytes)
#define  KISS_FFT_TMP_FREE(ptr) KISS_FFT_FREE(ptr)

