/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "utils.h"
#include "common.h"
#include "shared.h"
#include "../common/transceiver.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/process.h"
#include "../Hermes2/modulate.h"
#include "../Hermes2/sound.h"
#include "../time/interface.h"
#include <gtk/gtk.h>
#include <locale.h>
#include <math.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

#define ERROR_DIALOG_IDS \
  "hermes2_error_dialog", \
  NULL

//----------------------------------------------------------------------

/*  Read_Line()
 *
 *  Loads a line from a file, aborts on failure. Lines beginning
 *  with a '#' are ignored as comments. At the end of file EOF is
 *  returned. Lines assumed maximum 80 characters long.
 */
  int
Read_Line( char *buff, FILE *pfile, const char *mesg )
{
  uint16_t num_chr; // Number of characters read, excluding lf/cr
  int chr;          // Character read by getc()
  char error_mesg[MESG_STRING_SIZE];

  // Prepare error message
  snprintf( error_mesg, sizeof(error_mesg),
      _("Read_Line(): Error reading %s"), mesg );

  // Abort if file pointer closed
  if( pfile == NULL )
  {
    buff[0] = '\0';
    Error_Dialog( error_mesg, SHOW_OK );
    return( ERROR );
  }

  // Clear buffer at start
  buff[0] = '\0';
  num_chr = 0;

  // Get next character, return if chr = EOF
  chr = fgetc( pfile );
  if( chr == EOF )
  {
    Error_Dialog( error_mesg, SHOW_OK );
    Close_File( &pfile );
    return( ERROR );
  }

  /* Ignore commented lines and lines
   * starting with newline or carriage return */
  while( (chr == '#') || (chr == CR ) || (chr == LF ) )
  {
    // Go to the end of line (look for LF or CR)
    while( (chr != CR) && (chr != LF) )
    {
      // Get next character, return error if chr = EOF
      chr = fgetc( pfile );
      if( chr == EOF )
      {
        fprintf( stderr, "hermes2: %s\n", error_mesg );
        Error_Dialog( error_mesg, SHOW_OK );
        Close_File( &pfile );
        return( ERROR );
      }
    }

    // Dump any CR/LF remaining
    while( (chr == CR) || (chr == LF) )
    {
      // Get next character, return error if chr = EOF
      chr = fgetc( pfile );
      if( chr == EOF )
      {
        fprintf( stderr, "hermes2: %s\n", error_mesg );
        Error_Dialog( error_mesg, SHOW_OK );
        Close_File( &pfile );
        return( ERROR );
      }
    }

  } // End of while( (chr == '#') || ...

  /* Continue reading characters from file till
   * number of characters = 80 or EOF or CR/LF */
  while( num_chr < READ_LINE_BUF_SIZE )
  {
    // If LF/CR reached before filling buffer, return line
    if( (chr == LF) || (chr == CR) )
    {
      // Get next character
      chr = fgetc( pfile );
      if( chr == EOF )
      {
        // Terminate buffer as a string if chr = EOF
        buff[num_chr] = '\0';
        Error_Dialog( error_mesg, SHOW_OK );
        Close_File( &pfile );
        return( ERROR );
      }

      // Restore char in not EOF
      ungetc( chr, pfile );
      break;
    }

    // Enter new character to line buffer
    buff[num_chr++] = (char)chr;

    // Get next character
    chr = fgetc( pfile );
    if( chr == EOF )
    {
      // Terminate buffer as a string if chr = EOF
      buff[num_chr] = '\0';
      Error_Dialog( error_mesg, SHOW_OK );
      Close_File( &pfile );
      return( ERROR );
    }

    // Abort if end of line not reached at 80 char.
    if( (num_chr == READ_LINE_BUF_SIZE) &&
        (chr != LF) && (chr != CR) )
    {
      // Terminate buffer as a string
      buff[num_chr] = '\0';
      snprintf( error_mesg, sizeof(error_mesg),
          _("Error reading %s\n"
            "Line longer than %d characters"), mesg, READ_LINE_BUF_SIZE );
      fprintf( stderr, "hermes2: %s\n%s\n", error_mesg, buff );
      Error_Dialog( error_mesg, SHOW_OK );
      Close_File( &pfile );
      return( ERROR );
    }

  } // End of while( num_chr < max_chr )

  // Terminate buffer as a string
  buff[num_chr] = '\0';

  // Retrun EOF if [END] key is read
  if( strcmp(buff, "[END]") == 0 )
    return( EOF );

  return( SUCCESS );
} // End of Read_Line()

//----------------------------------------------------------------------

/* Strlcpy()
 *
 * Copies n-1 chars from src string into dest string. Unlike other
 * such library fuctions, this makes sure that the dest string is
 * null terminated by copying only n-1 chars to leave room for the
 * terminating char. n would normally be the sizeof(dest) string but
 * copying will not go beyond the terminating null of src string
 */
  void
Strlcpy( char *dest, const char *src, size_t n )
{
  char ch = src[0];
  uint16_t idx = 0;

  // Leave room for terminating null in dest
  n--;

  // Copy till terminating null of src or to n-1
  while( (ch != '\0') && (n > 0) )
  {
    dest[idx] = src[idx];
    idx++;
    ch = src[idx];
    n--;
  }

  // Terminate dest string
  dest[idx] = '\0';

} // Strlcpy()

//----------------------------------------------------------------------

/* Strlcat()
 *
 * Concatenates at most n-1 chars from src string into dest string.
 * Unlike other such library fuctions, this makes sure that the dest
 * string is null terminated by copying only n-1 chars to leave room
 * for the terminating char. n would normally be the sizeof(dest)
 * string but copying will not go beyond the terminating null of src
 */
  void
Strlcat( char *dest, const char *src, size_t n )
{
  char ch = dest[0];
  uint16_t idd = 0; // dest index
  uint16_t ids = 0; // src  index

  // Find terminating null of dest
  while( (ch != '\0') )
  {
    idd++;
    ch = dest[idd];
  }

  // Copy n-1 chars to leave room for terminating null
  n--;
  ch = src[ids];
  while( (n > 0) && (ch != '\0') )
  {
    dest[idd] = src[ids];
    ids++;
    ch = src[ids];
    idd++;
    n--;
  }

  // Terminate dest string
  dest[idd] = '\0';

} // Strlcat()

//------------------------------------------------------------------

/* Strtod()
 *
 * Replaces strtod() to take into account the
 * locale-dependent decimal point character
 */
double Strtod( const char *nptr )
{
  double d = 0.0;
  char *s = NULL;
  static BOOLEAN first_call = True;
  static char dp = '.';


  // Find locale-dependent decimal point character
  if( first_call )
  {
    struct lconv *lcnv;
    lcnv = localeconv();
    dp = *lcnv->decimal_point;
    first_call = False;
  }

  /* Look for a . or , decimal point character
   * in the supplied number buffer (string) */
  uint8_t len = (uint8_t)strlen( nptr );
  uint8_t idx;
  for( idx = 0; idx < len; idx++ )
    if( (nptr[idx] == ',') || (nptr[idx] == '.') )
      break;

  // If a decimal point character is found, replace
  Mem_Alloc( (void **) &s, len + 1 );
  Strlcpy( s, nptr, len + 1 );
  if( idx < len ) s[idx] = dp;
  d = strtod( s, NULL );
  Mem_Free( (void **) &s );

  return( d );
} // End of Strtod()

//------------------------------------------------------------------

// ***  Memory allocation/freeing utils ***
  void
Mem_Alloc( void **ptr, size_t req )
{
  Mem_Free( ptr );
  *ptr = calloc( 1, req );
  if( *ptr == NULL )
  {
    perror( _("hermes2: A memory allocation request failed") );
    exit( -1 );
  }

  memset( *ptr, 0, req );
} // End of void Mem_Alloc()

//----------------------------------------------------------------------

  void
Mem_Realloc( void **ptr, size_t req )
{
  *ptr = realloc( *ptr, req );
  if( *ptr == NULL )
  {
    perror( _("hermes2: A memory re-allocation request failed") );
    exit( -1 );
  }
} // End of void Mem_Realloc()

//----------------------------------------------------------------------

  void
Mem_Free( void **ptr )
{
  if( *ptr != NULL ) free( *ptr );
  *ptr = NULL;

} // End of void Mem_Free()

//----------------------------------------------------------------------

/* Open_File()
 *
 * Opens a file, aborts on error
 */
  uint8_t
Open_File( FILE **fp, const char *fname, const char *mode )
{
  if( *fp != NULL )
  {
    fprintf( stderr, "Open_File(): cannot open file %s: file pointer != NULL\n", fname );
    return( FAILURE );
  }

  *fp = fopen( fname, mode );
  if( *fp == NULL )
  {
    fprintf( stderr, "Open_File(): cannot open file %s\n", fname );
    perror( fname );
    return( FAILURE );
  }

  return( SUCCESS );
} // End of Open_File()

//----------------------------------------------------------------------

/*  Close_File()
 *
 *  Closes a file pointer
 */
  void
Close_File( FILE **fp )
{
  if( *fp != NULL )
  {
    sync();
    fclose( *fp );
    *fp = NULL;
  }

} // Close_File()

//------------------------------------------------------------------------

/* Fprintf_Error()
 *
 * Handles error warnings on fprintf() failure
 */
  void
Fprintf_Error( FILE *fp, const char *filename )
{
  char err_str[MESG_STRING_SIZE]; // Error messages string

  size_t s = sizeof( err_str );
  snprintf( err_str, s, "hermes2: %s", filename );
  err_str[s-1] = '\0';
  perror( err_str );
  snprintf( err_str, s,
      _("Cannot write to configuration file:\n%s\n"), filename );
  Error_Dialog( err_str, HIDE_OK );
  Close_File( &fp );

} // Fprintf_Error()

//------------------------------------------------------------------------

/* Colorize()
 *
 * Pseudo-colorizes FFT Spectrum Display pixels
 * according to the IFFT's output bin values
 */
  void
Colorize( guchar *pix, uint8_t pixel_val )
{
  // Default pseudo-color palette
  static const uint8_t
    R[256] =
    {
       64,  62,  60,  58,  56,  54,  52,  50,  48,  46,  44,  42,  40,  38,  37,  36,
       35,  34,  33,  32,  31,  30,  29,  28,  27,  26,  25,  24,  23,  22,  21,  10,
       19,  18,  17,  16,  15,  14,  13,  12,  11,  10,   9,   8,   7,   6,   5,   4,
        3,   2,   1,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   5,  10,  15,  20,  25,  30,  35,
       40,  45,  50,  55,  60,  65,  70,  75,  80,  85,  90,  95, 100, 105, 110, 115,
      120, 125, 130, 135, 140, 145, 150, 155, 160, 165, 170, 175, 180, 185, 190, 195,
      200, 205, 210, 215, 220, 225, 230, 235, 240, 245, 250, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255
    };

  static const uint8_t
    G[256] =
    {
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   5,  10,  15,  20,  25,  30,  35,  40,  45,  50,  55,  60,  65,
       70,  75,  80,  85,  90,  95, 100, 105, 110, 115, 120, 125, 130, 135, 140, 145,
      150, 155, 160, 165, 170, 175, 180, 185, 190, 195, 200, 205, 210, 215, 220, 225,
      230, 235, 240, 245, 250, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 250, 245, 240,
      235, 230, 225, 220, 215, 210, 205, 200, 195, 190, 185, 180, 175, 170, 165, 160,
      155, 150, 145, 140, 135, 130, 125, 120, 115, 110, 105, 100,  95,  90,  85,  80,
       75,  70,  65,  60,  55,  50,  45,  40,  35,  30,  25,  20,  15,  10,   5,   0
    };

  static const uint8_t
    B[256] =
    {
       64,  68,  72,  76,  82,  86,  90,  94,  98, 102, 106, 110, 114, 118, 122, 126,
      132, 134, 138, 142, 148, 152, 156, 160, 164, 168, 172, 176, 180, 184, 188, 192,
      196, 200, 204, 208, 212, 216, 220, 224, 227, 230, 233, 236, 239, 242, 245, 248,
      250, 252, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
      255, 255, 255, 255, 255, 255, 255, 250, 245, 240, 235, 230, 225, 220, 215, 210,
      205, 200, 195, 190, 185, 180, 175, 170, 165, 160, 155, 150, 145, 140, 135, 130,
      125, 120, 115, 110, 105, 100,  95,  90,  85,  80,  75,  70,  65,  60,  55,  50,
       45,  40,  35,  30,  25,  20,  15,  10,   5,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
        0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0
    };

  // Enter the palette triplet to the pixmap
  pix[0] = R[pixel_val];
  pix[1] = G[pixel_val];
  pix[2] = B[pixel_val];

} // Colorize()

//----------------------------------------------------------------------

/* Gtk_Builder()
 *
 * Returns a GtkBuilder with required objects from file
 */
  void
Gtk_Builder( GtkBuilder **builder, gchar *file, gchar **object_ids )
{
  GError *gerror = NULL;
  guint ret = 0;

  // Create a builder from object ids
  *builder = gtk_builder_new();
  ret = gtk_builder_add_objects_from_file( *builder, file, object_ids, &gerror );
  if( !ret )
  {
    fprintf( stderr, _("hermes2: Failed to add objects to builder:\n%s\n"),
        gerror->message );
    exit( -1 );
  }

  // Connect signals if gmodule is supported
  if( !g_module_supported() )
  {
    fprintf( stderr, _("hermes2: libgmodule not supported\n") );
    exit( -1 );
  }
  gtk_builder_connect_signals( *builder, NULL );

} // Gtk_Builder()

//------------------------------------------------------------------

/* Builder_Get_Object()
 *
 * Gets a named object from the GtkBuilder builder object
 */
  GtkWidget *
Builder_Get_Object( GtkBuilder *builder, gchar *name )
{
  GObject *object = gtk_builder_get_object( builder, name );
  if( object == NULL )
  {
    fprintf( stderr,
        _("!! hermes2: builder failed to get named object: %s\n"), name );
    exit( -1 );
  }

  return( GTK_WIDGET(object) );
} // Builder_Get_Object()

//------------------------------------------------------------------

typedef struct _ERR_DIALOG
{
  gchar *mesg;
  gboolean hide_ok;
} err_dialog_t;

/* _Error_Dialog()
 *
 * Opens an error dialog box and displays a message
 */
  static gboolean
_Error_Dialog( gpointer data )
{
  if( !hermes2_gui.error_dialog )
  {
    GtkBuilder *builder;
    gchar *object_ids[] = { ERROR_DIALOG_IDS };

    // Do not stop Hermes2 if in guest mode
    if( !Flag[GUEST_TRANSMITTING] && !Flag[GUEST_RECEIVING] )
    {
      // Disable time stations receiving on error
      if( time_window )
      {
        gtk_button_clicked(
            GTK_BUTTON( Builder_Get_Object(time_window_builder, "time_quit_button")) );
      }

      // Disable receiving and spectrum display on error
      for( uint8_t idx = 0; idx < Indices.Num_of_TRXs; idx++ )
      {
        gtk_toggle_button_set_active(
            GTK_TOGGLE_BUTTON(Transceiver[idx]->startrx_togglebutton), FALSE );
        gtk_toggle_button_set_active(
            GTK_TOGGLE_BUTTON(Transceiver[idx]->spectrum_off_radiobutton), TRUE );
      }
    } // if( !Flag[GUEST_TRANSMITTING] && !Flag[GUEST_RECEIVING] )

    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.error_dialog = Builder_Get_Object( builder, "hermes2_error_dialog" );
    GtkLabel *label = GTK_LABEL( Builder_Get_Object( builder, "hermes2_error_label") );
    gtk_label_set_text( label, ((err_dialog_t *)data)->mesg );

    // For non-recoverable errors, stop hermes2 and force quit
    if( ((err_dialog_t *)data)->hide_ok )
    {
      Hermes2_Stop();
      GtkWidget *button = Builder_Get_Object( builder, "hermes2_error_ok_button" );
      gtk_widget_hide( button );
    }

    gtk_widget_show( hermes2_gui.error_dialog );
    g_object_unref( builder );
  }

  return( FALSE );
} // _Error_Dialog()

/*  Error_Dialog()
 *
 *  Opens an error dialog box and displays a message
 */
  void
Error_Dialog( gchar *mesg, gboolean hide_ok )
{
  static err_dialog_t err_dialog;
  err_dialog.mesg    = mesg;
  err_dialog.hide_ok = hide_ok;
  g_idle_add( _Error_Dialog, (gpointer)&err_dialog );
  while( g_main_context_iteration(NULL, FALSE) );

} // Error_Dialog()

//----------------------------------------------------------------------

/* Init_Semaphore()
 *
 * Allocates or frees a semaphore object
 */
  BOOLEAN
Init_Semaphore( sem_t *semaphore, BOOLEAN init )
{
  // Initialize semaphore
  if( init )
  {
    if( sem_init(semaphore, 0, 0) != 0 )
    {
      Error_Dialog(
          _("Failed to initialize a semaphore"), HIDE_OK );
      perror( _("!!hermes2: initialize semaphore") );
      return( False );
    }
  }
  else  // Free semaphore
  {
    // Free the semaphore FIXME may not be needed
    //int sval;
    //sem_getvalue( semaphore, &sval );
    //while( sval-- ) sem_post( semaphore );

    // Destroy digimode semaphore
    sem_destroy( semaphore );
  }

  return( True );
} // Init_Semaphore()

//----------------------------------------------------------------------

// Struct to save data needed to set text to a GtkLabel
typedef struct _LABEL_TEXT
{
  GtkLabel *label;
  gchar    *text;
} label_text_t;

// Circular buffer to save label data
#define LABEL_DATA_SIZE   8
static label_text_t label_data[LABEL_DATA_SIZE];

// Locks the thread that invokes the idle callback
static sem_t label_sem;

// Unlocks the thread that invokes the idle callback
  static void
Unlock_Label( gpointer data )
{
  int sval;
  sem_getvalue( &label_sem, &sval );
  if( !sval ) sem_post( &label_sem );
}

// Idle add function to set text to a GtkLabel
  static gboolean
_Set_Label_Text( gpointer data )
{
  if( GTK_IS_LABEL(((label_text_t *)data)->label) )
    gtk_label_set_text(
        ( (label_text_t *)data )->label,
        ( (label_text_t *)data )->text );

  return( FALSE );
} // _Set_Label_Text()

// Ring index to incoming and local label data
static uint8_t label_data_idx = 0, label_ring_idx = 0;

// Thread function to initiate idle add function
  static void *
Set_Label_Thread( void *data )
{
  // Keep calling idle add function till ring buffer processed
  while( !Flag[HERMES2_QUIT] )
  {
    static uint8_t idx = 0;

    // Set up lock and wait for idle add to complete
    g_idle_add_full(
        G_PRIORITY_HIGH_IDLE,
        _Set_Label_Text,
        (gpointer)&label_data[idx],
        Unlock_Label );

    // We need to wait for the g_idle_add_full() function to complete
    while( g_main_context_iteration(NULL, FALSE) );

    // Increment and reset ring index
    sem_wait( &label_sem );
    label_ring_idx++;
    if( label_ring_idx >= LABEL_DATA_SIZE ) label_ring_idx = 0;
    idx = label_ring_idx;

    // Wait for new data to be entered in ring buffer
    if( label_ring_idx == label_data_idx ) sem_wait( &label_sem );
  } // while( !Flag[HERMES2_QUIT] )

  // Free semaphore and exit thread
  Init_Semaphore( &label_sem, False );
  return( NULL );
} // Set_Label_Thread()

/* Set_Label_Text()
 *
 * Sets the text in a GtkLabel using g_idle_add() function above
 */
  void
Set_Label_Text( GtkLabel *label, const gchar *text )
{
  // Flag to indicate thread exited
  static BOOLEAN label_thread = False;

  // Load incoming data to the circular buffer
  label_data[label_data_idx].label = label;
  Mem_Realloc( (void **)&(label_data[label_data_idx].text), strlen(text) + 1 );
  Strlcpy( label_data[label_data_idx].text, text, strlen(text) + 1 );
  label_data_idx++;
  if( label_data_idx >= LABEL_DATA_SIZE ) label_data_idx = 0;

  // Create thread if not already
  if( !label_thread )
  {
    // Create semaphore to lock idle add thread
    label_thread = True;
    Init_Semaphore( &label_sem, True );

    // Create a PTHREAD_CREATE_DETACHED type attribute
    pthread_attr_t attr;
    pthread_attr_init( &attr );
    pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );

    // Start the idle add thread
    pthread_t thrd;
    if( !Pthread_Create(
          &thrd, &attr, Set_Label_Thread, NULL,
          "_Failed to create Set_label_Thread() thread") )
    {
      label_thread = False;
      return;
    }

    pthread_attr_destroy( &attr );
  }
  else if( label_data_idx != label_ring_idx )
  {
    // Activate the thread function above if new data
    Unlock_Label( NULL );
  }

} // Set_Label_Text()

//----------------------------------------------------------------------

// Struct to save data needed to set text to a GtkEntry
typedef struct _ENTRY_TEXT
{
  GtkEntry *entry;
  gchar    *text;
} entry_text_t;

// Circular buffer to save entry data
#define ENTRY_DATA_SIZE   8
static entry_text_t entry_data[ENTRY_DATA_SIZE];

// Locks the thread that invokes the idle callback
static sem_t entry_sem;

// Unlocks the thread that invokes the idle callback
  static void
Unlock_Entry( gpointer data )
{
  int sval;
  sem_getvalue( &entry_sem, &sval );
  if( !sval ) sem_post( &entry_sem );
}

// Idle add function to set text to a GtkEntry
  static gboolean
_Set_Entry_Text( gpointer data )
{
  if( GTK_IS_ENTRY(((entry_text_t *)data)->entry) )
    gtk_entry_set_text(
        ( (entry_text_t *)data )->entry,
        ( (entry_text_t *)data )->text );

  return( FALSE );
} // _Set_Entry_Text()

// Ring index to incoming entry data
static uint8_t entry_data_idx = 0;

// Ring index to outgoing data
static uint8_t entry_ring_idx = 0;

// Thread function to initiate idle add function
  static void *
Set_Entry_Thread( void *data )
{
  // Keep calling idle add function till ring buffer processed
  while( !Flag[HERMES2_QUIT] )
  {
    static uint8_t idx = 0;

    // Set up lock and wait for idle add to complete
    g_idle_add_full(
        G_PRIORITY_HIGH_IDLE,
        _Set_Entry_Text,
        (gpointer)&entry_data[idx],
        Unlock_Entry );

    // We need to wait for the g_idle_add_full() function to complete
    while( g_main_context_iteration(NULL, FALSE) );

    // Increment and reset ring index
    sem_wait( &entry_sem );
    entry_ring_idx++;
    if( entry_ring_idx >= ENTRY_DATA_SIZE ) entry_ring_idx = 0;
    idx = entry_ring_idx;

    // Wait for new data to be entered in ring buffer
    if( entry_ring_idx == entry_data_idx ) sem_wait( &entry_sem );

  } // while( !Flag[HERMES2_QUIT] )

  // Free semaphore and exit thread
  Init_Semaphore( &entry_sem, False );
  return( NULL );
} // Set_Entry_Thread()

/* Set_Entry_Text()
 *
 * Sets the text in a GtkEntry using g_idle_add() function above
 */
  void
Set_Entry_Text( GtkEntry *entry, const gchar *text )
{
  // Flag to indicate thread exited
  static BOOLEAN entry_thread = False;

  // Load incoming data to the circular buffer
  entry_data[entry_data_idx].entry = entry;
  Mem_Realloc( (void **)&(entry_data[entry_data_idx].text), strlen(text) + 1 );
  Strlcpy( entry_data[entry_data_idx].text, text, strlen(text) + 1 );
  entry_data_idx++;
  if( entry_data_idx >= ENTRY_DATA_SIZE )
    entry_data_idx = 0;

  // Create thread if not already
  if( !entry_thread )
  {
    // Create semaphore to lock idle add thread
    entry_thread = True;
    Init_Semaphore( &entry_sem, True );

    // Create a PTHREAD_CREATE_DETACHED type attribute
    pthread_attr_t attr;
    pthread_attr_init( &attr );
    pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );

    // Start the idle add thread
    pthread_t thrd;
    if( !Pthread_Create(
          &thrd, &attr, Set_Entry_Thread, NULL,
          "_Failed to create Set_Entry_Thread() thread") )
    {
      entry_thread = False;
      return;
    }

    pthread_attr_destroy( &attr );
  }
  else if( entry_data_idx != entry_ring_idx )
  {
    // Activate the thread function above if new data
    Unlock_Entry( NULL );
  }

} // Set_Entry_Text()

//----------------------------------------------------------------------

// Struct to save data needed to set an icon to a GtkImage
typedef struct _ICON_DATA
{
  GtkImage   *image;
  gchar      *name;
  GtkIconSize size;
} icon_data_t;

// Circular buffer to save icon data
#define ICON_DATA_SIZE   8
static icon_data_t icon_data[ICON_DATA_SIZE];

// Locks the thread that invokes the idle callback
static sem_t icon_sem;

// Unlocks the thread that invokes the idle callback
  static void
Unlock_Icon( gpointer data )
{
  int sval;
  sem_getvalue( &icon_sem, &sval );
  if( !sval ) sem_post( &icon_sem );
}

// Idle add function to set an icon to a GtkImage
  static gboolean
_Set_Icon( gpointer data )
{
  if( GTK_IS_IMAGE(((icon_data_t *)data)->image) )
    gtk_image_set_from_icon_name(
        ( (icon_data_t *)data )->image,
        ( (icon_data_t *)data )->name,
        ( (icon_data_t *)data )->size );

  return( FALSE );
} // _Set_Icon()

// Ring index to incoming icon data
static uint8_t icon_data_idx = 0;

// Ring index to outgoing data
static uint8_t icon_ring_idx = 0;

// Thread function to initiate idle add function
  static void *
Set_Icon_Thread( void *data )
{
  // Keep calling idle add function till ring buffer processed
  while( !Flag[HERMES2_QUIT] )
  {
    static uint8_t idx = 0;

    // Set up lock and wait for idle add to complete
    g_idle_add_full(
        G_PRIORITY_HIGH_IDLE,
        _Set_Icon,
        (gpointer)&icon_data[idx],
        Unlock_Icon );

    // We need to wait for the g_idle_add_full() function to complete
    while( g_main_context_iteration(NULL, FALSE) );

    // Increment and reset ring index
    sem_wait( &icon_sem );
    icon_ring_idx++;
    if( icon_ring_idx >= ICON_DATA_SIZE ) icon_ring_idx = 0;
    idx = icon_ring_idx;

    // Wait for new data to be entered in ring buffer
    if( icon_ring_idx == icon_data_idx ) sem_wait( &icon_sem );

  } // while( !Flag[HERMES2_QUIT] )

  // Free semaphore and exit thread
  Init_Semaphore( &icon_sem, False );
  return( NULL );
} // Set_Icon_Thread()

/* Set_Icon()
 *
 * Sets an icon to a Gtkimage using g_idle_add() function above
 */
  void
Set_Icon( GtkImage *image, const gchar *name, GtkIconSize size )
{
  // Flag to indicate thread exited
  static BOOLEAN icon_thread = False;

  // Load incoming data to the circular buffer
  icon_data[icon_data_idx].image = image;
  Mem_Realloc( (void **)&(icon_data[icon_data_idx].name), strlen(name) + 1 );
  Strlcpy( icon_data[icon_data_idx].name, name, strlen(name) + 1 );
  icon_data[icon_data_idx].size = size;
  icon_data_idx++;
  if( icon_data_idx >= ICON_DATA_SIZE ) icon_data_idx = 0;

  // Create thread if not already
  if( !icon_thread )
  {
    // Create semaphore to lock idle add thread
    icon_thread = True;
    Init_Semaphore( &icon_sem, True );

    // Create a PTHREAD_CREATE_DETACHED type attribute
    pthread_attr_t attr;
    pthread_attr_init( &attr );
    pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );

    // Start the idle add thread
    pthread_t thrd;
    if( !Pthread_Create(
          &thrd, &attr, Set_Icon_Thread, NULL,
          "_Failed to create Set_Icon_Thread() thread") )
    {
      icon_thread = False;
      return;
    }

    pthread_attr_destroy( &attr );
  }
  else if( icon_data_idx != icon_ring_idx )
  {
    // Activate the thread function above if new data
    Unlock_Icon( NULL );
  }

} // Set_Icon()

//----------------------------------------------------------------------

// Struct to save data needed to set state of a toggle button
typedef struct _TOGGLE_BTN
{
  GtkToggleButton *button;
  gboolean flag;
} toggle_btn_t;

// Circular buffer to save button data
#define BUTTON_DATA_SIZE   4
static toggle_btn_t button_data[BUTTON_DATA_SIZE];

// Locks the thread that invokes the idle callback
static sem_t button_sem;

/* Unlocks the Set_Button_Thread() thread function
 * function when the idle callback has completed
 */
  static void
Unlock_Button( gpointer data )
{
  int sval;
  sem_getvalue( &button_sem, &sval );
  if( !sval ) sem_post( &button_sem );
}

// Idle add function to set the state of toggle button
  static gboolean
_Set_Toggle_Btn( gpointer data )
{
  gtk_toggle_button_set_active(
      ( (toggle_btn_t *)data )->button,
      ( (toggle_btn_t *)data )->flag );

  return( FALSE );
} // _Set_Toggle_Btn()

// Ring index to incoming icon data
static uint8_t button_data_idx = 0;

// Ring index to outgoing data
static uint8_t button_ring_idx = 0;

// Thread function to initiate idle add function
  static void *
Set_Button_Thread( void *data )
{
  // Keep calling idle add function till ring buffer processed
  while( !Flag[HERMES2_QUIT] )
  {
    static uint8_t idx = 0;

    // Set up lock and wait for idle add to complete
    g_idle_add_full(
        G_PRIORITY_HIGH_IDLE,
        _Set_Toggle_Btn,
        (gpointer)&button_data[idx],
        Unlock_Button );

    // We need to wait for the g_idle_add_full() function to complete
    while( g_main_context_iteration(NULL, FALSE) );

    // Increment and reset ring index
    sem_wait( &button_sem );
    button_ring_idx++;
    if( button_ring_idx >= BUTTON_DATA_SIZE ) button_ring_idx = 0;
    idx = button_ring_idx;

    // Exit if all of ring buffer handled
    if( button_ring_idx == button_data_idx ) sem_wait( &button_sem );
  } // while( !Flag[HERMES2_QUIT] )

  // Free semaphore and exit thread
  Init_Semaphore( &button_sem, False );
  return( NULL );
} // Set_Button_Thread()

/* Set_Toggle_Btn()
 *
 * Sets the state of a toggle button using g_idle_add() function above
 */
  void
Set_Toggle_Btn( GtkToggleButton *button, gboolean flag )
{
  // Flag to indicate thread exited
  static BOOLEAN button_thread = False;

  // Load incoming data to the circular buffer
  button_data[button_data_idx].button = button;
  button_data[button_data_idx].flag   = flag;
  button_data_idx++;
  if( button_data_idx >= BUTTON_DATA_SIZE )
    button_data_idx = 0;

  // Create thread if not already
  if( !button_thread )
  {
    // Create semaphore to lock idle add thread
    button_thread = True;
    Init_Semaphore( &button_sem, True );

    // Create a PTHREAD_CREATE_DETACHED type attribute
    pthread_attr_t attr;
    pthread_attr_init( &attr );
    pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_DETACHED );

    // Start the idle add thread
    pthread_t thrd;
    if( !Pthread_Create(
          &thrd, &attr, Set_Button_Thread, NULL,
          "_Failed to create Set_Button_Thread() thread") )
    {
      button_thread = False;
      return;
    }

    pthread_attr_destroy( &attr );
  }
  else if( button_data_idx != button_ring_idx )
  {
    // Activate the thread function above if new data
    Unlock_Button( NULL );
  }

} // Set_Toggle_Btn()

//----------------------------------------------------------------------

/* Pthread_Create()
 *
 * Creates a pthread to run start_routine()
 */
  BOOLEAN
Pthread_Create(
    pthread_t *thread,
    const pthread_attr_t *attr,
    void *(*start_routine)(void *),
    void *arg, gchar *err_msg )
{
  int ret = pthread_create( thread, attr, start_routine, arg );
  if( ret != SUCCESS )
  {
    perror( err_msg );
    Error_Dialog( err_msg, HIDE_OK );
    return( False );
  }

  return( True );
} // Pthread_Create()

//----------------------------------------------------------------------

// Data of GtkTextBuffer to get its offset iter
typedef struct _TEXT_BUF
{
  GtkTextBuffer *buffer;
  GtkTextIter *iter;
  gint offset;
} text_buf_t;

// Idle add function to get GtkTextBuffer offset iter
  static gboolean
_Get_Iter_at_Offset( gpointer data )
{
  gtk_text_buffer_get_iter_at_offset(
      ((text_buf_t *)data)->buffer,
      ((text_buf_t *)data)->iter,
      ((text_buf_t *)data)->offset );

  return( FALSE );
} // _Get_Iter_at_Offset()

/* Get_Iter_at_Offset()
 *
 * Gets the iteration of a text buffer by idle callback
 */
  void
Get_Iter_at_Offset( GtkTextBuffer *text_buffer, GtkTextIter *iter )
{
  static text_buf_t data;

  gint offset = gtk_text_buffer_get_char_count( text_buffer );
  data.buffer = text_buffer;
  data.iter   = iter;
  data.offset = offset;

  g_idle_add( _Get_Iter_at_Offset, (gpointer)&data );

} // Get_Iter_at_Offset()

//----------------------------------------------------------------------

// Functions to queue drawingarea draw by idle callback
  gboolean
static _Queue_Draw( gpointer data )
{
  gtk_widget_queue_draw( (GtkWidget *)data );
  return( FALSE );
}

/* Queue_Draw()
 *
 * Queues the drawing of a widget (usually
 * a drawingarea) by using an idle callback
 */
  void
Queue_Draw( GtkWidget *darea )
{
  g_idle_add( _Queue_Draw, (gpointer)darea );
  while( g_main_context_iteration(NULL, FALSE) );
}

//----------------------------------------------------------------------

/* Flush_Tx_Buffer()
 *
 * Flushes the SDR's I/Q transmit buffers
 * to avoid propping up the Receiver's AGC
 */
  BOOLEAN
Flush_Tx_Buffer( uint32_t buf_len )
{
  // Flush Transmit buffers
  Modulator = DUC_Buffer_Transmit;
  xmit_buffer.xmit_buf_len = (uint32_t)buf_len;
  for( uint32_t idx = 0; idx < xmit_buffer.xmit_buf_len; idx++ )
  {
    xmit_buffer.xmit_buf_i[idx] = 0;
    xmit_buffer.xmit_buf_q[idx] = 0;
  }
  sem_wait( &duc_send_semaphore );

  // Unlikely Transmit failure
  if( !xmit_buffer.status )
  {
    MOX_Control( MOX_OFF );
    return( False );
  }

  return( True );
} // Flush_Tx_Buffer()

//----------------------------------------------------------------------

// Sets a marked up string to a gtk label
  void
Set_Label_Markup( GtkLabel *label, uint8_t mkup, const char *text )
{
  gchar markup[HERMES2_MARKUP_SIZE];

  switch( mkup )
  {
    case MARKUP_TEXT:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_TEXT, text );
      break;
    case MARKUP_GREEN:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_GREEN, text );
      break;
    case MARKUP_RED:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_RED, text );
      break;
    case MARKUP_ORANGE:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_ORANGE, text );
      break;
    case MARKUP_CYAN:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_CYAN, text );
      break;
    case MARKUP_GREY:
      snprintf( markup, HERMES2_MARKUP_SIZE, MKUP_GREY, text );
      break;
  }

  gtk_label_set_markup( label, markup );
} // Set_Label_Markup()

//------------------------------------------------------------------

