/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_TRANSCEIVER_H
#define COMMON_TRANSCEIVER_H   1

#include "common.h"
#include <gtk/gtk.h>
#include <semaphore.h>
#include <pthread.h>
#include <stdint.h>

//----------------------------------------------------------------------

// Transceiver operational data
#define NUM_DATA_BUFFERS   3

// Maximum number of DDC Receivers to support
#define MAX_RECEIVERS    4

//----------------------------------------------------------------------

// Filter data struct
typedef struct _FILTER_DATA
{
  BOOLEAN init_filter;

  double
    cutoff,  // Cutoff frequency as fraction of sample rate
    ripple;  // Passband ripple as a percentage

  uint32_t
    npoles,  // Number of poles, _must_ be even
    type;    // Filter type, as below this struct

  // a and b Coefficients of the filter
  double *a, *b;

  // Saved input and output values
  double *x, *y;

  // Ring buffer index for above
  uint32_t ring_idx;

  // Input samples buffer and its length
  double  *samples_buffer;
  uint32_t samples_buffer_len;

} filter_data_t;

//----------------------------------------------------------------------

// Filter type for above struct
enum FILTER_TYPE
{
  FILTER_LOWPASS = 0,
  FILTER_HIGHPASS,
  FILTER_BANDPASS
};

//----------------------------------------------------------------------

// Data used to draw the FFT Spectrum
typedef struct _SPECTRUM_DATA
{
  // Parent Transceiver object index
  uint8_t trx_index;

  // FFT Spectrum Monitors window
  GtkWidget *spectrum_drawingarea;

  // Waterfall window pixbuf
  GdkPixbuf *waterfall_pixbuf;
  guchar    *waterfall_pixels;

  double
    fft_scale,   // Scale factor to keep FFT values reasonable
    *fft_in_i,   // FFT input I and Q buffers
    *fft_in_q;

  // FFT output bin values
  int *fft_bin_values;

  uint16_t
    fft_in_length,   // FFT Input Data Length (size of input buffers)
    fft_in_count,    // Count of data entered to FFT input buffers
    fft_out_length,  // FFT Output Length (size of output bins buffer)
    fft_order;       // FFT order (log2 N, the FFT length)

  uint32_t
    center_freq,     // Center of the frequency range covered by monitors
    rx_bandwidth,    // Transceiver Demodulator Bandwidth
    fft_bandwidth;   // FFT and Spectrum Roofing Filter Bandwidth in Hz

  // Watefall Frame Rate
  uint8_t fft_frame_rate;

  // Whether FFT Spectrum Monitors are running
  BOOLEAN
    spectrum_init,      // FFT Spectrum Initialized
    fft_forward,        // Forward or Reverse FFT
    receiver_spectrum,  // FFT Spectrum of DDC Transceiver stream
    wideband_spectrum,  // FFT Spectrum of Wideband stream
    spectrum_active,    // An FFT Spectrum Display activated (one of above)
    spectrum_running;   // The Spectrum Display thread is running

  pthread_t spectrum_thread;

} spectrum_data_t;

//----------------------------------------------------------------------

/* Strucuture holding data of the Transceiver object, e.g.
 * the the variables related to a main client window */
typedef struct _TRANSCEIVER_OBJECT
{
  GtkWidget  *tcvr_window;  // The Transceiver GUI window
  GtkBuilder *builder;      // The Builder object for the Transceiver window

  // Position of Window
  gint tcvr_window_x, tcvr_window_y;

  // The index number of the Transceiver object
  uint8_t index;

  // Receiver-related Transceiver window widgets
  GtkWidget
    *afc_checkbutton,
    *smeter_levelbar,
    *rx_bw_combobox,
    *rx_weaver_combobox,
    *rx_bands_combobox,
    *fft_bw_combobox,
    *fft_rate_combobox,
    *spectrum_off_radiobutton,
    *receiver_spectrum_radiobutton,
    *wideband_spectrum_radiobutton,
    *startrx_togglebutton,
    *rsid_apply_button,
    *rsid_image;

  // Transmitter-related Transceiver window widgets
  GtkWidget
    *mox_togglebutton,
    *power_levelbar,
    *vswr_levelbar,
    *tx_bands_combobox,
    *tx_bw_combobox,
    *tx_weaver_combobox,
    *tx_power_hscale,
    *mic_hscale,
    *mic_label,
    *mic_check,
    *comp_checkbutton,
    *lpf_checkbutton;

  // Per Transceiver Flags
  BOOLEAN
    tx_freq_lock,       // Lock Transmitter Frequency
    link_tx_rx_freq,    // Link Transmitter to Receiver Frequency
    rx_zero_right,      // Enable zeroing of rx frequency digits to the right
    tx_zero_right,      // Enable zeroing of tx frequency digits to the right
    set_tx_freq_status, // Status of setting the DUC center frequency
    set_rx_freq_status, // Status of setting the DDC center frequency
    new_receiver,       // New Receiver object flag (initialize)
    receive_active,     // Whether receiving IQ data from a DDC
    timedec_active,     // Whether Time Station Receiver is running
    rx_cw_modes,        // Rx CW modes in use
    tx_cw_modes,        // Tx CW modes in use
    afc_enable_status,  // Status On|Off of AFC enable checkbutton
    afc_thread_running, // The AFC Thread function is running
    squelch_on_status,  // Status of Squelch Enable check button
    demodulator_init,   // Initialize the Demodulator function
    demodulator_exit,   // Signal Demodulator function to exit
    mic_filter,         // Enable microphone low pass filter
    mic_compress,       // Enable microphone compressor
    block_freq_change,  // Block Receiver and Transmitter freq change
    tx_rsid_enable,     // Enable Tx RSID transmission
    receiver_muted,     // Receiver muted during transmit
    mute_receiver,      // Enable Receiver mute during transmit
    rx_rsid_enable,     // Enable Rx RSID reception
    rx_rsid_running;    // Enable Rx RSID reception

  // Receiver-related per Transceiver data
  uint8_t
    squelch_scale,      // Setting of Squelch scale (slider)
    agc_decay_scale,    // Setting of ADAGC Decay scale (slider)
    snd_vol_scale,      // Setting of Sound Volume scale (slider)
    rx_modulation_mode; // RF Carrier Modulation Mode

  // Tranmsitter-related per Transceiver data
  uint8_t
    mic_level,          // Transmit microphone level
    tx_modulation_mode; // Transmitter Modulation Mode

  // Tx and Rx common - Guest Modulation Mode
  uint8_t guest_modulation_mode;

  // Receiver-related per Transceiver data
  uint32_t rx_frequency;  // Receive Center Frequency in Hz

  // Tranmsitter-related per Transceiver data
  uint32_t
    tx_frequency,        // Transmit Center Frequency in Hz
    tx_fm_deviation;     // Tx frequency deviation in Hz

  // Receiver-related per Transceiver data
  uint16_t
    squelch_value,       // Noise Squelch threshold value
    sound_decimate,      // Decimation factor (hermes2 Sample-Rate/Sound Sample-Rate)
    rx_bandwidth,        // Bandwidth of filter used in Rx demodulators, in Hz
    rx_weaver_frequency; // Rx Phasing frequency used for SSB/CW demod (Weaver method)

  // Tranmsitter-related per Transceiver data
  uint16_t
    tx_bandwidth,        // Bandwidth of filter used in Tx modulators, in Hz
    tx_weaver_frequency; // Tx Phasing frequency used for SSB/CW demod (Weaver method)

  // Tx and Rx Weaver offsets
  int16_t
    rx_weaver_offset,   // TRx center frequency offset due to Phasing frequency
    tx_weaver_offset;   // Tx center frequency offset due to Phasing frequency

  // Receiver-related float data
  double
    S_meter,        // S-Meter level
    smeter_offset,  // S-Meter offset need for calculating above
    adagc_decay,    // Audio Derived AGC decay factor
    adagc_scale,    // Scaling of demod signal before startrx
    sound_volume,   // Sound Volume Control setting (0.0 - 1.0)
    demod_scale;    // Scale factor to keep demod o/p in range

  gint
    rx_bw_idx,          // Rx Demodulator Bandwidth combobox index
    tx_bw_idx,          // Tx Modulator Bandwidth combobox index
    rx_bands_idx,       // Rx Frequency Bands combobox index
    tx_bands_idx,       // Tx Frequency Bands combobox index
    rx_weaver_freq_idx, // Rx Weaver Frequency combobox index
    tx_weaver_freq_idx, // Tx Weaver Frequency combobox index
    fft_bw_idx,         // FFT Bandwidth combobox index
    fft_rate_idx;       // FFT Frame Rate combobox index

  // Enable local (PC sound card) Audio
  BOOLEAN sound_pc_local;

  // Local I/Q data buffers for data streaming
  double
    *data_buf_i[NUM_DATA_BUFFERS],
    *data_buf_q[NUM_DATA_BUFFERS];

  // AFC control semaphore
  sem_t afc_semaphore;

  // I/Q Samples buffers for the LP Filters
  double *demod_id, *demod_qd;

  // Filter data for the main roofing filter
  filter_data_t roof_filter_i, roof_filter_q;

  // Demodulator filter data structs for samples buffers
  filter_data_t demod_filter_data_i, demod_filter_data_q;

  pthread_t
    rsid_thread,   // RSID thread
    sound_thread,  // PC Sound thread
    afc_thread;    // AFC thread

  // Frequency offset data
  struct FREQ_OFFSET
  {
    double ave_offset;
    double norm_offset;
    uint32_t limit;
    uint32_t count;
  } freq_offset_data;

  // Monitor display buffer
  spectrum_data_t spectrum_data;

  // Pointer to current demodulator function
  BOOLEAN (*Demodulator)( struct _TRANSCEIVER_OBJECT *TRx, uint32_t buf_len );

  // Pointer to the next (new) demodulator function
  BOOLEAN (*New_Demodulator)( struct _TRANSCEIVER_OBJECT *TRx, uint32_t buf_len );

  // The RISD mode to transmit for TxID
  uint16_t RSID_Mode;

  // Te RSID mode selected by user to listen for
  char RSID_select[16];

} Transceiver_t;

//----------------------------------------------------------------------

/* transceiver.c */
void ADC_Add_Transceiver(void);
void Copy_TCVR_Settings(uint8_t src, uint8_t dest);
void Free_Data_Buffers(Transceiver_t *TRx);
void Transceiver_Window_Delete(GtkWidget *widget);
void Transceiver_Window_Destroy(GObject *object);

//----------------------------------------------------------------------

#endif

