/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "transceiver.h"
#include "common.h"
#include "cfft.h"
#include "filters.h"
#include "shared.h"
#include "shared.h"
#include "utils.h"
#include "../common/guest_utils.h"
#include "../hpsdr/discovery.h"
#include "../Hermes2/bookmarks.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../Hermes2/display.h"
#include "../Hermes2/process.h"
#include "../Hermes2/spectrum.h"
#include "../rsid/rsid.h"
#include "../time/interface.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

#define TCVR_WINDOW_IDS \
  "tcvr_window", \
  "power_adjustment", \
  "mic_adjustment", \
  "volume_adjustment", \
  "agc_adjustment", \
  "squelch_adjustment", \
  "rx_volume_adjustment", \
  "rsid_image", \
  NULL

//----------------------------------------------------------------------

/* Adc_Add_Transceiver()
 *
 * Adds a new Transceiver object after on_adc_addrx_button_clicked
 * signal and builds an associated Transceiver window for it. For
 * Hermes Lite 2 there is only one ADC (adc0).
 */
  void
ADC_Add_Transceiver( void )
{
  gchar *object_ids[] = { TCVR_WINDOW_IDS };
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  gchar title[24];
  size_t mreq;

  // Open transceiver windows, max of number of DDC's per ADC
  if( Indices.Num_of_TRXs >= ddv->num_of_ddcs ) return;
  Indices.TRx_Index = Indices.Num_of_TRXs;

  // Allocate new Transceiver object
  Transceiver[Indices.TRx_Index] = NULL;
  Mem_Alloc( (void **) &Transceiver[Indices.TRx_Index], sizeof(Transceiver_t) );
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Default settings for a new Transceiver
  TRx->new_receiver       = True;
  TRx->demodulator_exit   = False;
  TRx->guest_modulation_mode = RX_MODE_ITEMS;
  TRx->demod_id = NULL;
  TRx->demod_qd = NULL;
  TRx->spectrum_data.fft_in_i = NULL;
  TRx->spectrum_data.fft_in_q = NULL;
  TRx->spectrum_data.fft_bin_values   = NULL;
  TRx->spectrum_data.spectrum_init    = False;
  TRx->roof_filter_i.samples_buffer   = NULL;
  TRx->roof_filter_q.samples_buffer   = NULL;
  TRx->spectrum_data.waterfall_pixbuf = NULL;
  TRx->spectrum_data.waterfall_pixels = NULL;
  TRx->afc_thread_running = False;
  RsId.rxid_initialized   = False;
  RsId.txid_initialized   = False;
  TRx->RSID_select[0]     = '\0';

  for( uint8_t idx = 0; idx < NUM_DATA_BUFFERS; idx++ )
  {
    TRx->data_buf_i[idx] = NULL;
    TRx->data_buf_q[idx] = NULL;
  }

  // Prepare buffers for Transceiver
  mreq = (size_t)hermes2_rc.ddc_buffer_length * sizeof(double);
  for( uint8_t idx = 0; idx < NUM_DATA_BUFFERS; idx++ )
  {
    Mem_Alloc( (void **) &(TRx->data_buf_i[idx]), mreq );
    Mem_Alloc( (void **) &(TRx->data_buf_q[idx]), mreq );
    memset( TRx->data_buf_i[idx], 0, mreq );
    memset( TRx->data_buf_q[idx], 0, mreq );
  }

  // Create Roof filter buffers
  Mem_Alloc( (void **) &(TRx->roof_filter_i.samples_buffer), mreq );
  Mem_Alloc( (void **) &(TRx->roof_filter_q.samples_buffer), mreq );
  memset( TRx->roof_filter_i.samples_buffer, 0, mreq );
  memset( TRx->roof_filter_q.samples_buffer, 0, mreq );

  // Create a new transceiver window on ADC 0
  Gtk_Builder( &(TRx->builder), (gchar *)hermes2_rc.hermes2_glade, object_ids );
  TRx->tcvr_window = Builder_Get_Object( TRx->builder, "tcvr_window");
  TRx->index = Indices.TRx_Index;

  // Add a title to make windows unigue. Only one ADC (adc=0) in HL2.
  snprintf( title, sizeof(title), _("Hermes2 TRX%d/ADC%d"), TRx->index, 0 );
  gtk_window_set_title( GTK_WINDOW(TRx->tcvr_window), title );

  // Get the widgets of Transceiver window (of Rx)
  TRx->afc_checkbutton    = Builder_Get_Object( TRx->builder, "afc_checkbutton" );
  TRx->smeter_levelbar    = Builder_Get_Object( TRx->builder, "smeter_levelbar" );
  TRx->rx_bw_combobox     = Builder_Get_Object( TRx->builder, "rx_bw_combobox" );
  TRx->rx_weaver_combobox = Builder_Get_Object( TRx->builder, "rx_weaver_combobox" );
  TRx->rx_bands_combobox  = Builder_Get_Object( TRx->builder, "rx_bands_combobox" );
  TRx->fft_bw_combobox    = Builder_Get_Object( TRx->builder, "fft_bw_combobox" );
  TRx->fft_rate_combobox  = Builder_Get_Object( TRx->builder, "fft_rate_combobox" );
  TRx->spectrum_off_radiobutton =
    Builder_Get_Object( TRx->builder, "spectrum_off_radiobutton" );
  TRx->receiver_spectrum_radiobutton =
    Builder_Get_Object( TRx->builder, "receiver_spectrum_radiobutton" );
  TRx->wideband_spectrum_radiobutton =
    Builder_Get_Object( TRx->builder, "wideband_spectrum_radiobutton" );
  TRx->startrx_togglebutton = Builder_Get_Object( TRx->builder, "rx_start_togglebutton" );
  TRx->rsid_apply_button    = Builder_Get_Object( TRx->builder, "rsid_apply_button" );
  TRx->rsid_image           = Builder_Get_Object( TRx->builder, "rsid_image" );

  // Of Tx
  TRx->mox_togglebutton   = Builder_Get_Object( TRx->builder, "mox_togglebutton" );
  TRx->power_levelbar     = Builder_Get_Object( TRx->builder, "power_levelbar" );
  TRx->vswr_levelbar      = Builder_Get_Object( TRx->builder, "vswr_levelbar" );
  TRx->tx_bands_combobox  = Builder_Get_Object( TRx->builder, "tx_bands_combobox" );
  TRx->tx_bw_combobox     = Builder_Get_Object( TRx->builder, "tx_bw_combobox" );
  TRx->tx_weaver_combobox = Builder_Get_Object( TRx->builder, "tx_weaver_combobox" );
  TRx->tx_power_hscale    = Builder_Get_Object( TRx->builder, "tx_power_hscale" );
  TRx->mic_hscale         = Builder_Get_Object( TRx->builder, "mic_hscale" );
  TRx->mic_label          = Builder_Get_Object( TRx->builder, "mic_label" );
  TRx->mic_check          = Builder_Get_Object( TRx->builder, "mic_check_box" );
  TRx->comp_checkbutton   = Builder_Get_Object( TRx->builder, "mic_comp_checkbutton" );
  TRx->lpf_checkbutton    = Builder_Get_Object( TRx->builder, "mic_lpf_checkbutton" );

  // Append available Demod B/W and Weaver freq to relevant combobox's
  Append_BWidth_Weaver( TRx );

  // Set default modulation modes
  uint8_t mode = TX_MODE_USBM; // FIXME may need to be removed
  Rx_Modulation_Mode_Changed( TRx->index, mode );
  Tx_Modulation_Mode_Changed( TRx->index, mode );

  // Wait for GTK to complete its tasks
  gtk_widget_show( TRx->tcvr_window );
  while( g_main_context_iteration(NULL, FALSE) );

  // Count up number of Transceiver instances and signal HL2
  Indices.Num_of_TRXs++;
  hermes2_rc.num_of_receivers = Indices.Num_of_TRXs;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] &= 0x87;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] &= 0x87;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] |= (uint8_t)( (hermes2_rc.num_of_receivers - 1) << 3 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] |= (uint8_t)( (hermes2_rc.num_of_receivers - 1) << 3 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

} // ADC_Add_Transceiver()/

//----------------------------------------------------------------------

/* Copy_TCVR_Settings()
 *
 * Copies some of the settings from one Transceiver_t struct to another
 */
  void
Copy_TCVR_Settings( uint8_t src, uint8_t dest )
{
  const Transceiver_t *TRx_src = Transceiver[src];
  Transceiver_t *TRx_dest = Transceiver[dest];

  TRx_dest->tx_bands_idx      = TRx_src->tx_bands_idx;
  TRx_dest->mic_level         = TRx_src->mic_level;
  TRx_dest->mic_compress      = TRx_src->mic_compress;
  TRx_dest->mic_filter        = TRx_src->mic_filter;
  TRx_dest->rx_bands_idx      = TRx_src->rx_bands_idx;
  TRx_dest->fft_bw_idx        = TRx_src->fft_bw_idx;
  TRx_dest->fft_rate_idx      = TRx_src->fft_rate_idx;
  TRx_dest->snd_vol_scale     = TRx_src->snd_vol_scale;
  TRx_dest->agc_decay_scale   = TRx_src->agc_decay_scale;
  TRx_dest->squelch_scale     = TRx_src->squelch_scale;
  TRx_dest->squelch_on_status = TRx_src->squelch_on_status;
  TRx_dest->rx_zero_right     = TRx_src->rx_zero_right;
  TRx_dest->tx_zero_right     = TRx_src->tx_zero_right;
  TRx_dest->mute_receiver     = TRx_src->mute_receiver;
  TRx_dest->rx_rsid_enable    = TRx_src->rx_rsid_enable;
  TRx_dest->link_tx_rx_freq   = TRx_src->link_tx_rx_freq;
  TRx_dest->tx_rsid_enable    = TRx_src->tx_rsid_enable;
  TRx_dest->tcvr_window_x     = TRx_src->tcvr_window_x;
  TRx_dest->tcvr_window_y     = TRx_src->tcvr_window_y;

} // Copy_TCVR_Settings()

//----------------------------------------------------------------------

// Free resources
  void
Free_Data_Buffers( Transceiver_t *TRx )
{
  for( uint8_t idx = 0; idx < NUM_DATA_BUFFERS; idx++ )
  {
    Mem_Free( (void **) &(TRx->data_buf_i[idx]) );
    Mem_Free( (void **) &(TRx->data_buf_q[idx]) );
  }
}

//----------------------------------------------------------------------

/* Transceiver_Window_Delete()
 *
 * Handles the on_tcvr_window_delete_event callback
 */
  void
Transceiver_Window_Delete( GtkWidget *widget )
{
  GtkWidget *togg;

  /* Stop all activity and wait for threads to terminate before
   * destroying Transceiver objects and freeing resources */
  Guest_Quit_Activate( hermes2_gui.guest_window );
  Hermes2_Stop();

  for( uint8_t idx = 0; idx < Indices.Num_of_TRXs; idx++ )
  {
    // Stop Time Receive and Decode
    if( time_window && (Indices.Time_TRx_Idx == idx) )
    {
      togg = Builder_Get_Object( time_window_builder, "time_receive_togglebutton");
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(togg), FALSE );
      while( g_main_context_iteration(NULL, FALSE) );
    }

  } // for( uint8_t idx = 0; idx < Num_of_TRXs; idx++ )

} // Transceiver_Window_Delete()

//----------------------------------------------------------------------

/* Transceiver_Window_Destroy()
 *
 * Handles the on_tcvr_window_destroy callback
 */
  void
Transceiver_Window_Destroy( GObject *object )
{
  // Get the transceiver index from the window title
  const gchar *title = gtk_window_get_title( GTK_WINDOW(object) );
  title = strstr( title, "TRX" );
  uint8_t rx_idx = (uint8_t)atoi( title + 3 );
  uint8_t idx;
  Transceiver_t *TRx = Transceiver[rx_idx];


  // Close these windows to avoid confusion
  for( idx = 0; idx < Indices.Num_of_TRXs; idx++ )
  {
    // Close Time window if open
    if( time_window )
    {
      gtk_widget_destroy( time_window );
      while( g_main_context_iteration(NULL, FALSE) );
    }
  } // for( uint8_t idx = 0; idx < Num_of_TRXs; idx++ )

  // Close Bookmarks window if open
  if( hermes2_gui.bmk_window && (Indices.Bookmarks_TRx_Idx == rx_idx) )
  {
    Save_Bookmarks_File();
    gtk_widget_destroy( hermes2_gui.bmk_window );
    while( g_main_context_iteration(NULL, FALSE) );
  }

  // Count down number of transceivers
  Indices.Num_of_TRXs--;
  hermes2_rc.num_of_receivers = Indices.Num_of_TRXs;
  if( hermes2_rc.num_of_receivers )
  {
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] &= 0x87;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] &= 0x87;
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] |= (uint8_t)( (hermes2_rc.num_of_receivers - 1) << 3 );
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] |= (uint8_t)( (hermes2_rc.num_of_receivers - 1) << 3 );
    Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
    Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
  }

  // Free the various allocated objects and buffers
  g_object_unref( TRx->builder );
  g_object_unref( G_OBJECT(TRx->spectrum_data.waterfall_pixbuf) );

  Free_Data_Buffers( TRx );
  Free_Demod_Buffers( TRx );
  Mem_Free( (void **) &(TRx->roof_filter_i.samples_buffer) );
  Mem_Free( (void **) &(TRx->roof_filter_q.samples_buffer) );
  Deinit_Chebyshev_Filter( &(TRx->roof_filter_i) );
  Deinit_Chebyshev_Filter( &(TRx->roof_filter_q) );
  Deinit_Chebyshev_Filter( &(TRx->demod_filter_data_i) );
  Deinit_Chebyshev_Filter( &(TRx->demod_filter_data_q) );
  Deinit_FFT( &(TRx->spectrum_data) );
  Free_Scope_Points( rx_idx );

  // Disable the common Guest modulation mode
  TRx->guest_modulation_mode = RX_MODE_ITEMS;

  // Re-arrange windows sequence
  for( idx = rx_idx; idx < Indices.Num_of_TRXs; idx++ )
  {
    // Transfer window titles to match new window index
    gchar new_title[24];

    // Only one ADC in Hermes Lite 2 (adc 0)
    snprintf( new_title, sizeof(new_title),  _("Hermes2 TRX%d/ADC%d"), idx, 0 );
    gtk_window_set_title( GTK_WINDOW(Transceiver[idx + 1]->tcvr_window), new_title );

    // Move Transceiver objects down into the location of deleted window
    memcpy( Transceiver[idx], Transceiver[idx + 1], sizeof(Transceiver_t) );
    Transceiver[idx]->index = idx;
  }

  // Free the last Transceiver object
  Mem_Free( (void **) &Transceiver[Indices.Num_of_TRXs] );

  // Keep Transceiver Indices in range
  if( Indices.TRx_Index >= Indices.Num_of_TRXs )
    Indices.TRx_Index = Indices.Num_of_TRXs - 1;
  if( Indices.Bookmarks_TRx_Idx >= Indices.Num_of_TRXs )
    Indices.Bookmarks_TRx_Idx = (int8_t)Indices.Num_of_TRXs - 1;
  if( Indices.Time_TRx_Idx >= Indices.Num_of_TRXs )
    Indices.Time_TRx_Idx = Indices.Num_of_TRXs - 1;

} // Transceiver_Window_Destroy()

//----------------------------------------------------------------------

