/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_SHARED_H
#define COMMON_SHARED_H    1

#include "hermes2_rc.h"
#include "transceiver.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/pc_to_hw.h"
#include <gtk/gtk.h>
#include <semaphore.h>
#include <sys/socket.h>
#include <stdint.h>

//----------------------------------------------------------------------

// Clamp an angle to max 2pi
#define CLAMP_2PI(x) ( (x) > M_2PI ? (x -= M_2PI) : (x) )

// Clamp an angle between +/- 2pi
#define CLAMP_N2PI(x) (((x) > M_2PI) ? (x -= M_2PI) : (((x) < -M_2PI) ? (x += M_2PI) : (x)))

// Clamp an angle between +/- pi
#define CLAMP_PI(x) (((x) > M_PI) ? (x -= M_2PI) : (((x) < -M_PI) ? (x += M_2PI) : (x)))

// Clamp a value between +/- 32767 (int16_t)
#define CLAMP_S16(x) (((x) > 32767) ? (x = 32767) : (((x) < -32767) ? (x = 32767) : (x)))

/* Clamps a double value between min and max */
  inline void
dClamp( double *x, double min, double max )
{
  if( *x < min ) *x = min;
  else if( *x > max ) *x = max;
}

//----------------------------------------------------------------------

// Runtime config data
extern hermes2_rc_t hermes2_rc;

//----------------------------------------------------------------------

// Command Control and Audio IQ Data packet PC to Metis
extern Cmnd_Ctrl_Audio_IQ_Data_t Cmnd_Ctrl_Audio_IQ_Data;

//----------------------------------------------------------------------

// Radio Transceiver (DDC/DUC processing) object
extern Transceiver_t *Transceiver[MAX_RECEIVERS];

/* Pointers to the current I and Q float buffers in above */
extern double *demod_id_cpy, *demod_qd_cpy;
extern size_t demod_iq_buf_siz;

//----------------------------------------------------------------------

// Pointer to Modulator function
extern void (*Modulator)( void *data );

// Pointer to follow-up Modulator function
extern void (*Next_Modulator)( void *data );

//----------------------------------------------------------------------

// Device discovery variables
#define MAX_DEVICES   16
extern discovered_device_t Device[MAX_DEVICES];

//----------------------------------------------------------------------

// Send-Receive socket
extern int data_socket_fd;

//----------------------------------------------------------------------

// Audio samples for digital modes (psk31, rtty, hell etc )
#define DIGIMODE_BUFFER_SIZE    2048
#define NUM_DIGIMODE_BUFFERS   8
extern int16_t digimode_buffer[NUM_DIGIMODE_BUFFERS][DIGIMODE_BUFFER_SIZE];
extern uint8_t digi_buf_input;

// Data transfer semaphore for digimodes
extern sem_t digimode_semaphore;

// Rx data transfer semaphore for RSID
extern sem_t rsid_semaphore;

// Synchronises sending samples packet to DUC
extern sem_t duc_send_semaphore;

//----------------------------------------------------------------------

// Indices and counts related to Transceiver objects
typedef struct _INDICES
{
  uint8_t
    Num_of_TRXs,   // Number of Transceiver Objects created
    TRx_Index,     // Index of the current Transceiver Object
    Time_TRx_Idx;  // Transceiver Index related to the Time Tx decoder

  // Transceiver Index related to the Bookmarks Window
  int8_t Bookmarks_TRx_Idx;
} Indices_t;

extern Indices_t Indices;

//----------------------------------------------------------------------

// Global window and dialog widgets
typedef struct _HERMES2_GUI
{
  GtkWidget
    *window,
    *device_combobox,
    *sample_rate_combobox,
    *lna_gain_spinbutton,
    *lna_gain_auto,
    *lna_gain_hw,
    *tx_pa_enable,
    *tx_enable_icon,
    *tx_overload_icon,
    *tx_rx_duplex,
    *pa_cur_label,
    *temp_label,
    *tx_fifo_label,
    *rx_antenna1,
    *rx_antenna2,
    *adc_ol_icon,
    *overflow_icon,
    *underflow_icon,
    *bmk_window,
    *quit_dialog,
    *error_dialog,
    *message_dialog,
    *discovery_dialog,
    *config_dialog,
    *save_dialog,
    *guest_window;

  GtkBuilder
    *window_builder,
    *config_dialog_builder,
    *bmk_window_builder;

} hermes2_gui_t;

extern hermes2_gui_t hermes2_gui;

//----------------------------------------------------------------------

// DDC Audio Samples ring buffer
typedef struct _RING_BUFF
{
  uint32_t
    buff_in,
    buff_out,
    buff_size;

  int16_t
    *buff_left,
    *buff_right;

} audio_ring_buff_t;

extern audio_ring_buff_t audio_ring_buff;

//----------------------------------------------------------------------

// Xmit DUC signal samples buffer
typedef struct _XMIT_BUFFER
{
  int16_t *xmit_buf_i, *xmit_buf_q;

  // Required Xmit DUC signal samples buffer size
  uint32_t xmit_buf_size;

  // Actual Xmit DUC signal samples buffer length
  uint32_t xmit_buf_len;

  // The status of packet transmission
  BOOLEAN status;

} xmit_buffer_t;

extern xmit_buffer_t xmit_buffer;

//----------------------------------------------------------------------

/* Socket addresses for the different
 * packages to be sendto() or recvfrom() */
typedef struct _SOCK_ADDR
{
  struct sockaddr_in base_address;

  // Length of above Socket address
  socklen_t  base_address_length;

} sock_addr_t;

extern sock_addr_t sock_addr;

//----------------------------------------------------------------------

// Flow control flags
enum _FLAGS
{
  FTX_DEC_DISPLAY_CB    = 0, // The idle add callback to display decoder results is active
//
  GUEST_NO_CALLBACKS       , // Don't respond to callback if needed
  GUEST_RECORD_QSO         , // Enable recording of QSO's
  GUEST_CAPITALIZE         , // Capitalize letters in Transmit
  GUEST_TRANSMIT_MODE      , // Guest in Transmit mode
  GUEST_TRANSMITTING       , // hermes2 Guest is transmitting
  GUEST_RECEIVE_MODE       , // Guest is in Receive mode
  GUEST_RECEIVING          , // hermes2 Guest is receiving
  GUEST_DEMOD_IQ_DATA      , // Guest mode request for processed IQ data from Demodulator
  GUEST_DEMOD_WVR_DATA     , // Guest mode request for Weaver IQ paths from Demodulator
  GUEST_TRANSMIT_MACRO     , // Start transmission of Macro
  GUEST_TRANSMIT_TAG       , // Transmit text of tag in macro
  GUEST_TRANSMIT_KEYBD     , // Transmission from Keyboard
  GUEST_TRANSMIT_ID        , // Transmit ID in Morse code
  GUEST_SAVE_RECORD        , // Save Record to file
  GUEST_ENABLE_SCOPE       , // Display signal strength 'scope
  GUEST_CLEAR_TX_WINDOW    , // Clear the Transmit window
  GUEST_CLEAR_RX_WINDOW    , // Clear the Receive window
  GUEST_KEYBD_BUSY         , // Transmitting from Keyboard
  GUEST_QUIT               , // Quit Guest Mode
//
  HELL_AUTO_DESKEW         , // Enable automatic deskewing
  HELL_SAVE_PIXBUF         , // Save the Receive window pixbuf
  HELL_MODE_FELDHELL       , // Select FeldHell mode
  HELL_LOOPBACK            , // Loopback Tx to Rx
  HELL_NEW_BAUD_RATE       , // New baud rate selected by user
  HELL_REVERSE_VIDEO       , // Reverse Transmiter video
  HELL_DAREA_CONFIG        , // Rx drawingarea configured
//
  HERMES2_PCSOUND_SETUP     , // PC Sound O/P (via ALSA) Setup OK
  HERMES2_CAPTURE_SETUP     , // ALSA Mic Sound Capture Setup OK
  HERMES2_MIXER_SETUP       , // ALSA Sound Mixer Setup OK
  HERMES2_PCSOUND_RUNNING   , // PC Sound O/P Thread has exited
  HERMES2_RCCONFIG_SETUP    , // hermes2 Configuration read and set-up
  HERMES2_OPERATION_PENDING , // An operation is pending, wait for it
  HERMES2_SAVE_BOOKMARKS    , // Save Stations File flag
  HERMES2_BLOCK_GTK_SIGNALS , // Block response to a signal
  HERMES2_VERBOSE_MESSAGES  , // Enable Verebose message printouts
  HERMES2_ERROR_MESSAGES    , // Enable Error message printouts
  HERMES2_SUSPEND_DATA_IO   , // Suspend Data IO operations
  HERMES2_CREATE_CONFIG     , // Create a default configuration file
  HERMES2_INITIALIZED       , // hermes2 was initialized OK
  HERMES2_SEND_DUC_PACKET   , // Signal to send samples packets to DUC
  HERMES2_QUIT              , // Hermes2 commanded to Quit
//
  MORSE_ADAPT_SPEED        , // Enable speed tracking
  MORSE_MAN_THRESHOLD      , // Enable manual detector threshold
  MORSE_DISPLAY_SIGNAL     , // Display output of signal detector
  MORSE_DISPLAY_RATIO      , // Display situation of Ratio threshold
  MORSE_DISPLAY_LEVEL      , // Display situation of Level threshold
  MORSE_SCOPE_READY        , // Scope data ready to display
  MORSE_SCOPE_HOLD         , // Hold scope display
  MORSE_SELECT_LEVEL       , // Select Level or Ratio situation display
  MORSE_ENABLE_RECEIVE     , // Enable reception of Morse code
  MORSE_RCCONFIG_SETUP     , // morserc file read ok
  MORSE_MARK_TONE          , // Input signal is mark tone (key down)
  MORSE_SPACE_TONE         , // Input signal is space tone (key up)
  //
  OLIVIA_MODE_32_1000      , // Olivia Mode 32/1000
  OLIVIA_MODE_16_1000      , // Olivia Mode 16/1000
  OLIVIA_MODE_16_500       , // Olivia Mode 16/500
  OLIVIA_MODE_8_500        , // Olivia Mode 8/500
  OLIVIA_MODE_8_250        , // Olivia Mode 8/250
  OLIVIA_MODE_4_250        , // Olivia Mode 4/250
  OLIVIA_MODE_4_125        , // Olivia Mode 4/125
  OLIVIA_FLUSH_RECEIVER    , // Flush the Olivia Receiver
  OLIVIA_EXIT_RECEIVE      , // Exit Olivia Receive Mode
  OLIVIA_TUNE_OFFSET       , // Tune Transceiver to freq offset calculated by Synchronizer
  OLIVIA_LABELS_CB         , // The idle add callback to set labels is active
//
  PSK31_SQUELCH_OPEN       , // Squelch is open
  PSK31_MODE_BPSK          , // Mode is BPSK
  PSK31_MODE_QPSK_USB      , // Mode is QPSK-USB
  PSK31_MODE_QPSK_LSB      , // Mode is QPSK-LSB
  PSK31_CHANNEL_A_SELECT   , // Detector channel A selected
  PSK31_ENABLE_AFC         , // Enable Auto Freq. Control
//
  RTTY_ENABLE_USOS         , // Enable USOS (Un Shift On Space)
  RTTY_SQUELCH_OPEN        , // Signal level allows squelch to open
  RTTY_LTRS_SHIFT          , // Transmitting in LTRS shift mode
  RTTY_REVERSE_AFSK        , // Use Revese AFSK frequency shift
  RTTY_ENABLE_DIVERSITY    , // Enable sudo-diversity in RTTY decoder
  RTTY_ENABLE_SIGNAL       , // Enable display of FSK signals
//
  WEFAX_RECEIVE_STOP       , // Stop WEFAX reception and clean up
  WEFAX_SKIP_ACTION        , // Skip current action
  WEFAX_INIMAGE_PHASING    , // Enable In-Image phasing pulse detection
  WEFAX_SAVE_IMAGE_PGM     , // Save the image buffer to PGM file
  WEFAX_SAVE_IMAGE_JPG     , // Save the image buffer to JPG file
  WEFAX_SAVE_BOOKMARKS     , // Save the bookmarks list
  WEFAX_BMK_LIST_OK        , // Bookmarks List is ready
  WEFAX_START_NEW_IMAGE    , // Restart WEFAX dec. after params change
  WEFAX_SAVE_IMAGE         , // Enable saving of WEFAX image
  WEFAX_WEFAX_QUIT         , // Wefax in quit sequence
//
  TRANSMIT_MORSE_MESG      , // Transmit a message in Morse code
  TIME_NEW_STATION         , // New time station selected
  BLOCK_FUNCTION           , // Flag to block entry to a function
  BEEP_ACTIVE              , // The Beep() function is active
  RSIDTX_ACTIVE            , // RSID transmisasaion active
  WSPR_DECODER_THREAD      , // WSPR Decoder thread is running
  NUM_OF_FLAGS               // Total number of flags
};

// An array of single-byte True or False control flags
extern BOOLEAN Flag[];

//----------------------------------------------------------------------

#endif

