/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

// Fast Fourier Transform
// ( c ) 1999-2004, Pawel Jalocha

#ifndef FFT_H
#define FFT_H    1

#include "../common/common.h"
#include <complex.h>
#include <stdint.h>

//----------------------------------------------------------------------

// Radix-2 FFT function
typedef struct _R2FFT
{
  uint32_t  Size;          // FFT size (needs to be power of 2)
  uint32_t *BitRevIdx;     // Bit-reverse indexing table for data (un)scrambling
  complex double *Twiddle; // Twiddle factors (sine/cos values)

  // Pointers to functions that were in class r2FFT
  void    (*Free)(struct _R2FFT *);
  BOOLEAN (*Preset)(struct _R2FFT *, uint32_t);
  void    (*Scramble)(const struct _R2FFT *, complex double *);
  void    (*SeparTwoReals)(const struct _R2FFT *, const complex double *, complex double *, complex double *);
  void    (*JoinTwoReals)(const struct _R2FFT *, const complex double *, const complex double *, complex double *);
  void    (*CoreProc)(struct _R2FFT *, complex double *);
  void    (*Process)(struct _R2FFT *, complex double *);
  void    (*FFTbf)(complex double *, complex double *, const complex double *);
  void    (*FFT2)(complex double *, complex double *);

} r2FFT_t;

//----------------------------------------------------------------------

// Classic radix-2 butterflies

// Special 2-point FFT for the first pass
  static inline void
r2FFT_FFT2( complex double *x0, complex double *x1 )
{
  complex double x1W;

  x1W = *x1;
  *x1 = creal(*x0) - creal(*x1) + (complex double)I * (cimag(*x0) - cimag(*x1));
  *x0 += x1W;
}

//----------------------------------------------------------------------

  static inline void
r2FFT_FFTbf( complex double *x0, complex double *x1, const complex double *W )
{
  complex double x1W;
  x1W =
    creal(*x1) * creal(*W) + cimag(*x1) * cimag(*W) +
    (complex double)I * ( -creal(*x1) * cimag(*W) + cimag(*x1) * creal(*W) ) ;
  *x1 = creal(*x0) - creal(x1W) + (complex double)I * (cimag(*x0) - cimag(x1W));
  *x0 += x1W;
}

//----------------------------------------------------------------------

extern void r2FFT_Initialize(r2FFT_t *Self);

//----------------------------------------------------------------------

#endif

