/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

// Fast Fourier Transform
// ( c ) 1999-2004, Pawel Jalocha

#include "rfft.h"
#include "common.h"
#include "utils.h"
#include <complex.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <math.h>

//----------------------------------------------------------------------

  static void
r2FFT_Free( r2FFT_t *Self )
{
  Mem_Free( (void **) &Self->BitRevIdx );
  Mem_Free( (void **) &Self->Twiddle );
}

//----------------------------------------------------------------------

// Preset tables for given ( maximum ) processing size
  static BOOLEAN
r2FFT_Preset( r2FFT_t *Self, uint32_t MaxSize )
{
  uint32_t idx, ridx, mask, rmask;
  uint32_t Size4;

  if( MaxSize < 4 )
  {
    Self->Free( Self );
    return( False );
  }
  Self->Size = MaxSize;

  while( (MaxSize & 1) == 0 ) MaxSize >>= 1;
  if( MaxSize != 1 )
  {
    Self->Free( Self );
    return( False );
  }

  Mem_Realloc( (void **) &Self->BitRevIdx,
      (size_t)Self->Size * sizeof(uint32_t) );
  Mem_Realloc( (void **) &Self->Twiddle,
      (size_t)Self->Size * sizeof(complex double) );

  Size4 = Self->Size / 4;
  for( idx = 0; idx < Size4; idx++ )
  {
    double phase = ( M_2PI * (double)idx ) / (double)Self->Size;
    Self->Twiddle[idx] = cos( phase ) + (complex double)I * sin( phase );
  }

  for( ; idx < Self->Size; idx++ )
  {
    Self->Twiddle[idx] = -cimag( Self->Twiddle[idx - Size4] ) +
      (complex double)I * creal( Self->Twiddle[idx - Size4] );
  }

  for( idx = 0; idx < Self->Size; idx++ )
  {
    for( ridx = 0, mask = Self->Size/2, rmask = 1; mask; mask >>= 1, rmask <<= 1 )
    {
      if( idx & mask ) ridx |= rmask;
    }

    Self->BitRevIdx[idx] = ridx;
  }

  return( True );
}

//----------------------------------------------------------------------

// Scramble/unscramble (I)FFT input
  static void
r2FFT_Scramble( const r2FFT_t *Self, complex double *x )
{
  uint32_t idx;
  complex double tmp;
  for( idx = 0; idx < Self->Size; idx++ )
  {
    uint32_t ridx;
    if( (ridx = Self->BitRevIdx[idx]) > idx )
    {
      tmp     = x[idx];
      x[idx]  = x[ridx];
      x[ridx] = tmp;
    }
  }
}

//----------------------------------------------------------------------

// Separate the result of a two real channels FFT
  static void
r2FFT_SeparTwoReals(
    const r2FFT_t *Self,
    const complex double *Buff,
    complex double *Out0,
    complex double *Out1 )
{
  uint32_t idx, HalfSize = Self->Size / 2;

  Out0[0] = creal( Buff[0] ) + (complex double)I * creal( Buff[HalfSize] );
  Out1[0] = cimag( Buff[0] ) + (complex double)I * cimag( Buff[HalfSize] );

  for( idx = 1; idx < HalfSize; idx++)
  {
    Out0[idx] =
      creal( Buff[idx] ) + creal( Buff[Self->Size - idx] ) +
      (complex double)I * ( cimag(Buff[idx]) - cimag(Buff[Self->Size - idx]) );

    Out1[idx] =
      cimag( Buff[idx] ) + cimag( Buff[Self->Size - idx] ) +
      (complex double)I * ( -creal(Buff[idx]) + creal(Buff[Self->Size - idx]) );
  }
}

//----------------------------------------------------------------------

/*
 * The oposite of SeparTwoReals() but we NEGATE the
 * .Im part for Inverse FFT and we NEGATE the Inp1[]
 * so that after Process() both .Re and .Im come out
 * right ( no need to negate the .Im part ) as a
 * "by-product" we multiply the transform by 2
 */
  static void
r2FFT_JoinTwoReals(
    const r2FFT_t *Self,
    const complex double *Inp0,
    const complex double *Inp1,
    complex double *Buff )
{
  uint32_t idx, HalfSize = Self->Size / 2;

  Buff[0] = 2.0 * creal(Inp0[0]) + (complex double)I * 2.0 * creal(Inp1[0]);

  for( idx = 1; idx < HalfSize; idx++ )
  {
    Buff[idx] =
      creal( Inp0[idx] ) + cimag( Inp1[idx] ) +
      (complex double)I * ( -cimag(Inp0[idx]) + creal(Inp1[idx]) );

    Buff[Self->Size - idx] =
      creal( Inp0[idx] ) - cimag( Inp1[idx] ) +
      (complex double)I * ( cimag(Inp0[idx]) + creal(Inp1[idx]) );
  }

  Buff[HalfSize] = 2.0 * cimag(Inp0[0]) + (complex double)I * 2.0 * cimag(Inp1[0]);
}

//----------------------------------------------------------------------

/*
 * Core process: the classic tripple loop of butterflies
 * radix-2 FFT: the first and the second pass are by hand
 * looks like there is no gain by separating the second pass
 * and even the first pass is in question ?
 */
  static void
r2FFT_CoreProc( r2FFT_t *Self, complex double *x )
{
  uint32_t Groups, GroupSize2, Group, Bf, TwidIdx;
  uint32_t Size2 = Self->Size / 2;

  for( Bf = 0; Bf < Self->Size; Bf += 2 )
    Self->FFT2( &x[Bf], &x[Bf+1] ); // first pass

  for( Groups = Size2 / 2, GroupSize2 = 2; Groups; Groups >>= 1, GroupSize2 <<= 1 )
    for( Group = 0, Bf = 0; Group < Groups; Group++, Bf += GroupSize2 )
      for( TwidIdx = 0; TwidIdx < Size2; TwidIdx += Groups, Bf++ )
      {
        Self->FFTbf( &x[Bf], &x[Bf + GroupSize2], &Self->Twiddle[TwidIdx] );
      }
}

//----------------------------------------------------------------------

// Complex FFT process in place, includes unscrambling
  static void
r2FFT_Process( r2FFT_t *Self, complex double *x )
{
  Self->Scramble( Self, x );
  Self->CoreProc( Self, x );
}

//----------------------------------------------------------------------

// Initializes r2FFT - my function
  void
r2FFT_Initialize( r2FFT_t *Self )
{
  Self->Free          = r2FFT_Free;
  Self->Preset        = r2FFT_Preset;
  Self->Scramble      = r2FFT_Scramble;
  Self->SeparTwoReals = r2FFT_SeparTwoReals;
  Self->JoinTwoReals  = r2FFT_JoinTwoReals;
  Self->CoreProc      = r2FFT_CoreProc;
  Self->Process       = r2FFT_Process;
  Self->FFTbf         = r2FFT_FFTbf;
  Self->FFT2          = r2FFT_FFT2;

  Self->BitRevIdx = NULL;
  Self->Twiddle   = NULL;
}

//----------------------------------------------------------------------

