/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

/*
 * All data are fixed-point short integers, in which -32768
 * to +32768 represent -1.0 to +1.0 respectively. Integer
 * arithmetic is used for speed, instead of the more natural
 * floating-point.
 *
 * For the forward FFT (time -> freq), fixed scaling is
 * performed to prevent arithmetic overflow, and to map a 0dB
 * sine/cosine wave (i.e. amplitude = 32767) to two -6dB freq
 * coefficients. The return value is always 0.
 *
 * For the inverse FFT (freq -> time), fixed scaling cannot be
 * done, as two 0dB coefficients would sum to a peak amplitude
 * of 64K, overflowing the 32k range of the fixed-point integers.
 * Thus, the fix_fft() routine performs variable scaling, and
 * returns a value which is the number of bits LEFT by which
 * the output must be shifted to get the actual amplitude
 * (i.e. if fix_fft() returns 3, each value of fr[] and fi[]
 * must be multiplied by 8 (2**3) for proper scaling.
 * Clearly, this cannot be done within fixed-point short
 * integers. In practice, if the result is to be used as a
 * filter, the scale_shift can usually be ignored, as the
 * result will be approximately correctly normalized as is.
 * Written by:  Tom Roberts  11/8/89
 * Made portable:  Malcolm Slaney 12/15/94 malcolminterval.com
 * Enhanced:  Dimitrios P. Bouras  14 Jun 2006 dbouras@ieee.org
 *
 * Modified by Neoklis Kyriazis 1 July 2018 nkcyham@yahoo.com
 * The fixed Sinwave table is now dynamically assigned by an
 * initialization function to match FFT size. Function names
 * have also been changed and the FFT routine now accepts FFT
 * size rather than FFT order and it is modified to some extend.
 */

#include "ifft.h"
#include "common.h"
#include "utils.h"
#include <math.h>
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

//----------------------------------------------------------------------

// Max values of iFFT sin table
#define IFFT_SINEWAVE_MAGN   32767.0

//----------------------------------------------------------------------

/* Initialize_iFFT()
 *
 * Initializes iFFT() by creating a dynamically allocated
 * Sinewave table and by calculating the FFT order (log2 N)
 */
  BOOLEAN
Initialize_iFFT( ifft_data_t *ifft_data )
{
  uint16_t a;

  // Abort if ifft_width is not a power of 2
  if( ifft_data->ifft_width & (ifft_data->ifft_width - 1) )
  {
    fprintf( stderr,
        _("FFT size %d is not a power of 2 - aborting\n"), ifft_data->ifft_width );
    Error_Dialog( _("FFT size is not a power of 2"), HIDE_OK );
    return( False );
  }

  // Calculate the order of fft
  a = ifft_data->ifft_width;
  for( ifft_data->ifft_order = 0; a != 1; ifft_data->ifft_order++ )
    a /= 2;

  // If not already initialized
  if( !ifft_data->ifft_init )
  {
    // FFT width (size) and data length
    ifft_data->data_len = 2 * ifft_data->ifft_width;

    // Allocate the iFFT input data and bins buffers
    size_t mreq = (size_t)ifft_data->data_len * sizeof( int16_t );
    Mem_Alloc( (void **) &(ifft_data->ifft_data), mreq );
    Mem_Alloc( (void **) &(ifft_data->ifft_bins), mreq / 2 );

    // Allocate average bin values buffer
    mreq = (size_t)ifft_data->ifft_width * sizeof(uint16_t);
    Mem_Alloc( (void **) &(ifft_data->bin_ave), mreq );
    memset( (void *)(ifft_data->bin_ave), 0, mreq );

    /* Allocate the Sine Wave table. This is twice the
     * length needed in iFFT() as it is also used in
     * iFFT_Real() which requires twice the resolution */
    uint16_t b = ( ifft_data->data_len * 3 ) / 4;
    mreq = (size_t)b * sizeof( int16_t );
    Mem_Alloc( (void **) &(ifft_data->Sinewave), mreq );

    // Build the Sinewave table
    double dw = M_2PI / (double)ifft_data->data_len;
    double w  = 0.0;
    for( a = 0; a < b; a++ )
    {
      ifft_data->Sinewave[a] = (int16_t)( IFFT_SINEWAVE_MAGN * sin(w) );
      w += dw;
    }

  } // if( !ifft_data->ifft_init )

  ifft_data->ifft_init = 1;
  return( True );

} // Initialize_iFFT()

//----------------------------------------------------------------------

/*
 * Imply() - fixed-point multiplication & scaling.
 * Scaling ensures that result remains 16-bit.
 */
  static inline int16_t
Imply( int16_t a, int16_t b )
{
  // Shift right one less bit (i.e. 15-1)
  uint32_t c = ( (uint32_t)a * (uint32_t)b ) >> 14;

  // Last bit shifted out = rounding-bit
  b = (int16_t)( c & 0x01 );

  // Last shift + rounding bit
  a = (int16_t)( c >> 1 ) + b;

  return( a );
} // Imply()

//----------------------------------------------------------------------

/* iFFT()
 *
 * This computes an in-place complex-to-complex Radix-2 forward
 * or reverse FFT, BUT in 16-bit (int16_t) integer arithmetic only!
 * reX and imX are the real and imaginary values of 2^m points.
 * ifft_width is the number of points in the time domain as well
 * the number of "bins" returned by the function. The "forward"
 * flag controls the forward or reverse FFT direction.
 */
  static void
iFFT( ifft_data_t *ifft_data )
{
  uint16_t a, b, b2, b3, c2, c3, d, e, t, dt;
  uint16_t step, dft, bfly;
  int16_t tr, ti, wr, wi, qr, qi;


  /* Decompose time domain signal (bit reversal).
   * This is only needed if the time domain points
   * in reX[] and imX[] were not already decomposed
   * while data was being entered into these buffers */
  a = ifft_data->ifft_width - 1;
  for( b = 1; b < a; b++ )
  {
    uint16_t c = 0;
    for( d = 0; d < ifft_data->ifft_order; d++ )
    {
      // Bit reversal of indices
      c <<= 1;
      c |= (b >> d) & 0x01;
    }

    /* Enter data to locations
     * with bit reversed indices */
    if( b < c )
    {
      b2  = 2 * b;
      c2  = 2 * c;
      b3 = b2 + 1;
      c3 = c2 + 1;

      tr = ifft_data->ifft_data[b2];
      ti = ifft_data->ifft_data[b3];
      ifft_data->ifft_data[b2] = ifft_data->ifft_data[c2];
      ifft_data->ifft_data[b3] = ifft_data->ifft_data[c3];
      ifft_data->ifft_data[c2] = tr;
      ifft_data->ifft_data[c3] = ti;
    }
  } // for( b = 1; b < a; b++ )

  // Compute the FFT
  a = 1;
  e = ifft_data->ifft_order - 1;
  dt = ifft_data->data_len / 4;

  /* Loop over the number of decomposition
   * (bit reversal) steps (the FFT order) */
  for( step = 0; step < ifft_data->ifft_order; step++ )
  {
    /* Fixed scaling, for proper normalization --
     * there will be log2(n) passes, so this results
     * in an overall factor of 1/n, distributed to
     * maximize arithmetic accuracy.  */

    // Loop over sub-dft's
    b = a;
    a <<= 1;
    for( dft = 0; dft < b; dft++ )
    {
      /* This works for a Sinewave table which is twice the
       * required length so that it can be used in iFFT_Real() */
      t =  2 *  (uint16_t)( 0x01 << (e - step) ) * dft;
      wr = +ifft_data->Sinewave[t + dt];
      wi = -ifft_data->Sinewave[t];

      // Maintain scaling to avoid overflow
      wr >>= 1;
      wi >>= 1;

      // Loop over each butterfly and build up the freq domain
      for( bfly = dft; bfly < ifft_data->ifft_width; bfly += a )
      {
        b2  = 2 * bfly;
        b3 = b2 + 1;
        c2  = 2 * (bfly + b);
        c3 = c2 + 1;

        tr = Imply( wr, ifft_data->ifft_data[c2] ) - Imply( wi, ifft_data->ifft_data[c3] );
        ti = Imply( wi, ifft_data->ifft_data[c2] ) + Imply( wr, ifft_data->ifft_data[c3] );
        qr = ifft_data->ifft_data[b2];
        qi = ifft_data->ifft_data[b3];
        qr >>= 1;
        qi >>= 1;

        ifft_data->ifft_data[c2] = qr - tr;
        ifft_data->ifft_data[c3] = qi - ti;
        ifft_data->ifft_data[b2] = qr + tr;
        ifft_data->ifft_data[b3] = qi + ti;

      } // for( bfly = a; bfly < ifft_data->ifft_ifft_width; bfly += d )
    } // for( dft = 0; dft < e; dft++ )
  } // for( step = 0; step < ifft_data->ifft_ifft_order; step++ )

} // iFFT()

//----------------------------------------------------------------------

/* Compute the forward or inverse Fourier Transform of data, with
 * data containing real valued points only. The output is complex
 * valued after the first two entries, stored in alternating real
 * and imaginary parts. The first two returned entries are the real
 * parts of the first and last value from the conjugate symmetric
 * output, which are necessarily real. The length must be a power
 * of 2.
 * "data" The real data stored as alternating real and imaginary
 * parts with pairs transposed in time, e.g. data should be an array
 * of points in the time domain in sequence X1, X0, X3, X2 ... Xn+1, Xn.
 * "forward" True for a forward transform, False for inverse transform
 */
  void
iFFT_Real( ifft_data_t *ifft_data )
{
  int16_t g, j, dt;


  /* Do the FFT on data. NOTE that time domain
   * data must have consecutive pairs of points
   * swapped, e.g. X1, X0, X3, X2 ... Xn, Xn-1 */
  iFFT( ifft_data );

  // Carry out the FFT on data
  g  = ifft_data->ifft_width / 2;
  dt = ifft_data->data_len / 4;
  for( j = 1; j <= g; j++ )
  {
    int16_t tkr, tki, tjr, tji, wr, wi;
    int16_t a, b, c, d, e, f;
    int16_t j2, j3, k2, k3;

    k2 = 2 * ((int16_t)ifft_data->ifft_width - j);
    k3 = k2 + 1;
    j2 = 2 * j;
    j3 = j2 + 1;

    tkr = ifft_data->ifft_data[k2]; // Real and imaginary parts of t_k = t_(n/2 - j)
    tki = ifft_data->ifft_data[k3];
    tjr = ifft_data->ifft_data[j2]; // Real and imaginary parts of t_j
    tji = ifft_data->ifft_data[j3];

    wr = ifft_data->Sinewave[j];
    wi = ifft_data->Sinewave[j + dt];

    a = Imply( (tjr - tkr), wr );
    b = Imply( (tji + tki), wi );
    c = Imply( (tjr - tkr), wi );
    d = Imply( (tji + tki), wr );

    e = tjr + tkr;
    f = tji - tki;

    // Compute entry y[j]
    ifft_data->ifft_data[j2] = (e + (a + b)) / 2;
    ifft_data->ifft_data[j3] = (f + (d - c)) / 2;

    // Compute entry y[k]
    ifft_data->ifft_data[k2] = (e - (b + a)) / 2;
    ifft_data->ifft_data[k3] = ((d - c) - f) / 2;

  } // for( j = 1; j <= length / 4; j++ )

  // Compute final y0 and y_{N/2}, store in data[0], data[1]
  int16_t tmp = ifft_data->ifft_data[0];
  ifft_data->ifft_data[0] += ifft_data->ifft_data[1];
  ifft_data->ifft_data[1]  = tmp - ifft_data->ifft_data[1];

} // iFFT_Real()

//----------------------------------------------------------------------

/* iFFT_Data()
 *
 * Collects signal samples and does an iFFT when
 * enough samples are collected for the Waterfall Display
 */
  BOOLEAN
iFFT_Data( int16_t sample, ifft_data_t *ifft_data )
{
  static int32_t
    data  = 0,
    count = 0; // Count of calls to this function

  // Return value
  BOOLEAN ret = False;

  // Summate (decimate) samples for the iFFT
  data += sample;

  // Reset stride (decimation) counter
  count++;
  if( count >= ifft_data->ifft_stride )
  {
    static int32_t idx = 0;

    // Normalize iFFT input samples
    data /= ifft_data->ifft_stride;

    /* Place data in swapped order, X1, X0, X3, X2 ... Xn+1, Xn
     * so that iFFT_Real() works properly with real data input */
    if( idx & 0x01 )
      ifft_data->ifft_data[idx - 1] = (int16_t)data;
    else
      ifft_data->ifft_data[idx + 1] = (int16_t)data;
    idx++;

    // Clear for next summation
    data  = 0;
    count = 0;

    // Display waterfall when input buffer full
    if( idx >= ifft_data->data_len )
    {
      ret = True;
      idx = 0;
    }

  } // if( count >= ifft_stride )

  return( ret );
} // iFFT_Data()

//------------------------------------------------------------------------

// Length and multiplier of amplitude averaging window
#define IFFT_AMPL_AVE_WINDOW    3
#define IFFT_AMPL_AVE_MULTPL    2

// Scale FFT output bins
#define IFFT_SCALE      64

//----------------------------------------------------------------------

/* iFFT_Bin_Value()
 *
 * Calculates iFFT bin values with auto level control
 */
  uint8_t
iFFT_Bin_Value( ifft_data_t *ifft_data, uint16_t idx, BOOLEAN reset )
{
  // Maximum value of ifft bins
  static uint32_t bin_max = 1000, max = 0;

  // Scale output from iFFT
  ifft_data->ifft_data[idx]     /= IFFT_SCALE;
  ifft_data->ifft_data[idx + 1] /= IFFT_SCALE;

  // Calculate sliding window average of max bin value
  if( reset )
  {
    bin_max = max;
    if( !bin_max ) bin_max = 1;
    max = 0;
  }
  else
  {
    // Value of ifft output "bin"
    static uint32_t bin_val = 0;

    // Calculate average signal power at each frequency (bin)
    bin_val  = bin_val * IFFT_AMPL_AVE_MULTPL;
    bin_val += (uint32_t)( ifft_data->ifft_data[idx] * ifft_data->ifft_data[idx] );
    bin_val += (uint32_t)( ifft_data->ifft_data[idx + 1] * ifft_data->ifft_data[idx + 1] );
    bin_val /= IFFT_AMPL_AVE_WINDOW;

    // Record max bin value
    if( max < bin_val ) max = bin_val;

    // Scale bin values to 255 depending on max value
    uint16_t ret = (uint16_t)( (255 * bin_val) / bin_max );
    if( ret > 255 ) ret = 255;

    // Save bin values in ifft bins buffer. idx increments in steps of 2
    ifft_data->ifft_bins[idx / 2] = (int16_t)ret;
    return( (uint8_t)ret );
  }

  return( 0 );
} // iFFT_Bin_Value()

//------------------------------------------------------------------------

// Free resources
  void
Free_iFFT( ifft_data_t *ifft_data )
{
  Mem_Free( (void **) &(ifft_data->ifft_data) );
  Mem_Free( (void **) &(ifft_data->ifft_bins) );
  Mem_Free( (void **) &(ifft_data->bin_ave) );
  Mem_Free( (void **) &(ifft_data->Sinewave) );
  ifft_data->ifft_init = False;
}

//----------------------------------------------------------------------

