/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "hermes2_rc.h"
#include "common.h"
#include "shared.h"
#include "transceiver.h"
#include "utils.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/hpsdr_init.h"
#include "../hpsdr/hw_to_pc.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/display.h"
#include "../Hermes2/interface.h"
#include "../Hermes2/callback_func.h"
#include "../Hermes2/demodulate.h"
#include "../morse/interface.h"
#include "../morse/shared.h"
#include "../time/interface.h"
#include <alsa/asoundlib.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

//------------------------------------------------------------------------

// Number of Antenna selection radio buttons
#define NUM_OF_ANTENNAS  2

//----------------------------------------------------------------------

// Default frequencies and modes for each band
#define DEFAULT_BANDS \
{ \
    1000000,  1850000,  3600000,  5360000,  7100000, 10100000, \
   14100000, 18100000, 21200000, 24950000, 28300000  \
}

//----------------------------------------------------------------------

#define DEFAULT_MODES { 1, 4, 4, 4, 4, 1, 1, 1, 1, 1, 1 }

//----------------------------------------------------------------------

#define CONFIG_VERSION  1
#define HERMES2_CONFIG_FORMAT \
  _("### Hermes2 Configuration File ###\n"\
      "#\n"\
      "# Config Version\n"\
      "%d\n"\
      "# CW Hang Time mS\n"\
      "%u\n"\
      "# PTT Hang Time mS\n"\
      "%u\n"\
      "# TX Buffer Latency mS\n"\
      "%u\n"\
      "# Tx Tune Power Setting\n"\
      "%5.1f\n"\
      "# Enable Local (PC) Sound\n"\
      "%u\n"\
      "# ALSA Sound HWCTL Device\n"\
      "%s\n"\
      "# ALSA Sound PCM Device\n"\
      "%s\n"\
      "# ALSA Sound Capture Source\n"\
      "%s\n"\
      "# ALSA Sound Capture Volume control\n"\
      "%s\n"\
      "# ALSA Sound Capture Channel\n"\
      "%d\n"\
      "# ALSA Sound Capture Level\n"\
      "%d\n"\
      "# Beep sound volume\n"\
      "%u\n"\
      "# DDC Buffer Size index\n"\
      "%u\n"\
      "# Device Frequency Correction PPM\n"\
      "%+13.6E\n"\
      "# Hermes2 Window Position\n"\
      "%d,%d\n"\
      "# Bookmarks Window Size\n"\
      "%d,%d\n"\
      "# Bookmarks Window Position\n"\
      "%d,%d\n"\
      "# Time Station Window Position\n"\
      "%d,%d\n"\
      "# Rx/TX Duplex Setting\n"\
      "%u\n"\
      "# Rx DDC Sample Rate index\n"\
      "%d\n"\
      "# LNA Gain Auto Status\n"\
      "%u\n"\
      "# LNA Gain H/W Controlled Status\n"\
      "%u\n"\
      "# LNA Gain Setting\n"\
      "%d\n"\
      "# Receiver Antenna\n"\
      "%u\n"\
      "# Tx RF Delay\n"\
      "%u\n"\
      "# TX PA Enable Setting\n"\
      "%u\n"\
      "# Number of Receiver Slices\n"\
      "%u\n")

#define RECEIVER_SETTINGS_FORMAT \
  _("### Receiver %d Settings ###\n"\
      "# Receiver Bands Combobox index\n"\
      "%d\n"\
      "# FFT Bandwidth index\n"\
      "%u\n"\
      "# FFT Frame Rate index\n"\
      "%u\n"\
      "# Sound Volume Scale Value\n"\
      "%u\n"\
      "# ADAGC Decay Scale Value\n"\
      "%u\n"\
      "# Squelch Scale Value\n"\
      "%u\n"\
      "# Squelch Check Button\n"\
      "%u\n"\
      "# Rx Zero Right Check Button\n"\
      "%u\n"\
      "# Rx Mute on Tx Check Button\n"\
      "%u\n"\
      "# RxID Enable Check Button\n"\
      "%u\n"\
      "# Link Tx to Rx Toggle Button\n"\
      "%u\n" )

#define TRANSMITTER_SETTINGS_FORMAT \
  _("### Transmitter Settings ###\n"\
      "#\n"\
      "# Tx Frequency Bands Combobox Index\n"\
      "%d\n"\
      "# Tx Zero Right Check Button\n"\
      "%u\n"\
      "# Tx Power O/P Limit\n"\
      "%3.1f\n"\
      "# Tx Microphone Level\n"\
      "%u\n"\
      "# Tx Microphone Compress Enable\n"\
      "%u\n"\
      "# Tx Microphone LP Filter\n"\
      "%u\n"\
      "# TxID Enable Check Button\n"\
      "%u\n#"\
      "# Transceiver Window Position\n"\
      "%d,%d\n" )

// Rx and Tx current frequencies per band
#define BANDS_CONFIG_FORMAT _( " %8u %3u    %8u %3u\n" )

// This is the default config saved to hermes2.conf, if it is missing at startup
#define DEFAULT_CONFIG \
_( "### Hermes2 Configuration File ###\n\
#\n\
# Config Version\n\
1\n\
# CW Hang Time mS\n\
500\n\
# PTT Hang Time mS\n\
15\n\
# TX Buffer Latency mS\n\
20\n\
# Tx Tune Power Setting\n\
50.0\n\
# Enable Local (PC) Sound\n\
1\n\
# ALSA Sound HWCTL Device\n\
hw:1\n\
# ALSA Sound PCM Device\n\
hw:1,0\n\
# ALSA Sound Capture Source\n\
Capture\n\
# ALSA Sound Capture Volume control\n\
Capture\n\
# ALSA Sound Capture Channel\n\
0\n\
# ALSA Sound Capture Level\n\
50\n\
# Beep sound volume\n\
2000\n\
# DDC Buffer Size index\n\
0\n\
# Device Frequency Correction PPM\n\
0\n\
# Hermes2 Window Position\n\
0,0\n\
# Bookmarks Window Size\n\
100,100\n\
# Bookmarks Window Position\n\
0,0\n\
# Time Station Window Position\n\
0,0\n\
# Rx/TX Duplex Setting\n\
1\n\
# Rx DDC Sample Rate index\n\
2\n\
# LNA Gain Auto Status\n\
1\n\
# LNA Gain H/W Controlled Status\n\
1\n\
# LNA Gain Setting\n\
20\n\
# Receiver Antenna\n\
0\n\
# RF Delay mS\n\
20\n\
# TX PA Enable Setting\n\
1\n\
# Number of Receiver Slices\n\
1\n\
### Receiver 0 Settings ###\n\
# Receiver Bands Combobox index\n\
5\n\
# FFT Bandwidth index\n\
1\n\
# FFT Frame Rate index\n\
3\n\
# Sound Volume Scale Value\n\
25\n\
# ADAGC Decay Scale Value\n\
90\n\
# Squelch Scale Value\n\
50\n\
# Squelch Check Button\n\
1\n\
# Rx Zero Right Check Button\n\
1\n\
# Rx Mute on Tx Check Button\n\
1\n\
# RxID Enable Check Button\n\
1\n\
# Link Tx to Rx Toggle Button\n\
0\n\
### Transmitter Settings ###\n\
#\n\
# Tx Frequency Bands Combobox Index\n\
5\n\
# Tx Zero Right Check Button\n\
1\n\
# Tx Power O/P Limit\n\
5.0\n\
# Tx Microphone Level\n\
30\n\
# Tx Microphone Compress Enable\n\
1\n\
# Tx Microphone LP Filter\n\
1\n\
# TxID Enable Check Button\n\
1\n\
## Transceiver Window Position\n\
0,0\n\
# Rx Band Rx Mode Tx Band Tx Mode\n\
  1000000  19     1800000  20\n\
  1850000  25     1850000  20\n\
  3600000  25     3600000  20\n\
  5360000  25     5360000  20\n\
  7100000  25     7100000  20\n\
 10140000  25    10140000  20\n\
 14100000  25    14100000  20\n\
 18100000  25    18100000  20\n\
 21200000  25    21200000  20\n\
 24950000  25    24950000  20\n\
 28200000  25    28200000  20\n\
#\n\
[END]\n\
#" )

#define HERMES2_CONFIG_FILE  "/.hermes2/hermes2/hermes2.config"

//----------------------------------------------------------------------

/* Verify_Config()
 *
 * Verifies that the existing hermes2.conf file is of the
 * right version and readble. If not, prompts the user
 * to enable the creation of a new configuration file
 */
  gboolean
Verify_Config( gpointer data )
{
  // Path to config file
  char rc_fpath[FILE_NAME_SIZE];
  FILE *fp = NULL;

  // Setup file path to Hermes2 config file
  Strlcpy( rc_fpath, getenv("HOME"), sizeof(rc_fpath) );
  Strlcat( rc_fpath, HERMES2_CONFIG_FILE, sizeof(rc_fpath) );
  if( Open_File(&fp, rc_fpath, "r") == SUCCESS )
  {
    // Buffer for Read_Line()
    char line[READ_LINE_BUF_SIZE];

    // Read Application Version
    int ret = Read_Line( line, fp, _("hermes2.config file version") );

    /* Produce fresh default config file
     * if version number new or unreadable */
    if( ret == ERROR )
    {
      fprintf( stderr,
          _("hermes2: failed to read hermes2.config file version\n") );
      Flag[HERMES2_CREATE_CONFIG] = True;
      Message_Dialog(
          _("Failed to read hermes2.config file version.\n"
            "Create a new configuration file for hermes2?") );
      return( FALSE );
    }
    else if( CONFIG_VERSION != atoi(line) )
    {
      fprintf( stderr,
          _("hermes2: existing hermes2.config file version incompatible: %s\n"), line );
      Flag[HERMES2_CREATE_CONFIG] = True;
      Message_Dialog(
          _("Existing hermes2.config file version incompatible.\n"
            "Create a new configuration file for hermes2?") );
      Close_File( &fp );
      return( FALSE );
    }
  } // if( Open_File(&fp, cfg_file, "r") )
  else // No config file
  {
    fprintf( stderr,
        _("hermes2: failed to open hermes2 config file: %s\n"), rc_fpath );
    Flag[HERMES2_CREATE_CONFIG] = True;
    Message_Dialog(
        _("Failed to open hermes2 configuration file.\n"
          "Create a new configuration file for hermes2?") );
    return( FALSE );
  }

  // Read configuration file
  Close_File( &fp );
  g_idle_add( Read_Config, NULL );

  return( FALSE );
} // Verify_Config()

//------------------------------------------------------------------------

/* Create_Hermes2_Config()
 *
 * Creates a new hermes2.conf configuration file
 * from settings in the Configuration Dialog.
 */
  void
Create_Hermes2_Config( void )
{
  GtkWidget *widget;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];


  // Get the HPSDR device type index
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_device_combobox" );
  gchar *txt = gtk_combo_box_text_get_active_text( GTK_COMBO_BOX_TEXT(widget) );
  if( txt != NULL )
  {
    for( hermes2_rc.device_index = 0;
        hermes2_rc.device_index < MAX_DEVICES;
        hermes2_rc.device_index++ )
    {
      if( strcmp(txt, Device[hermes2_rc.device_index].device_name) == 0 )
        break;
    }
  }
  else hermes2_rc.device_index = 0;
  g_free( txt );

  // Get the ADC TCXO frequency error
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_freq_error_entry" );
  hermes2_rc.device_freq_error = atof( gtk_entry_get_text(GTK_ENTRY(widget)) );

  // Get the ALSA HWCTL device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_hwctl_device_entry" );
  Strlcpy( hermes2_rc.hwctl_device,
      gtk_entry_get_text(GTK_ENTRY(widget)), sizeof(hermes2_rc.hwctl_device) );

  // Get the ALSA PCM device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_pcm_device_entry" );
  Strlcpy( hermes2_rc.pcm_device,
      gtk_entry_get_text(GTK_ENTRY(widget)), sizeof(hermes2_rc.pcm_device) );

  // Get the ALSA Capture device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_capture_dev_entry" );
  Strlcpy( hermes2_rc.capture_src,
      gtk_entry_get_text(GTK_ENTRY(widget)), sizeof(hermes2_rc.capture_src) );

  // Get the ALSA Capture Volume device
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_capture_vol_entry" );
  Strlcpy( hermes2_rc.capture_vol,
      gtk_entry_get_text(GTK_ENTRY(widget)), sizeof(hermes2_rc.capture_vol) );

  // Get the ALSA Capture Channel
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_cap_chn_combobox" );
  hermes2_rc.capture_chan = gtk_combo_box_get_active(GTK_COMBO_BOX(widget) );

  /* Set right or left channel buffer index */
  if( (hermes2_rc.capture_chan == SND_MIXER_SCHN_FRONT_LEFT) ||
      (hermes2_rc.capture_chan == SND_MIXER_SCHN_REAR_LEFT)  ||
      (hermes2_rc.capture_chan == SND_MIXER_SCHN_MONO) )
    hermes2_rc.use_snd_chan = 0;
  else if( (hermes2_rc.capture_chan == SND_MIXER_SCHN_FRONT_RIGHT) ||
           (hermes2_rc.capture_chan == SND_MIXER_SCHN_REAR_RIGHT) )
    hermes2_rc.use_snd_chan = 1;

  // Get the Use Local PC Sound flag
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_pc_sound_checkbutton" );
  if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) )
    hermes2_rc.sound_pc_local = True;
  else
    hermes2_rc.sound_pc_local = False;

  // Get the DDC buffer size combobox index
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ddc_buff_combobox" );
  hermes2_rc.ddc_buffer_idx = gtk_combo_box_get_active( GTK_COMBO_BOX(widget) );

  // Get the Beep Volume level
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_beep_volume_spinbutton" );
  hermes2_rc.beep_volume = (uint16_t)atoi( gtk_entry_get_text(GTK_ENTRY(widget)) );

  // Get the Sound Volume value
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_volume_spinbutton" );
  TRx->snd_vol_scale = (uint8_t)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(widget) );

  // Get the Rx Frequency Band Index
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_rx_band_combobox" );
  TRx->rx_bands_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget) );

  // Get the default DDC sample rate index
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ddc_rate_combobox" );
  hermes2_rc.sample_rate_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget) );

  // Get the default FFT frame rate
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_fft_rate_combobox" );
  TRx->fft_rate_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget) );

  // Get the Tx Frequency Band Index
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_band_combobox" );
  TRx->tx_bands_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget) );

  // Get the TX Tune Power %
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_tune_power_spinbutton" );
  hermes2_rc.tx_tune_power = gtk_spin_button_get_value( GTK_SPIN_BUTTON(widget) );
  hermes2_rc.tx_tune_power = MAX_PA_POWER * hermes2_rc.tx_tune_power / 100.0;

  // Get the PPT Hang Time in mSec
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ptt_hang_time_spinbutton" );
  hermes2_rc.ptt_hang_time = (uint8_t)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(widget) );

  // Get the CW Hang Time in mSec
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_cw_hang_time_spinbutton" );
  hermes2_rc.cw_hang_time = (uint16_t)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(widget) );

  // Get the TX Buffer Latency in mSec
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_buf_latency_spinbutton" );
  hermes2_rc.tx_buf_latency = (uint8_t)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(widget) );

  // Get the RF Delay spinbutton value in mS
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_rf_delay_spinbutton" );
  hermes2_rc.rf_delay = (uint8_t)gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON(widget) );

  // Save default config to file
  if( !Save_Hermes2_Config(FROM_CONFIG) )
  {
    fprintf( stderr, _("!! hermes2: Failed to save default config file\n") );
    Error_Dialog( _("Failed to save default config file"), HIDE_OK );
    return;
  }

  // Close Transceiver windows before reading config
  int8_t cnt = (int8_t)Indices.Num_of_TRXs;
  while( cnt )
  {
    cnt--;
    gtk_widget_destroy( Transceiver[cnt]->tcvr_window );
    while( g_main_context_iteration(NULL, FALSE) );
  }

  // Read configuration file
  g_idle_add( Read_Config, NULL );

} // Create_Hermes2_Config()

//------------------------------------------------------------------------

/* Restore_TCVR_GUI()
 *
 * Restores the state of the Transceiver Window widget
 */
  void
Restore_TCVR_GUI( void )
{
  // Restore Settings of Transceiver Objects
  for( uint8_t idx = 0; idx < Indices.Num_of_TRXs; idx++ )
  {
    // Set active Transceiver object
    Transceiver_t *TRx = Transceiver[idx];
    GtkWidget *widget;
    Indices.TRx_Index = idx;

    /* Block the multiple frequency changes that
    * take place during restoration of the GUI state */
    TRx->block_freq_change = True;

    // Temporarily disable Rx->Tx flag as it interfers with Tx frequency
    BOOLEAN temp = TRx->link_tx_rx_freq;
    TRx->link_tx_rx_freq = False;

    // Set Transmitter Bands combobox
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->tx_bands_combobox), TRx->tx_bands_idx );

    // Set TX Power Limit
    gtk_range_set_value( GTK_RANGE(TRx->tx_power_hscale), hermes2_rc.tx_power_limit );

    // Set Microphone Level
    gtk_range_set_value( GTK_RANGE(TRx->mic_hscale), TRx->mic_level );

    // Set status of Compressor Enable check button
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->comp_checkbutton), TRx->mic_compress );

    // Set status of Mic LP Filter Enable check button
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->lpf_checkbutton), TRx->mic_filter );

    // Set Receiver Bands combobox
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->rx_bands_combobox), TRx->rx_bands_idx );

    // Set FFT Bandwidth combo box
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->fft_bw_combobox), TRx->fft_bw_idx );

    // Set FFT Frame Rate combo box
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->fft_rate_combobox), TRx->fft_rate_idx );

    // Set the Sound Volume Scale
    widget = Builder_Get_Object( TRx->builder, "volume_hscale" );
    gtk_range_set_value( GTK_RANGE(widget), TRx->snd_vol_scale );

    // Set the ADAGC Decay Scale
    widget = Builder_Get_Object( TRx->builder, "agc_hscale" );
    gtk_range_set_value( GTK_RANGE(widget), TRx->agc_decay_scale );

    // Set the Squelch Scale
    widget = Builder_Get_Object( TRx->builder, "squelch_hscale" );
    gtk_range_set_value( GTK_RANGE(widget), TRx->squelch_scale );

    // Set status of Squelch Enable check button
    widget = Builder_Get_Object( TRx->builder, "squelch_checkbutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->squelch_on_status );

    // Set status of Rx Zero Right check button
    widget = Builder_Get_Object( TRx->builder, "rx_zero_checkbutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->rx_zero_right );

    // Set status of Tx Zero Right check button
    widget = Builder_Get_Object( TRx->builder, "tx_zero_checkbutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->tx_zero_right );

    // Set and display status of Rx Mute on Tx Enable check button
    widget = Builder_Get_Object( TRx->builder, "mute_on_tx_checkbutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->mute_receiver );

    // Set and display status of RxID Enable check button
    widget = Builder_Get_Object( TRx->builder, "rx_rsid_enable" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->rx_rsid_enable );
    g_idle_add( RSID_Status, NULL );

    // Restore flag and set Link Tx to Rx Enable toggle button
    TRx->link_tx_rx_freq = temp;
    widget = Builder_Get_Object( TRx->builder, "to_tx_togglebutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->link_tx_rx_freq );

    // Set status of TxID check button
    widget = Builder_Get_Object( TRx->builder, "tx_rsid_enable" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), TRx->tx_rsid_enable );

    // Set the position of Transceiver windows
    Set_Window_Geometry(
        TRx->tcvr_window,
        TRx->tcvr_window_x,
        TRx->tcvr_window_y,
        0, 0 );

    // Wait for the GTK to complete its tasks
    while( g_main_context_iteration(NULL, FALSE) );

    // Release frequency change block and set Tx/Rx frequency and filter settings
    TRx->block_freq_change = False;
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );  // Tx
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );  // Rx

  } // for( int idx = 0; idx < Num_of_TRXs; idx++ )

} // Restore_TCVR_GUI()

//------------------------------------------------------------------------

#define RX_ANTENNA1     1
#define RX_ANTENNA2     2

/* Restore_Hermes2_GUI()
 *
 * Restores the state of the Hermes2 Window including window geometry
 */
  void
Restore_Hermes2_GUI( void )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Set status of the Duplex check button
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(hermes2_gui.tx_rx_duplex), ddv->duplex );

  // Set the sample rate combobox
  gtk_combo_box_set_active(
      GTK_COMBO_BOX(hermes2_gui.sample_rate_combobox), hermes2_rc.sample_rate_idx );

  // Set status of LNA Gain Auto check button
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(hermes2_gui.lna_gain_auto), ddv->lna_gain_auto );

   // Set status of LNA Gain Hardware Controlled check button
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(hermes2_gui.lna_gain_hw), ddv->lna_gain_hw );

  // Set LNA Gain spinbutton value
  gtk_spin_button_set_value(
      GTK_SPIN_BUTTON(hermes2_gui.lna_gain_spinbutton), (gdouble)ddv->lna_gain );

  // Set status of PA Enable check button
  gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON(hermes2_gui.tx_pa_enable), ddv->tx_pa_enable );

  // Set the Rx antenna radio buttons
  if( hermes2_rc.rx_antenna == RX_ANTENNA1 )
  {
    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON(hermes2_gui.rx_antenna1), TRUE);
  }
  else if( hermes2_rc.rx_antenna == RX_ANTENNA2 )
  {
    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON(hermes2_gui.rx_antenna2), TRUE);
  }

  // Restore main hermes2 window geometry
  Set_Window_Geometry(
      hermes2_gui.window,
      hermes2_rc.hermes2_window_x,
      hermes2_rc.hermes2_window_y,
      0, 0 );

  // Open Bookmarks Window if it was open at exit
  if( Indices.Num_of_TRXs )
  {
    if( hermes2_rc.bookmarks_window_width && hermes2_rc.bookmarks_window_height )
      Bookmarks_Button_Clicked( NULL );
  }

} // Restore_Hermes2_GUI()

//------------------------------------------------------------------------

// Object ID's for GtkBuilder
#define HERMES2_WINDOW_IDS \
  "hermes2_window", \
  "gain_adjustment", \
  "mox_togglebutton", \
  NULL

//----------------------------------------------------------------------

/* Read_Config()
 *
 * Loads the hermes2.conf configuration file
 */
  gboolean
Read_Config( gpointer data )
{
  char
    rc_fpath[FILE_NAME_SIZE],   // File path to hermes2 rc config file
    line[READ_LINE_BUF_SIZE];   // Buffer for Read_Line()

  int ret = SUCCESS, opn;

  // Config and mnemonics file pointer
  FILE *fp = NULL;

  discovered_device_t *ddv = &Device[hermes2_rc.device_index];


  // Setup file path to config file
  Strlcpy( rc_fpath, getenv("HOME"), sizeof(rc_fpath) );
  Strlcat( rc_fpath, HERMES2_CONFIG_FILE, sizeof(rc_fpath) );
  Flag[HERMES2_RCCONFIG_SETUP] = False;

  // Open hermes2 config file.
  opn = Open_File( &fp, rc_fpath, "r" );

  // If hermes2.conf file failed to open
  if( opn != SUCCESS )
  {
    Flag[HERMES2_CREATE_CONFIG] = True;
    Message_Dialog(
        _("Failed to open hermes2.conf file.\n"\
          "Create a new configuration file for hermes2?") );
    return( FALSE );
  }

  // while() is used so that many break points on error can be implemented
  while( True )
  {
    // Read and bypass Config file version
    ret = Read_Line( line, fp, _("hermes2.config file version") );
    if( ret == ERROR ) break;

    // Read CW Hang time, default 512 mS
    ret = Read_Line( line, fp, _("CW Hang time") );
    if( ret == ERROR ) break;
    hermes2_rc.cw_hang_time = (uint16_t)atoi( line );

    // PTT Hang time, default 12mS
    ret = Read_Line( line, fp, _("PTT Hang time") );
    if( ret == ERROR ) break;
    hermes2_rc.ptt_hang_time = (uint8_t)atoi( line );

    // TX Buffer latency, default 20mS
    ret = Read_Line( line, fp, _("TX Buffer Latency") );
    if( ret == ERROR ) break;
    hermes2_rc.tx_buf_latency = (uint8_t)atoi( line );

    // Read Tx Tune Power Setting
    ret = Read_Line( line, fp, _("Tx Tune Power Setting") );
    if( ret == ERROR ) break;
    hermes2_rc.tx_tune_power = atof( line ) * MAX_PA_POWER / 100.0;

    // Read Sound Enable PC Local Flag
    ret = Read_Line( line, fp, _("Enable Local (PC) Sound") );
    if( ret == ERROR ) break;
    hermes2_rc.sound_pc_local = (BOOLEAN)atoi( line );

    // Read ALSA Sound HWCTL device name, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound HWCTL Device") );
    if( ret == ERROR ) break;
    Strlcpy( hermes2_rc.hwctl_device, line, sizeof(hermes2_rc.pcm_device) );

    // Read ALSA Sound PCM device name, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound PCM Device") );
    if( ret == ERROR ) break;
    Strlcpy( hermes2_rc.pcm_device, line, sizeof(hermes2_rc.pcm_device) );

    // Read ALSA Sound Capture Source, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound Capture Source") );
    if( ret == ERROR ) break;
    Strlcpy( hermes2_rc.capture_src, line, sizeof(hermes2_rc.capture_src) );

    // Read ALSA Sound Capture Volume control, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound Capture Volume control") );
    if( ret == ERROR ) break;
    Strlcpy( hermes2_rc.capture_vol, line, sizeof(hermes2_rc.capture_vol) );

    // Read ALSA Sound Capture Channel ID, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound Capture Channel ID") );
    if( ret == ERROR ) break;
    hermes2_rc.capture_chan = atoi( line );

    // Read ALSA Sound Capture Level, abort if EOF
    ret = Read_Line( line, fp, _("ALSA Sound Capture Level") );
    if( ret == ERROR ) break;
    hermes2_rc.capture_level = atoi( line );

    // Read Beep volume Setting
    ret = Read_Line( line, fp, _("Beep Volume Setting") );
    if( ret == ERROR ) break;
    hermes2_rc.beep_volume = (uint16_t)atoi( line );

    // Read Hermes2 DDC I/Q Data Buffer index
    ret = Read_Line( line, fp, _("DDC Buffer Size index") );
    if( ret == ERROR ) break;
    hermes2_rc.ddc_buffer_idx = atoi( line );
    if( hermes2_rc.ddc_buffer_idx >= HERMES2_BUF_SIZE_NUM )
      break;
    const char *buf[] = { HERMES2_BUF_SIZE };
    hermes2_rc.ddc_buffer_length = (uint32_t)( atoi(buf[hermes2_rc.ddc_buffer_idx]) * 1024 );

    // Read Device Frequency Correction PPM, abort if EOF
    ret = Read_Line( line, fp, _("Device Frequency Correction") );
    if( ret == ERROR ) break;
    hermes2_rc.device_freq_error = atof( line );

    // Read main window position
    ret = Read_Line( line, fp, _("Hermes2 Window position") );
    if( ret == ERROR ) break;
    uint8_t idx = 0;
    hermes2_rc.hermes2_window_x = (gint)atoi( line );
    do idx++;
    while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
    hermes2_rc.hermes2_window_y = (gint)atoi( &line[idx + 1] );

    // Read Bookmarks window size
    ret = Read_Line( line, fp, _("Bookmarks Window Size") );
    if( ret == ERROR ) break;
    idx = 0;
    hermes2_rc.bookmarks_window_width = (gint)atoi( line );
    do idx++;
    while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
    hermes2_rc.bookmarks_window_height = (gint)atoi( &line[idx + 1] );

    // Read Bookmarks window position
    ret = Read_Line( line, fp, _("Bookmarks Window Position") );
    if( ret == ERROR ) break;
    idx = 0;
    hermes2_rc.bookmarks_window_x = (gint)atoi( line );
    do idx++;
    while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
    hermes2_rc.bookmarks_window_y = (gint)atoi( &line[idx + 1] );

    // Read Time Stations window position
    ret = Read_Line( line, fp, _("Time Window Position") );
    if( ret == ERROR ) break;
    idx = 0;
    hermes2_rc.time_window_x = (gint)atoi( line );
    do idx++;
    while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
    hermes2_rc.time_window_y = (gint)atoi( &line[idx + 1] );

    // Read Rx/Tx Duplex setting
    ret = Read_Line( line, fp, _("Rx/Tx Duplex setting") );
    if( ret == ERROR ) break;
    ddv->duplex = (BOOLEAN)atoi( line );

    // Read DDC Sample Rate Index
    ret = Read_Line( line, fp, _("Rx Sample Rate Index") );
    if( ret == ERROR ) break;
    hermes2_rc.sample_rate_idx = atoi( line );

    // Read LNA Gain Auto Status
    ret = Read_Line( line, fp, _("LNA Gain Auto Status") );
    if( ret == ERROR ) break;
    ddv->lna_gain_auto = (BOOLEAN)atoi( line );

    // Read LNA Gain Hardware Control Status
    ret = Read_Line( line, fp, _("LNA Gain H/W Controlled Status") );
    if( ret == ERROR ) break;
    ddv->lna_gain_hw = (BOOLEAN)atoi( line );

    // Read ADC LNA Gain Setting
    ret = Read_Line( line, fp, _("LNA Gain Setting") );
    if( ret == ERROR ) break;
    ddv->lna_gain = (int8_t)atoi( line );

    // Read Receiver Antenna in use
    ret = Read_Line( line, fp, _("Receiver Antenna") );
    if( ret == ERROR ) break;
    hermes2_rc.rx_antenna = (uint8_t)atoi( line );

    // Read RF Delay setting
    ret = Read_Line( line, fp, _("Tx RF Delay") );
    if( ret == ERROR ) break;
    hermes2_rc.rf_delay  = (uint8_t)atoi( line );

    // Read TX PA Enable setting
    ret = Read_Line( line, fp, _(" TX PA Enable Setting") );
    if( ret == ERROR ) break;
    ddv->tx_pa_enable = (BOOLEAN)atoi( line );

    // Read Number of Receiver slices
    ret = Read_Line( line, fp, _("Number of Receiver Slices") );
    if( ret == ERROR ) break;
    uint8_t num_rx = (uint8_t)atoi( line );

    // Create the main window builder
    gchar *object_ids[] = { HERMES2_WINDOW_IDS };
    Gtk_Builder( &hermes2_gui.window_builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );

    // Read Window Settings for all Transceiver instances ("Receiver slices")
    for( uint8_t nr = 0; nr < num_rx; nr++ )
    {
      // Create a Transceiver object on the ADC (ADC=0, only one in HL2)
      ADC_Add_Transceiver();
      Transceiver_t *TRx = Transceiver[nr];

      // Read Receiver Bands Combobox Index
      ret = Read_Line( line, fp, _("Receiver Bands Combobox Index") );
      if( ret == ERROR ) break;
      TRx->rx_bands_idx = (gint)atoi( line );

      // Read FFT Bandwidth Index
      ret = Read_Line( line, fp, _("FFT Bandwidth Index") );
      if( ret == ERROR ) break;
      TRx->fft_bw_idx = atoi( line );

      // Read FFT Frame Rate Index
      ret = Read_Line( line, fp, _("FFT Frame Rate Index") );
      if( ret == ERROR ) break;
      TRx->fft_rate_idx = atoi( line );

      // Read Sound Volume Scale Value
      ret = Read_Line( line, fp, _("Sound Volume Scale Value") );
      if( ret == ERROR ) break;
      TRx->snd_vol_scale = (uint8_t)atoi( line );

      // Read ADAGC Decay Scale Value
      ret = Read_Line( line, fp, _("ADAGC Decay Scale Value") );
      if( ret == ERROR ) break;
      TRx->agc_decay_scale = (uint8_t)atoi( line );
      Transceiver[Indices.TRx_Index]->adagc_decay = 1.0 - TRx->agc_decay_scale / ADAGC_DIVISOR;

      // Read Squelch Scale value
      ret = Read_Line( line, fp, _("Squelch Scale Value") );
      if( ret == ERROR ) break;
      TRx->squelch_scale = (uint8_t)atoi( line );

      // Read Squelch Enable check button status
      ret = Read_Line( line, fp, _("Squelch Check Button") );
      if( ret == ERROR ) break;
      TRx->squelch_on_status = (BOOLEAN)atoi( line );

      // Read Rx Zero Right check button status
      ret = Read_Line( line, fp, _("Rx Zero Right Check Button") );
      if( ret == ERROR ) break;
      TRx->rx_zero_right = (BOOLEAN)atoi( line );

      // Read Rx Mute on Tx Enable check button status
      ret = Read_Line( line, fp, _("Rx Mute on Tx Check Button") );
      if( ret == ERROR ) break;
      TRx->mute_receiver = (BOOLEAN)atoi( line );

      // Read Rx RSID Enable check button status
      ret = Read_Line( line, fp, _("RxID Enable Check Button") );
      if( ret == ERROR ) break;
      TRx->rx_rsid_enable = (BOOLEAN)atoi( line );

      // Read Link Tx to Rx Enable toggle button status
      ret = Read_Line( line, fp, _("Link Tx to Rx Toggle Button") );
      if( ret == ERROR ) break;
      TRx->link_tx_rx_freq = (BOOLEAN)atoi( line );

      // Read Transmitter Frequency Bands index
      ret = Read_Line( line, fp, _("Tx Frequency Bands Combobox Index") );
      if( ret == ERROR ) break;
      TRx->tx_bands_idx = (gint)atoi( line );

      // Read Tx Zero Right check button status
      ret = Read_Line( line, fp, _("Tx Zero Right Check Button") );
      if( ret == ERROR ) break;
      TRx->tx_zero_right = (BOOLEAN)atoi( line );

      // Read Tx Power OP Limit value
      ret = Read_Line( line, fp, _("Tx Power O/P Limit") );
      if( ret == ERROR ) break;
      hermes2_rc.tx_power_limit = atof( line );

      // Read Tx Microphone Level value
      ret = Read_Line( line, fp, _("Tx Microphone Level") );
      if( ret == ERROR ) break;
      TRx->mic_level = (uint8_t)atoi( line );

      // Read Tx Mic Compressor checkbutton
      ret = Read_Line( line, fp, _("Tx Mic Compress Enable") );
      if( ret == ERROR ) break;
      TRx->mic_compress = (uint8_t)atoi( line );

      // Read Tx Mic Low Pass Filter checkbutton
      ret = Read_Line( line, fp, _("Tx Microphone LP Filter") );
      if( ret == ERROR ) break;
      TRx->mic_filter = (uint8_t)atoi( line );

      // Read Tx RSID Enable checkbutton
      ret = Read_Line( line, fp, _("TxID Enable Check Button") );
      if( ret == ERROR ) break;
      TRx->tx_rsid_enable = (uint8_t)atoi( line );

      // Read Transceiver's window position
      ret = Read_Line( line, fp, _("Transceiver Window Position") );
      if( ret == ERROR ) break;
      TRx->tcvr_window_x = (gint)atoi( line );
      idx = 0;
      do idx++;
      while( (idx < READ_LINE_BUF_SIZE - 2) && (line[idx] != ',') );
      TRx->tcvr_window_y = (gint)atoi( &line[idx + 1] );

      // Read Rx and Tx Band Frequencies
      for( uint8_t bnd = 0; bnd < NUMBER_OF_BANDS; bnd++ )
      {
        char *startptr, *endptr;

        ret = Read_Line( line, fp, _("Rx & Tx Band Frequencies and Modes") );
        if( (ret == ERROR) || (ret == EOF) ) break;

        // Bypass leading spaces and read Rx frequency
        startptr = line;
        hermes2_rc.rx_bands[bnd] = (uint32_t)strtol( startptr, &endptr, 10 );

        // Got to next space and read Rx mode
        startptr = endptr;
        hermes2_rc.rx_modes[bnd] = (uint8_t)strtol( startptr, &endptr, 10 );

        // Go to next space and read Tx frequency
        startptr = endptr;
        hermes2_rc.tx_bands[bnd] = (uint32_t)strtol( startptr, &endptr, 10 );

        // Got to next space and read Tx mode
        startptr = endptr;
        hermes2_rc.tx_modes[bnd] = (uint8_t)strtol( startptr, &endptr, 10 );
      } // for( uint8_t bnd = 0; bnd < NUMBER_OF_BANDS; bnd++ )

    } //for( uint8_t nr = 0; nr < num_rx; nr++ )

    break;
  } // while( True )

  // Failed reading parameters in config file
  if( ret == ERROR )
  {
    Flag[HERMES2_CREATE_CONFIG] = True;
    Message_Dialog(
        _("Error reading a parameter in hermes2.conf.\n"\
          "Create a new configuration file for hermes2?") );
    return( FALSE );
  }
  else Close_File( &fp );

  // Initialize HERMES2 functions
  if( !Hpsdr_Initialize() ) return( FALSE );

  // Create Main Window
  Hermes2_Window();

  // Set Device combobox to selected device index
  gtk_combo_box_set_active(
      GTK_COMBO_BOX(hermes2_gui.device_combobox), (gint)hermes2_rc.device_index );

  // Set CW hang time
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x10][1] = (uint8_t)( hermes2_rc.cw_hang_time >> 2 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x10][2] = (uint8_t)( hermes2_rc.cw_hang_time & 0x0003 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x10][1] = (uint8_t)( hermes2_rc.cw_hang_time >> 2 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x10][2] = (uint8_t)( hermes2_rc.cw_hang_time & 0x0003 );
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x10;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x10;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  // Set PTT hang time and TX buffer latency
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x17][3] = hermes2_rc.ptt_hang_time;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x17][4] = hermes2_rc.tx_buf_latency;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x17][3] = hermes2_rc.ptt_hang_time;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x17][4] = hermes2_rc.tx_buf_latency;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x17;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x17;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  // Enable further actions
  Flag[HERMES2_RCCONFIG_SETUP] = True;

  // Give the GUI time to map and stabilize
  //alarm( 1 );

  Restore_Hermes2_GUI();
  Restore_TCVR_GUI();

  return( FALSE );
} // Read_Config()

//------------------------------------------------------------------------

/* Get_GUI_Geometry()
 *
 * Gets the GUI Geometry to store in the hermes2_rc buffer
 */
  void
Get_GUI_Geometry( void )
{
  // Get geometry of hermes2 window
  Get_Window_Geometry(
      hermes2_gui.window,
      &(hermes2_rc.hermes2_window_x),
      &(hermes2_rc.hermes2_window_y),
      NULL, NULL );

  // Get geometry of Bookmarks window
  if( hermes2_gui.bmk_window )
  {
    Get_Window_Geometry(
        hermes2_gui.bmk_window,
        &(hermes2_rc.bookmarks_window_x),
        &(hermes2_rc.bookmarks_window_y),
        &(hermes2_rc.bookmarks_window_width),
        &(hermes2_rc.bookmarks_window_height) );
  }
  else
  {
    hermes2_rc.bookmarks_window_width  = 0;
    hermes2_rc.bookmarks_window_height = 0;
  }

  // Get geometry of time window
  if( time_window )
    Get_Window_Geometry(
        time_window,
        &(hermes2_rc.time_window_x),
        &(hermes2_rc.time_window_y),
        NULL, NULL );

  // Get window geometry of any Transceiver objects
  for( int idx = 0; idx < Indices.Num_of_TRXs; idx++ )
  {
    Get_Window_Geometry(
        Transceiver[idx]->tcvr_window,
        &(Transceiver[idx]->tcvr_window_x),
        &(Transceiver[idx]->tcvr_window_y),
        NULL, NULL );
  }

} // Get_GUI_State()

//------------------------------------------------------------------------

/* Save_Hermes2_Config()
 *
 * Saves the current hermes2 state to config file. Argument indicates
 * whether the function is being called from the Configuration Dialog
 */
  BOOLEAN
Save_Hermes2_Config( BOOLEAN from_config )
{
  FILE *fp = NULL;  // File pointer to write config file
  int ret;
  char rc_fpath[FILE_NAME_SIZE]; // Path to config file
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];


  // Setup file path to hermes2 config file
  Strlcpy( rc_fpath, getenv("HOME"), sizeof(rc_fpath) );
  Strlcat( rc_fpath, HERMES2_CONFIG_FILE, sizeof(rc_fpath) );

  // Open config file for writing
  if( Open_File(&fp, rc_fpath, "w") == FAILURE )
  {
    char err_str[MESG_STRING_SIZE]; // Error messages string
    snprintf( err_str, sizeof(err_str), "hermes2: %s", rc_fpath );
    perror( err_str );
    fprintf( stderr, _("hermes2: cannot open hermes2's config file: %s\n"), rc_fpath );
    Error_Dialog( _("Cannot open hermes2's config file"), HIDE_OK );
    return( False );
  }

  // Print hermes2 configuration values to config file
  ret = fprintf( fp,
      HERMES2_CONFIG_FORMAT,
      CONFIG_VERSION,
      hermes2_rc.cw_hang_time,
      hermes2_rc.ptt_hang_time,
      hermes2_rc.tx_buf_latency,
      hermes2_rc.tx_tune_power / MAX_PA_POWER * 100.0,
      hermes2_rc.sound_pc_local,
      hermes2_rc.hwctl_device,
      hermes2_rc.pcm_device,
      hermes2_rc.capture_src,
      hermes2_rc.capture_vol,
      hermes2_rc.capture_chan,
      hermes2_rc.capture_level,
      hermes2_rc.beep_volume,
      hermes2_rc.ddc_buffer_idx,
      hermes2_rc.device_freq_error,
      hermes2_rc.hermes2_window_x,
      hermes2_rc.hermes2_window_y,
      hermes2_rc.bookmarks_window_width,
      hermes2_rc.bookmarks_window_height,
      hermes2_rc.bookmarks_window_x,
      hermes2_rc.bookmarks_window_y,
      hermes2_rc.time_window_x,
      hermes2_rc.time_window_y,
      ddv->duplex,
      hermes2_rc.sample_rate_idx,
      ddv->lna_gain_auto,
      ddv->lna_gain_hw,
      ddv->lna_gain,
      hermes2_rc.rx_antenna,
      hermes2_rc.rf_delay,
      ddv->tx_pa_enable,
      hermes2_rc.num_of_receivers );

  if( ret < 0 )
  {
    Fprintf_Error( fp, rc_fpath );
    return( False );
  }

  // Print window positions and settings for any Transceiver objects
  for( uint8_t tcvr_idx = 0; tcvr_idx < Indices.Num_of_TRXs; tcvr_idx++ )
  {
    GtkWidget *widget;
    Transceiver_t *TRx = Transceiver[tcvr_idx];

    // If the function is called from the Config dialog, don't change these settings
    if( !from_config )
    {
      // Get indices to FFT settings combobox
      TRx->fft_bw_idx   = gtk_combo_box_get_active( GTK_COMBO_BOX(TRx->fft_bw_combobox) );
      TRx->fft_rate_idx = gtk_combo_box_get_active( GTK_COMBO_BOX(TRx->fft_rate_combobox) );

      // Get index of Receiver Bands combobox
      TRx->rx_bands_idx = gtk_combo_box_get_active( GTK_COMBO_BOX(TRx->rx_bands_combobox) );

      // Get index of Transmitter Bands combobox
      TRx->tx_bands_idx = gtk_combo_box_get_active( GTK_COMBO_BOX(TRx->tx_bands_combobox) );

      // Get value of Volume slider
      widget = Builder_Get_Object( TRx->builder, "volume_hscale" );
      TRx->snd_vol_scale = (uint8_t)( gtk_range_get_value(GTK_RANGE(widget)) );
    } // if( !from_config )

    // Get values of ADAGC and Squelch sliders
    widget = Builder_Get_Object( TRx->builder, "agc_hscale" );
    TRx->agc_decay_scale = (uint8_t)( gtk_range_get_value(GTK_RANGE(widget)) );
    widget = Builder_Get_Object( TRx->builder, "squelch_hscale" );
    TRx->squelch_scale = (uint8_t)( gtk_range_get_value(GTK_RANGE(widget)) );

    // Get status of Squelch Enable check button
    widget = Builder_Get_Object( TRx->builder, "squelch_checkbutton" );
    TRx->squelch_on_status = (BOOLEAN)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );

    // Get status of Rx Zero Right check button
    widget = Builder_Get_Object( TRx->builder, "rx_zero_checkbutton" );
    TRx->rx_zero_right = (BOOLEAN)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );

    // Get status of Tx Zero Right check button
    widget = Builder_Get_Object( TRx->builder, "tx_zero_checkbutton" );
    TRx->tx_zero_right = (BOOLEAN)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );

    // Get value of Tx Power Limit slider
    hermes2_rc.tx_power_limit = gtk_range_get_value( GTK_RANGE(TRx->tx_power_hscale) );

    // Get Rx and Tx RSID enable status
    widget = Builder_Get_Object( TRx->builder, "rx_rsid_enable" );
    TRx->rx_rsid_enable = (BOOLEAN)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );
    widget = Builder_Get_Object( TRx->builder, "tx_rsid_enable" );
    TRx->tx_rsid_enable = (BOOLEAN)gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) );

    // Print Rx GUI state to config file
    ret = fprintf( fp,
        RECEIVER_SETTINGS_FORMAT,
        tcvr_idx,
        TRx->rx_bands_idx,
        TRx->fft_bw_idx,
        TRx->fft_rate_idx,
        TRx->snd_vol_scale,
        TRx->agc_decay_scale,
        TRx->squelch_scale,
        TRx->squelch_on_status,
        TRx->rx_zero_right,
        TRx->mute_receiver,
        TRx->rx_rsid_enable,
        TRx->link_tx_rx_freq );

    if( ret < 0 )
    {
      Fprintf_Error( fp, rc_fpath );
      return( False );
    }

    // Print Tx GUI state to config file
    ret = fprintf( fp,
        TRANSMITTER_SETTINGS_FORMAT,
        TRx->tx_bands_idx,
        TRx->tx_zero_right,
        hermes2_rc.tx_power_limit,
        TRx->mic_level,
        TRx->mic_compress,
        TRx->mic_filter,
        TRx->tx_rsid_enable,
        TRx->tcvr_window_x,
        TRx->tcvr_window_y );

    if( ret < 0 )
    {
      Fprintf_Error( fp, rc_fpath );
      return( False );
    }

    // Print Bands data to hermes2 config file
    ret = fprintf( fp, "# Rx Band Rx Mode Tx Band Tx Mode\n" );
    if( ret < 0 )
    {
      Fprintf_Error( fp, rc_fpath );
      return( False );
    }
    for( uint8_t idx = 0; idx < NUMBER_OF_BANDS; idx++ )
    {
      ret = fprintf( fp,
          BANDS_CONFIG_FORMAT,
          hermes2_rc.rx_bands[idx], hermes2_rc.rx_modes[idx],
          hermes2_rc.tx_bands[idx], hermes2_rc.tx_modes[idx] );
      if( ret < 0 )
      {
        Fprintf_Error( fp, rc_fpath );
        return( False );
      }
    }
  } // for( uint8_t tcvr_idx = 0; tcvr_idx < Indices.Num_of_TRXs; tcvr_idx++ )

  // Terminate file
  ret = fprintf( fp, "#\n[END]\n#" );
  if( ret < 0 )
  {
    Fprintf_Error( fp, rc_fpath );
    return( False );
  }
  Close_File( &fp );

  // Wait for receive thread to finish
  ddv->rcve_thread_run = False;
  if( ddv->rcve_thread_running )
    pthread_join( ddv->data_receive_thread, NULL );

  return( True );
} // Save_Hermes2_Config()

//------------------------------------------------------------------------

/* Save_Default_Config()
 *
 * Saves a default starting configuration to config file
 */
  BOOLEAN
Save_Default_Config( void )
{
  FILE *fp = NULL;  // File pointer to write config file
  char rc_fpath[FILE_NAME_SIZE]; // Path to config file

  // Setup file path to hermes2 config file
  Strlcpy( rc_fpath, getenv("HOME"), sizeof(rc_fpath) );
  Strlcat( rc_fpath, HERMES2_CONFIG_FILE, sizeof(rc_fpath) );

  // Open config file for writing
  if( Open_File(&fp, rc_fpath, "w") == FAILURE )
  {
    char err_str[MESG_STRING_SIZE]; // Error messages string
    snprintf( err_str, sizeof(err_str), "hermes2: %s", rc_fpath );
    perror( err_str );
    fprintf( stderr, _("hermes2: cannot open hermes2's config file: %s\n"), rc_fpath );
    Error_Dialog( _("Cannot open hermes2's config file"), HIDE_OK );
    return( False );
  }

  // Print a default config to file
  if( fprintf(fp, "%s", DEFAULT_CONFIG) < 0 )
  {
    Fprintf_Error( fp, rc_fpath );
    return( False );
  }

  // Read configuration file
  Close_File( &fp );
  g_idle_add( Read_Config, NULL );

  return( True );
} // Save_Default_Config()

//------------------------------------------------------------------------

