/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef GUEST_UTILS_H
#define GUEST_UTILS_H   1

#include "common.h"
#include "ifft.h"
#include "shared.h"
#include "utils.h"
#include <gtk/gtk.h>
#include <stddef.h>
#include <stdio.h>
#include <stdint.h>

//----------------------------------------------------------------------

// Number and size of labels for commands
#define NUM_OF_LABELS   10
#define LABEL_LENGTH    25

// Keystroke buffer length
#define KEY_BUF_SIZE    256

#define RECORD_LENGTH    618  // Size in bytes of QSO Records in station log
#define REC_BUF_SIZE      619  // Size in bytes of QSO Records buffer (+'\0')
#define CARDS_PER_PAGE     6  // QSL cards per "page" in the postscipt file
#define LINE_LENGTH       77  // Limit of line length in text files

// For opening files
#define FILE_NEW    0x01
#define FILE_OPEN   0x02
#define FILE_FAIL   0x04

/* Waterfall columns either side of user click
 * point to search for a maximum bin FFT value
 */
#define CLICK_TUNE_RANGE   15

// Return values of guest signal detectors
#define ABORT    0
#define REPEAT   1
#define READY    2

//----------------------------------------------------------------------

// QSO record type definition
typedef struct _QSO_RECORD
{
  // QSO data
  char
    dx_call[15],  // Remote (dx) station callsign
    dx_rst[4],    // My RST report to Dx station
    my_rst[4],    // My RST report to Dx station
    dx_name[13],  // Remote station operator name
    dx_qth[13],   // Remote station QTH loc. name
    dx_loc[7],    // Remote station QTH locator
    date[12],     // Normal format QSO date
    date_adif[9], // QSO date in ADIF format
    time[6],      // Normal format QSO time
    time_adif[5], // QSO time in ADIF format
    freq[14],     // QSO operating frequency MHz
    mode[12],     // Mode of operation
    via[13];      // Via which path or relay Sat

  // Remarks on QSO
  char remarks[46];

  // The Mode's directory name in ~/.hermes2/
  char mode_dir[16];

  // Logging file pointers
  FILE
    *qso_record_fp,   // File for recording QSO's
    *adif_log_fp,     // File for ADIF QSO record
    *station_log_fp;  // File for station log

} qso_record_t;

extern qso_record_t qso_record;

//----------------------------------------------------------------------

// Keyboard buffer struct
typedef struct _KEYBD_BUF
{
  guint key_buf[KEY_BUF_SIZE];   // Buffer for characters typed on keyboard
  int16_t key_count;            // Counter of characters typed on keyboard
  int16_t char_pointer;         // Pointer to last character typed on keyboard
  int16_t char_count;           // Count of characters printed on text view
  int16_t space_count;          // Count of spaces typed on keyboard
} keybd_buf_t;

extern keybd_buf_t keybd_buf;

//----------------------------------------------------------------------

// Print Character data structure
typedef struct _PRINTCHR
{
  guint          printchr;     // Character to print on textview
  GtkTextBuffer *text_buffer;
  GtkTextView   *text_view;
  GtkTextIter   *iter;
  BOOLEAN        rx_mode;

} print_chr_t;

// Data for the Print_Character() function
extern print_chr_t tx_print_chr, rx_print_chr;

//----------------------------------------------------------------------

// Operator and QTH data
typedef struct _OP_DATA
{
  // Local operator data
  char
    call[15],
    name[13],
    zone[15],
    qth[13],
    loc[7];

  // Tx/Rx power, N.Fig, antennas
  char
    tx[12],
    tx_power[10],
    tx_ant[16],
    rx[12],
    rx_nfig[8],
    rx_ant[16];
} op_data_t;

// Common operator data struct
extern op_data_t op_data;

//----------------------------------------------------------------------

// Pixbuffer object structure for drawing mainly waterfall drawingareas
typedef struct _PIXBUFFER
{
  GtkWidget *canvas;
  GdkPixbuf *pixbuf;
  guchar    *pixels;

  uint16_t
    width,
    height,
    rowstride,
    n_channels;

} pixbuffer_t;

//----------------------------------------------------------------------

/* guest_utils.c */
void Queue_Character(print_chr_t *printchr);
void Process_RST(GtkEditable *editable);
void Locator_Changed(GtkEditable *editable, qso_record_t *qso_rec, const char *mode);
void Callsign_Changed(GtkEditable *editable, qso_record_t *qso_rec, const char *mode);
void Band_Changed(GtkEditable *editable, qso_record_t *qso_rec, const char *mode);
uint8_t Get_Record_Field(char *str, size_t siz, GtkEditable *editable);
void Clear_Record_Fields(uint8_t all, GtkBuilder *builder, const char *mode);
uint8_t Read_QSO_Record(qso_record_t *qso_rec, GtkBuilder *builder, const char *mode);
void New_Record(qso_record_t *qso_rec, GtkBuilder *builder, const char *mode);
uint8_t Save_QSO_Record(qso_record_t *qso_rec);
uint8_t Open_Record_File(const char *mode_dir);
double Atof(const char *nptr);
uint8_t File_Print(FILE **fp, const char *string);
uint8_t Locate_Line(char **line_idx, char *line, const char *string, FILE *fp);
uint8_t Locate_String(char **line_idx, char *line, const char *string);
uint8_t Find_Line(char *line, const char *str, FILE *fp);
void Find_String(const char *line, uint16_t *idx);
uint8_t Guest_Open_File(FILE **fpointer, const char *fpath, const char *mode);
uint8_t Create_Wfall_Pixbuf(pixbuffer_t *wfall, ifft_data_t *ifft_data, uint16_t width, uint16_t height);
void Guest_Quit_Activate(GtkWidget *window);
void Alloc_Xmit_Buffers(uint32_t buf_size);
void Realloc_Xmit_Buffers(uint32_t buf_size);
void Free_Xmit_Buffers(void);
void Tune_to_Monitor(double pos_x, uint8_t tune_range, uint16_t wfall_width, uint16_t center_freq, ifft_data_t *ifft_data);
void Display_Waterfall(pixbuffer_t *wfall, ifft_data_t *ifft_data);
gboolean Read_Station_Info(FILE *rcfile_ptr, char **macro, const char *label_name, GtkBuilder *builder);
void Save_Window_Position(GtkWidget *window, const char *fname);
void Get_Window_Geometry(GtkWidget *window, gint *x, gint *y, gint *width, gint *height);
void Set_Window_Geometry(GtkWidget *window, gint x, gint y, gint width, gint height);
void Gridloc_to_Position(const char *grid_locator, double *latitude, double *longitude);
void Bearing_Distance(double *bearing, double *distance, double home_lat, double home_long, double remote_lat, double remote_long);
void Launch_Xplanet(char *command);
void Kill_Xplanet(char *config_file, char *marker_file, char *arc_file);

//----------------------------------------------------------------------

#endif

