/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "guest_utils.h"
#include "common.h"
#include "filters.h"
#include "shared.h"
#include "utils.h"
#include "../hpsdr/settings.h"
#include "../Hermes2/interface.h"
#include "../rsid/rsid_modes.h"
#include <gtk/gtk.h>
#include <ctype.h>
#include <locale.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

// Keyboard buffer struct
keybd_buf_t keybd_buf;

// Data for the Print_Character()  function
print_chr_t tx_print_chr, rx_print_chr;

// QSO record struct
qso_record_t qso_record;

// Common operator data struct
op_data_t op_data;

//------------------------------------------------------------------------

// Length of char queue ring buffer
#define CHAR_QUEUE_SIZE   255
static print_chr_t printc[CHAR_QUEUE_SIZE];

// Counters of incoming and outgoing characters
static uint8_t input_cnt = 0, output_cnt = 0;

// Flag to indicate thread exited
static BOOLEAN queue_thread = False;

// Locks the thread that invokes the idle callback
static sem_t queue_sem;

// Unlocks the thread that invokes the idle callback
  static void
Unlock_Queue( gpointer data )
{
  int sval;
  sem_getvalue( &queue_sem, &sval );
  if( !sval ) sem_post( &queue_sem );
}

/*  _Print_Character()
 *
 *  Prints a character to a text view port
 */
  static gboolean
_Print_Character( gpointer data )
{
  gchar char2text[2];

  // Can happen if the client window is closed while there is more to print
  if( !GTK_IS_TEXT_BUFFER(printc[output_cnt].text_buffer) )
    return( FALSE );

  // Delete a char on backspace
  if( printc[output_cnt].printchr == GDK_KEY_BackSpace )
  {
    GtkTextIter start;
    gtk_text_buffer_get_iter_at_offset( printc[output_cnt].text_buffer, &start,
        gtk_text_buffer_get_char_count(printc[output_cnt].text_buffer) - 1 );
    gtk_text_buffer_delete(
        printc[output_cnt].text_buffer, &start, printc[output_cnt].iter );
    return( FALSE );
  }

  // Convert char to "text"
  char2text[0] = (gchar)( printc[output_cnt].printchr );
  char2text[1] = '\0';

  // Print character
  gtk_text_buffer_insert(
      printc[output_cnt].text_buffer, printc[output_cnt].iter, char2text, -1 );

  // Scroll Text View to bottom
  gtk_text_view_scroll_mark_onscreen(
      printc[output_cnt].text_view, gtk_text_buffer_create_mark(
        printc[output_cnt].text_buffer, "end", printc[output_cnt].iter, FALSE) );

  // Change Carriage Return to Line Feed
  if( (printc[output_cnt].printchr == CR) ||
      (printc[output_cnt].printchr == GDK_KEY_Return) )
    printc[output_cnt].printchr = LF;

  // Record to QSO record file
  if( Flag[GUEST_RECORD_QSO] && (qso_record.qso_record_fp != NULL) &&
      (Flag[GUEST_RECEIVING] || Flag[GUEST_TRANSMITTING]) )
  {
    fprintf( qso_record.qso_record_fp, "%c", printc[output_cnt].printchr );

    // Mark LF in record files
    if( printc[output_cnt].printchr == LF )
    {
      if( printc[output_cnt].rx_mode )
        fprintf( qso_record.qso_record_fp, "Rx-> " );
      else
        fprintf( qso_record.qso_record_fp, "Tx-> " );
    }
  }

  return( FALSE );
} // _Print_Character()

//------------------------------------------------------------------------

  static void *
Queue_Char_Thrd( void *data )
{
  // Keep calling idle add function till ring buffer processed
  while( True )
  {
    // Set up lock and wait for idle add to complete
    g_idle_add_full( G_PRIORITY_HIGH_IDLE, _Print_Character, NULL, Unlock_Queue );

    /* We need to wait for the g_idle_add_full() function
     * to complete before exiting this function. */
    while( g_main_context_iteration(NULL, FALSE) );

    // The wait is unlocked by Unlock_Queue() in g_idle_add_full() above
    sem_wait( &queue_sem );

    // Increment and reset ring index
    output_cnt++;
    if( output_cnt >= CHAR_QUEUE_SIZE ) output_cnt = 0;
    if( output_cnt == input_cnt ) break;

    // To avoid exiting thread while new calls are made
    usleep( 10000 );
  } // while( output_cnt != input_cnt )

  // Free semaphore and exit thread
  Init_Semaphore( &queue_sem, False );
  queue_thread = False;

  return( NULL );
} // Queue_Char_Thrd()

//------------------------------------------------------------------------

/* Forms a queue of incoming characters to be printed
 * on Rx or Tx text views and invokes the function above
 */
  void
Queue_Character( print_chr_t *printchr )
{
  // Save incoming character to circular buffer
  printc[input_cnt].printchr    = printchr->printchr;
  printc[input_cnt].text_buffer = printchr->text_buffer;
  printc[input_cnt].text_view   = printchr->text_view;
  printc[input_cnt].iter        = printchr->iter;
  printc[input_cnt].rx_mode     = printchr->rx_mode;
  input_cnt++;
  if( input_cnt >= CHAR_QUEUE_SIZE )
    input_cnt = 0;

  // Create thread if not already
  if( !queue_thread )
  {
    // Create semaphore to lock idle add thread
    queue_thread = True;
    Init_Semaphore( &queue_sem, True );

    // Start the idle add thread
    pthread_t thrd;
    if( !Pthread_Create(
          &thrd, NULL, Queue_Char_Thrd, NULL,
          "_Failed to create Queue_Char_Thrd() thread") )
    {
      queue_thread = False;
      return;
    }
  }

} // Queue_Character()

//------------------------------------------------------------------------

/*  Process_RST()
 *
 *  Processes RST-out and RST-in fields
 */
  void
Process_RST( GtkEditable *editable )
{
  const gchar *rst;
  uint8_t len, idx;

  // Validate RST: numerics only,
  // min 337 max 559, ignore blank field
  rst = gtk_entry_get_text( GTK_ENTRY(editable) );
  len = (uint8_t)strlen( rst );

  // Ignore blank fields
  if( len == 0 ) return;

  for( idx = 0; idx < len; idx++ )
  {
    // Reject non-numbers
    if( (rst[idx] < '0') ||
        (rst[idx] > '9') )
    {
      Error_Dialog(
          _("Invalid character entered\n"\
            "Numbers only allowed"), SHOW_OK );
      return;
    }

    // Reject RST < 337 or > 599
    switch( idx )
    {
      case 0:
        if( (rst[idx] < '3') ||
            (rst[idx] > '5') )
        {
          Error_Dialog(
              _("Number out of range\n"\
                "Min = 3 and Max = 5"), SHOW_OK );
          return;
        }
        break;

      case 1:
        if( rst[idx] < '3' )
        {
          Error_Dialog(
              _("Number out of range\n"\
                "Min = 3 and Max = 9"), SHOW_OK );
          return;
        }
        break;

      case 2:
        if( rst[idx] < '7' )
        {
          Error_Dialog(
              _("Number out of range\n"\
                "Min = 7 and Max = 9"), SHOW_OK );
          return;
        }
    } // switch( idx )
  } // for( idx = 0; idx < len; idx++ )

} // Process_RST()

//------------------------------------------------------------------------

/* Locator_Changed()
 *
 * Handles the change callback for the Locator entry
 */
  void
Locator_Changed(
    GtkEditable *editable,
    qso_record_t *qso_rec,
    const char *mode )
{
  char buff[7];

  uint8_t len = Get_Record_Field( buff, sizeof(buff), editable );
  if( len == 0 ) return;

  for( uint8_t idx = 0; idx < len; idx++ )
  {
    // Validate grid locator
    switch( idx )
    {
      case 0: case 1: // First letters
        if( (buff[idx] < 'A') || (buff[idx] > 'S') )
        {
          Error_Dialog(
              _("Invalid character\n"\
                "Letters A-S only"), SHOW_OK );
          return;
        }
        break;

      case 2: case 3: // Middle numbers
        if( (buff[idx] < '0') || (buff[idx] > '9') )
        {
          Error_Dialog(
              _("Invalid character\n"\
                "Numbers 0-9 only"), SHOW_OK );
          return;
        }
        break;

      case 4: case 5: // Last letters
        if( (buff[idx] < 'A') || (buff[idx] > 'X') )
        {
          Error_Dialog(
              _("Invalid character\n"\
                "Letters A-X only"), SHOW_OK );
          return;
        }

    } // switch( idx )
  } // for( idx = 0; idx < len; idx++ )

  // Enter operating mode
  Strlcpy( qso_rec->mode, mode, sizeof(qso_rec->mode) );

  // Enter data to field and QSO record structure
  Strlcpy( qso_rec->dx_loc, buff, sizeof(qso_rec->dx_loc) );
  gtk_entry_set_text( GTK_ENTRY(editable), qso_rec->dx_loc );

} // Locator_Changed()

//------------------------------------------------------------------------

/* Callsign_Changed()
 *
 * Handles the change callback on the callsign entry
 */
  void
Callsign_Changed(
    GtkEditable *editable,
    qso_record_t *qso_rec,
    const char *mode )
{
  uint8_t idx, idy;
  char buff[15];

  uint8_t len = Get_Record_Field( buff, sizeof(buff), editable );
  for( idx = 0; idx < len; idx++ )
  {
    // Allow only alpha-numerics and '/'
    if( ((buff[idx]  < 'A') || (buff[idx]  > 'Z')) &&
        ((buff[idx]  < '0') || (buff[idx]  > '9')) &&
        ((buff[idx] != '/') && (buff[idx] != ' ')) )
    {
      Error_Dialog(
          _("Invalid character entered\n"\
            "Alpha-numerics and '/' only"), SHOW_OK );
      return;
    }

  } // for( idx = 0; idx < len; idx++ )

  // Enter operating mode
  Strlcpy( qso_rec->mode, mode, sizeof(qso_rec->mode) );

  // Enter data to field and QSO record structure
  idy = 0;
  for( idx = 0; idx < len; idx++ )
    if( buff[idx] != ' ' ) // ignore space
      qso_rec->dx_call[idy++] = buff[idx];
  qso_rec->dx_call[idy] = '\0';
  gtk_entry_set_text( GTK_ENTRY(editable), qso_rec->dx_call );

} // Callsign_Changed()

//------------------------------------------------------------------------

/* Band_Changed()
 *
 * Handles the changed callback on the bands entry
 */
  void
Band_Changed(
    GtkEditable *editable,
    qso_record_t *qso_rec,
    const char *mode )
{
  uint8_t len, idx;
  char buff[14];

  // Get entry field, ignore blank field
  Strlcpy( buff, gtk_entry_get_text(GTK_ENTRY(editable)), sizeof(buff) );
  len = (uint8_t)strlen( buff );
  if( len == 0 ) return;

  // Reject non-numbers except '.' and space
  for( idx = 0; idx < len; idx++ )
    if( ((buff[idx]  < '0') || (buff[idx]  > '9')) &&
        ((buff[idx] != '.') && (buff[idx] != ' ')) )
    {
      Error_Dialog(
          _("Invalid character entered\n"\
            "Numbers and '.' only allowed"), SHOW_OK );
      return;
    }

  // Enter operating mode
  Strlcpy( qso_rec->mode, mode, sizeof(qso_rec->mode) );

  // Enter data to QSO record structure
  uint8_t m = 0;
  len = sizeof( qso_rec->freq );
  for( idx = 0; idx < len; idx++ )
  {
    if( buff[idx] != ' ' ) qso_rec->freq[m++] = buff[idx];
    if( buff[idx] == '\0' ) break;
  }

} // Band_Changed()

//------------------------------------------------------------------------

/* Get_Record_Field()
 *
 * Reads the string in a QSO Record entry and processes
 */
  uint8_t
Get_Record_Field( char *str, size_t siz, GtkEditable *editable )
{
  uint8_t len;

  // Read entry, ignore blank field
  Strlcpy( str, (char *)gtk_entry_get_text(GTK_ENTRY(editable)), siz );
  len = (uint8_t)strlen( str );
  if( len == 0 ) return( 0 );

  // Capitalize letters
  if( Flag[GUEST_CAPITALIZE] )
    for( uint8_t idx = 0; idx < len; idx++ )
      str[idx] = (char)toupper( str[idx] );

  return( len );
} // Get_Record_Field()

//------------------------------------------------------------------------

/*  Clear_Record_Fields()
 *
 *  Clears all QSO Record fields
 */
  void
Clear_Record_Fields( BOOLEAN all, GtkBuilder *builder, const char *mode )
{
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "callsign")), "" );
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "rst_in")),   "" );
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "rst_out")),  "" );
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "op_name")),  "" );
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "qth_name")), "" );
  gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "locator")),  "" );
  if( all )
    gtk_entry_set_text( GTK_ENTRY(Builder_Get_Object( builder, "band")), "" );

  qso_record.dx_call[0] = '\0';
  qso_record.my_rst[0]  = '\0';
  qso_record.dx_rst[0]  = '\0';
  Strlcpy( qso_record.dx_name, "OP", sizeof(qso_record.dx_name) );
  qso_record.dx_qth[0]  = '\0';
  qso_record.dx_loc[0]  = '\0';
  if( all ) qso_record.freq[0]  = '\0';

  Flag[GUEST_SAVE_RECORD] = False;
} // Clear_Record_Fields()

//------------------------------------------------------------------------

/*  Read_QSO_Record()
 *
 *  Reads and validates QSO record entries
 */
  BOOLEAN
Read_QSO_Record(
    qso_record_t *qso_rec,
    GtkBuilder* builder,
    const char *mode )
{
  GtkWidget *entry;

  // Enter field values to QSO record structure
  entry = Builder_Get_Object( builder, "callsign" );
  Strlcpy( qso_rec->dx_call,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->dx_call) );
  entry = Builder_Get_Object( builder, "rst_in" );
  Strlcpy( qso_rec->my_rst,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->my_rst) );
  entry = Builder_Get_Object( builder, "rst_out" );
  Strlcpy( qso_rec->dx_rst,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->dx_rst) );
  entry = Builder_Get_Object( builder, "op_name" );
  Strlcpy( qso_rec->dx_name,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->dx_name) );
  entry = Builder_Get_Object( builder, "qth_name" );
  Strlcpy( qso_rec->dx_qth,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->dx_qth) );
  entry = Builder_Get_Object( builder, "locator" );
  Strlcpy( qso_rec->dx_loc,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->dx_loc) );
  entry = Builder_Get_Object( builder, "band" );
  Strlcpy( qso_rec->freq,
      gtk_entry_get_text(GTK_ENTRY(entry)), sizeof(qso_rec->freq) );

  // Enter operating mode
  Strlcpy( qso_rec->mode, mode, sizeof(qso_rec->mode) );

  // Validate QSO Record
  if( (strlen(qso_rec->dx_call) > 2) &&
      (strlen(qso_rec->my_rst) >= 2) &&
      (strlen(qso_rec->dx_rst) >= 2) &&
      (strlen(qso_rec->freq)    > 0) )
    return( True );
  else
    return( False );

} // Read_QSO_Record()

//------------------------------------------------------------------------

/* New_Record()
 *
 * Handles the click on the New Record button
 */
  void
New_Record(
    qso_record_t *qso_rec,
    GtkBuilder *builder,
    const char *mode )
{
  // Variables for time and date
  time_t tp;    // Time type
  struct tm dt; // Date and time

  // For reading Tcvr status
  char freq[8], rst[6];

  // Save record if needed and completed
  if( Flag[GUEST_SAVE_RECORD] )
  {
    if( !Read_QSO_Record(qso_rec, builder, mode) )
    {
      Error_Dialog(
          _("QSO Record is not complete\n"\
            "Cannot save and open new Record"), SHOW_OK );
      return;
    }
    else Save_QSO_Record( qso_rec );
  }

  // Initialize for new entry
  Clear_Record_Fields( False, builder, mode );

  // Enter Receiver status (S-meter and Frequency)
  GtkWidget *entry;
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Enter station's RST
  double S = TRx->S_meter * 20.0;
  if( S < 3.0 ) S = 3.0;
  if( S >= 9.0 ) S = 9.0;
  uint16_t smt;
  if( S < 4.0 ) smt = 300;
  else if( S < 6.0 ) smt = 400;
  else smt = 500;
  smt += (uint16_t)S * 10 + 9;
  snprintf( rst,  sizeof(rst), "%u", smt );
  entry = Builder_Get_Object( builder, "rst_out" );
  gtk_entry_set_text( GTK_ENTRY(entry), rst );

  // Enter frequency in MHz
  double frq = (double)TRx->rx_frequency / 1.0E6;
  snprintf( freq, sizeof(freq), "%.3f", frq );
  entry = Builder_Get_Object( builder, "band" );
  gtk_entry_set_text( GTK_ENTRY(entry), freq );

  // Enter time and date to QSO record
  time( &tp );
  dt = *gmtime( &tp );
  strftime( qso_rec->date,     12, "%d/%b/%Y", &dt );
  strftime( qso_rec->time,      6, "%H:%M",    &dt );
  strftime( qso_rec->date_adif, 9, "%Y%m%d",   &dt );
  strftime( qso_rec->time_adif, 5, "%H%M",     &dt );

  // Enter mode of operation
  Strlcpy( qso_rec->mode, mode, sizeof(qso_rec->mode) );

  Flag[GUEST_SAVE_RECORD] = True;

} // New_Record()

//------------------------------------------------------------------------

/* Save_QSO_Record()
 *
 * Saves QSO record to .adif and .txt files
 */
  BOOLEAN
Save_QSO_Record( qso_record_t *qso_rec )
{
  // File name to save QSO record to
  char file_name[FILE_NAME_SIZE];

  // Buffer for reading and printing to files
  char buff[REC_BUF_SIZE];
  char *buff_idx;

  uint16_t
    num_records, // Number of QSO records
    num_logged;  // Number of QSL's printed

  // Guest_Open_File() return value
  uint8_t fopen_ret;

  // Create a file for QSO ADIF log if not open
  Strlcpy( file_name, getenv("HOME"),    sizeof(file_name) );
  Strlcat( file_name, "/.hermes2/",      sizeof(file_name) );
  Strlcat( file_name, qso_rec->mode_dir, sizeof(file_name) );
  Strlcat( file_name, "/log.adif",       sizeof(file_name) );

  // Open ADIF log file to append
  fopen_ret = Guest_Open_File( &qso_rec->adif_log_fp, file_name, "a" );
  if( fopen_ret == FILE_NEW )
  {
    snprintf( buff, sizeof(buff),
        "QSO Log file created in ADIF v1.0 format by %s <EOH>\n\n",
        PACKAGE_STRING );
    if( !File_Print(&qso_rec->adif_log_fp, buff) )
      return( False );
  }
  else if( fopen_ret == FILE_FAIL )
    return( False );

  // Print QSO record to adif format file
  snprintf( buff,  sizeof(buff),
      "<CALL:%u>%s<QSO_DATE:8>%s<TIME_ON:4>%s\n"
      "<FREQ:%u>%s<MODE:%u>%s<RST_SENT:3>%3s<EOR>\n\n",
      (uint8_t)strlen(qso_rec->dx_call), qso_rec->dx_call,
      qso_rec->date_adif, qso_rec->time_adif,
      (uint8_t)strlen(qso_rec->freq), qso_rec->freq,
      (uint8_t)strlen(qso_rec->mode), qso_rec->mode,
      qso_rec->dx_rst );
  if( !File_Print(&qso_rec->adif_log_fp, buff) )
    return( False );

  // Create a file for station log if not already open
  Strlcpy( file_name, getenv("HOME"),    sizeof(file_name) );
  Strlcat( file_name, "/.hermes2/",       sizeof(file_name) );
  Strlcat( file_name, qso_rec->mode_dir, sizeof(file_name) );
  Strlcat( file_name, "/log.txt",        sizeof(file_name) );

  // Open station log for read/write, create new if failure
  fopen_ret = Guest_Open_File( &qso_rec->station_log_fp, file_name, "r+" );
  if( fopen_ret == FILE_NEW )
  {
    // Print a header to log file
    snprintf( buff, sizeof(buff),
        _("|--------------------------------------------------------------------------|\n"\
          "|       Station Log File - Created in Text format by %10s       |\n"\
          "|--------------------------------------------------------------------------|\n"),
        PACKAGE_STRING );
    if( !File_Print(&qso_rec->station_log_fp, buff) )
      return( False );

    // Print an initial (0) count of records and qsl's
    snprintf( buff,  sizeof(buff),
        _("|  Num of QSO Records:%-6d  Records Logged:%-6d  Size of Records:%-4d  |\n"\
          "|--------------------------------------------------------------------------|\n"),
        0, 0, RECORD_LENGTH );
    if( !File_Print(&qso_rec->station_log_fp, buff) )
      return( False );
  }
  else if( fopen_ret == FILE_FAIL )
    return( False );

  // ***Increment and print Record and QSL count ***
  // Go to beginning of log file
  fseek( qso_rec->station_log_fp, 0, SEEK_SET );

  // Find line with "Number of QSO Records:"
  if( !Locate_Line(&buff_idx, buff, _("QSO Records:"), qso_rec->station_log_fp) )
    return( False );
  num_records = (uint16_t)atoi( buff_idx );

  // Find "Records Logged:" string
  if( !Locate_String(&buff_idx, buff, _("Records Logged:")) )
    return( False );
  num_logged = (uint16_t)atoi(buff_idx);

  // Go back to beginning of line and print new record
  if( fseek(qso_rec->station_log_fp, - (long)(1 + strlen(buff)), SEEK_CUR) != 0 )
    return( False );
  num_records++;
  snprintf( buff,  sizeof(buff),
      _("|  Num of QSO Records:%-6d  Records Logged:%-6d  Size of Records:%-4d  |\n"\
        "|--------------------------------------------------------------------------|\n"),
      num_records, num_logged, RECORD_LENGTH );
  if( !File_Print(&qso_rec->station_log_fp, buff) )
    return( False );

  // *** Print record in text log file ***
  if( fseek(qso_rec->station_log_fp, 0, SEEK_END) != 0 )
    return( False );
  snprintf( buff,  sizeof(buff),
      _("|DX/ CALL: %-14s NAME: %-12s QTH: %-12s  LOC: %-6s|\n"\
        "|MY/ CALL: %-14s ZONE: %-12s QTH: %-12s  LOC: %-6s|\n"\
        "|QSO DATE: %-11s    TIME: %-5s UTC   FREQ: %-13s QSL: NO    |\n"\
        "|QSO MODE: %-11s DX-REPT: %-3s      MY-REPT: %-3s     VIA: %-12s|\n"\
        "|MY TRANS: %-11s   POWER: %-9s    ANT: %-15s          |\n"\
        "|MY RECVR: %-11s   N.FIG: %-7s      ANT: %-15s          |\n"\
        "| REMARKS: %-45s                   |\n"\
        "|------------------------------------"\
        "--------------------------------------|\n"),
      qso_rec->dx_call, qso_rec->dx_name, qso_rec->dx_qth,
      qso_rec->dx_loc, op_data.call, op_data.zone, op_data.qth, op_data.loc,
      qso_rec->date, qso_rec->time, qso_rec->freq,
      qso_rec->mode, qso_rec->dx_rst, qso_rec->my_rst,
      qso_rec->via, op_data.tx, op_data.tx_power, op_data.tx_ant,
      op_data.rx, op_data.rx_nfig, op_data.rx_ant, qso_rec->remarks );
  if( !File_Print(&qso_rec->station_log_fp, buff) )
    return( False );

  fflush( qso_rec->station_log_fp );
  fflush( qso_rec->adif_log_fp );
  //Close_File( qso_rec->station_log_fp );
  //Close_File( qso_rec->adif_log_fp );

  Flag[GUEST_SAVE_RECORD] = False;

  return( True );

} // Save_QSO_Record()

//------------------------------------------------------------------------

/*  Open_Record_File()
 *
 *  Opens a file for raw QSO recording
 */
  BOOLEAN
Open_Record_File( const char *mode_dir )
{
  // Open file for QSO Record, abort on error
  if( Flag[GUEST_RECORD_QSO] )
  {
    // Create a file for recording QSO's
    if( qso_record.qso_record_fp == NULL )
    {
      // File path for QSO record
      char qso_record_fpath[FILE_NAME_SIZE];

      // Make file path for record file
      Strlcpy( qso_record_fpath, getenv("HOME"), sizeof(qso_record_fpath) );
      Strlcat( qso_record_fpath, "/.hermes2/",    sizeof(qso_record_fpath) );
      Strlcat( qso_record_fpath, mode_dir,       sizeof(qso_record_fpath) );
      Strlcat( qso_record_fpath, "/record.txt",  sizeof(qso_record_fpath) );

      // Open or create record file
      qso_record.qso_record_fp = fopen( qso_record_fpath, "a" );
      if( qso_record.qso_record_fp == NULL )
      {
        perror( qso_record_fpath );
        Error_Dialog(
            _("Failed to open QSO Record file\n"\
              "Quit Program and correct"), HIDE_OK );
        return( False );
      }

    } // if( QSO_record_fp == NULL )
  } // if( Flag[GUEST_RECORD_QSO) )

  return( True );
} // Open_Record_File(void)

//------------------------------------------------------------------------

/* Atof()
 *
 * Replaces atof() to take into account the
 * locale-dependent decimal point character
 */
  double
Atof( const char *nptr )
{
  uint8_t idx, len;
  double d = 0.0;
  char *s = NULL;
  static BOOLEAN first_call = True;
  static char dp = '.';

  // Find locale-dependent decimal point character
  if( first_call )
  {
    struct lconv *lcnv;
    lcnv = localeconv();
    dp = *lcnv->decimal_point;
    first_call = False;
  }

  /* Look for a . or , decimal point character
   * in the supplied number buffer (string) */
  len = (uint8_t)strlen( nptr );
  for( idx = 0; idx < (uint8_t)len; idx++ )
    if( (nptr[idx] == ',') || (nptr[idx] == '.') )
      break;

  // Create temporary string to modify decimal point
  Mem_Alloc( (void **)&s, len + 1 );
  strncpy( s, nptr, len + 1 );
  s[len] = '\0';

  // If a decimal point character is found, replace
  if( idx < (int)len ) s[idx] = dp;
  d = atof( s );
  Mem_Free( (void **)&s );

  return( d );
} // End of Atof()

//------------------------------------------------------------------

/*  File_Print()
 *
 *  Prints a string to a file
 */
  BOOLEAN
File_Print( FILE **fp, const char *string )
{
  if( fprintf(*fp, "%s", string) != (int)strlen(string) )
  {
    perror( "hell: fprintf()" );
    Error_Dialog(
        _("Error printing to file\n"\
          "Quit hell and correct"), HIDE_OK );
    Close_File( fp );
    return( False );
  }

  // Flush file
  if( fflush(*fp) != 0 )
  {
    perror( "hell: fflush()" );
    Error_Dialog(
        _("Failed to fflush file\n"\
          "Quit hell and correct"), HIDE_OK );
    return( False );
  }

  return( True );
} // File_Print()

//------------------------------------------------------------------------

/*  Locate_Line()
 *
 *  Finds a line in a text file containing a given string
 */
  BOOLEAN
Locate_Line( char **line_idx, char *line, const char *string, FILE *fp )
{
  int err;

  // Load lines and look for string
  do
  {
    err = Read_Line( line, fp, "Text file line" );
    *line_idx = strstr(line, string);
  }
  while( (err == SUCCESS) && (*line_idx == NULL) );

  // Usually EOF is the cause of error
  if( err )
  {
    Error_Dialog(
        _("Error searching text file\n"\
          "May be empty or corrupted"), SHOW_OK );
    return( False );
  }

  *line_idx += strlen(string);

  return( True );
} // Locate_Line()

//------------------------------------------------------------------------

/*  Locate_String()
 *
 *  Finds a string in a buffer
 */
  BOOLEAN
Locate_String( char **line_idx, char *line, const char *string )
{
  if( (*line_idx = strstr(line, string)) == NULL )
  {
    Error_Dialog(
        _("Error searching station_log.txt\n"\
          "May be empty or corrupted"), SHOW_OK );
    return( False );
  }

  *line_idx += strlen(string);

  return( True );
} // Locate_String()

//------------------------------------------------------------------------

/*  Find_Line()
 *
 *  Finds a line starting with a given string
 */
  BOOLEAN
Find_Line( char *line, const char *str, FILE *fp )
{
  do
  {
    if( Read_Line(line, fp, "Text file line") != SUCCESS )
      return( False );
  }
  while( strncmp( line, str, strlen(str) ) != 0 );

  return( True );
} // Find_Line()

//------------------------------------------------------------------------

/*  Find_String()
 *
 *  Finds a string following spaces
 */
  void
Find_String( const char *line, uint16_t *idx )
{
  // Look for a space
  while( line[++(*idx)] != ' ' )
    if( (line[*idx] == CR) || (line[*idx] == LF) )
    {
      Error_Dialog(
          _("Error reading Font file\n"\
            "Unexpected End-Of-Line"),SHOW_OK );
      return;
    }

  // Look for next non-space char
  while( line[++(*idx)] == ' ' );
  if( (line[*idx] == CR) || (line[*idx] == LF) )
  {
    Error_Dialog(
        _("Error reading Font file\n"\
          "Unexpected End-Of-Line"),SHOW_OK );
    return;
  }

} //  Find_String()

//------------------------------------------------------------------------

/*  Guest_Open_File()
 *
 *  Opens a file or creates one if it does not exist
 */
  uint8_t
Guest_Open_File( FILE **fpointer, const char *fpath, const char *mode )
{
  FILE *fp;

  // Check if file is open
  if( *fpointer != NULL ) return( FILE_OPEN );

  // Try opening the file read-only, to test if exists
  fp = fopen( fpath, "r" );

  // If fopen() fails, create a new file for writing
  if( fp == NULL ) fp = fopen( fpath, "w" );
  if( fp == NULL ) // Failure to open new file
  {
    char mesg[MESG_STRING_SIZE];
    perror( fpath );
    snprintf( mesg, sizeof(mesg),
        _("Failed to open\n%s\n"\
          "Quit hermes2 and correct"), fpath );
    Error_Dialog( mesg, HIDE_OK );
    return( FILE_FAIL );
  }
  else
  {
    // Close and reopen with requested mode
    Close_File( &fp );
    *fpointer = fopen( fpath, mode );
    if( *fpointer == NULL ) // Failure to re-open new file
    {
      char mesg[MESG_STRING_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("Failed to open\n%s\n"\
            "Quit hermes2 and correct"), fpath );
      Error_Dialog( mesg, HIDE_OK );
      return( FILE_FAIL );
    }

    // Get current file position
    long offset = ftell( *fpointer );
    if( offset < 0 )
    {
      char mesg[MESG_STRING_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("ftell() failed for\n%s\n"\
            "Quit hermes2 and correct"), fpath );
      Error_Dialog( mesg, HIDE_OK );
      return( FILE_FAIL );
    }

    // Test if a new file was created
    if( fseek(*fpointer, 0, SEEK_END) < 0 )
    {
      char mesg[MESG_STRING_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("fseek() failed for\n%s\n"\
            "Quit hermes2 and correct"), fpath );
      Error_Dialog( mesg, HIDE_OK );
      return( FILE_FAIL );
    }

    // Test if a new file was created
    if( ftell(*fpointer) == 0 ) return( FILE_NEW );
    else if( fseek(*fpointer, offset, SEEK_SET) < 0 ) // Reset file position
    {
      char mesg[MESG_STRING_SIZE];
      perror( fpath );
      snprintf( mesg, sizeof(mesg),
          _("fseek() failed for\n%s\n"\
            "Quit hermes2 and correct"), fpath );
      Error_Dialog( mesg, HIDE_OK );
      return( FILE_FAIL );
    }
  }

  return( FILE_OPEN );
} // Guest_Open_File()

//------------------------------------------------------------------------

/* Create_Wfall_Pixbuf()
 *
 * Creates a pixbuf for the scope display
 */
  BOOLEAN
Create_Wfall_Pixbuf(
    pixbuffer_t *wfall,
    ifft_data_t *ifft_data,
    uint16_t width, uint16_t height )
{
  // Destroy existing pixbuff
  if( wfall->pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(wfall->pixbuf) );
    wfall->pixbuf = NULL;
  }

  // Create waterfall pixbuf
  wfall->pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8, (int)width, (int)height );
  if( wfall->pixbuf == NULL ) return( False );

  wfall->pixels     = gdk_pixbuf_get_pixels( wfall->pixbuf );
  wfall->width      = (uint16_t)gdk_pixbuf_get_width ( wfall->pixbuf );
  wfall->height     = (uint16_t)gdk_pixbuf_get_height( wfall->pixbuf );
  wfall->rowstride  = (uint16_t)gdk_pixbuf_get_rowstride( wfall->pixbuf );
  wfall->n_channels = (uint8_t)gdk_pixbuf_get_n_channels( wfall->pixbuf );
  gdk_pixbuf_fill( wfall->pixbuf, 0 );

  /* Initialize ifft. The waterfall width has to be odd
   * to provide a center line and the FFT width has to
   * be a power of 2 so we compensate by adding 1 */
  ifft_data->ifft_width = (uint16_t)wfall->width + 1;
  if( !Initialize_iFFT(ifft_data) )
    return( False );

  return( True );

} // Create_Wfall_Pixbuf()

//------------------------------------------------------------------------

/* Guest_Quit_Activate()
 *
 * Handles the Quit signal from a Guest popup
 * menu or Quit button or Delete signal by user
 */
  void
Guest_Quit_Activate( GtkWidget *window )
{
  // Occasionally it is possible for the function to be called before it exits
  static BOOLEAN no_reentry = False;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Avoid re-entry before exiting
  if( no_reentry ) return;
  no_reentry = True;

  // If no mode is running, destroy window
  if( !Flag[GUEST_TRANSMITTING] && !Flag[GUEST_RECEIVING] )
  {
    if( window != NULL ) gtk_widget_destroy( window );
    if( TRx != NULL ) TRx->RSID_Mode = MODE_NULL;
    no_reentry = False;
    return;
  }

  // Signal Guest Rx or Tx threads to quit
  Flag[GUEST_QUIT] = True;

  // Post to Digimode semaphore to free it
  int sval;
  sem_getvalue( &digimode_semaphore, &sval );
  if( !sval ) sem_post( &digimode_semaphore );

  // Join Guest receive thread if active
  if( Flag[GUEST_RECEIVING] )
    pthread_join( hermes2_rc.guest_rx_thread, NULL );

  // Post to DUC transmit semaphore to free it
  sem_getvalue( &duc_send_semaphore, &sval );
  if( !sval ) sem_post( &duc_send_semaphore );

  // Join Guest transmit thread if active
  if( Flag[GUEST_TRANSMITTING] )
    pthread_join( hermes2_rc.guest_tx_thread, NULL );

  while( g_main_context_iteration(NULL, FALSE) );

  // Destroy guest window
  if( window ) gtk_widget_destroy( window );
  Flag[GUEST_QUIT] = False;
  TRx->RSID_Mode   = MODE_NULL;
  no_reentry       = False;

} // Guest_Quit_Activate()

//------------------------------------------------------------------------

/* Alloc_Xmit_Buffers()
 *
 * (Re) allocates and clears transmit buffers
 */
  void
Alloc_Xmit_Buffers( uint32_t buf_size )
{
  // Allocate memory to transmit samples buffers
  xmit_buffer.xmit_buf_size = buf_size;
  size_t mreq = (size_t)xmit_buffer.xmit_buf_size * sizeof(int32_t);
  Mem_Alloc( (void **) &(xmit_buffer.xmit_buf_i), mreq );
  memset( xmit_buffer.xmit_buf_i, 0, mreq );
  Mem_Alloc( (void **) &(xmit_buffer.xmit_buf_q), mreq );
  memset( xmit_buffer.xmit_buf_q, 0, mreq );
} // Alloc_Xmit_Buffers()

//------------------------------------------------------------------------

/* Realloc_Xmit_Buffers()
 *
 * (Re) allocates and clears transmit buffers
 */
  void
Realloc_Xmit_Buffers( uint32_t buf_size )
{
  // Allocate memory to transmit samples buffers
  xmit_buffer.xmit_buf_size = buf_size;
  size_t mreq = (size_t)xmit_buffer.xmit_buf_size * sizeof(int32_t);
  Mem_Realloc( (void **) &(xmit_buffer.xmit_buf_i), mreq );
  Mem_Realloc( (void **) &(xmit_buffer.xmit_buf_q), mreq );
} // Realloc_Xmit_Buffers()

//------------------------------------------------------------------------

// Free resources
  void
Free_Xmit_Buffers( void )
{
  Mem_Free( (void **) &(xmit_buffer.xmit_buf_i) );
  Mem_Free( (void **) &(xmit_buffer.xmit_buf_q) );
}

//------------------------------------------------------------------------

/* Tune_to_Monitor()
 *
 * Tunes the hermes2 tuner to the frequency of the strongest
 * signal near a mouse click in the Spectrum or Signal window
 */
  void
Tune_to_Monitor(
    double pos_x,
    uint8_t tune_range,
    uint16_t wfall_width,
    uint16_t center_freq,
    ifft_data_t *ifft_data )
{
  uint16_t
    from, to,   // Range to scan for a max bin value
    bin_max,    // Max bin value found in this range
    max_idx,    // ifft idx at which the max is found
    ifft_idx,   // Idx used to search ifft bin values
    audio_freq; // Audio frequency in waterfal window

  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  /* Calculate ifft index corresponding to pointer x
   * in waterfall window. This is +1 over the pointer
   * position since the first FFT bin is not used */
  ifft_idx = (uint16_t)( pos_x + 1.5 );

  // Look above and below click point for max bin val
  if( ifft_idx < tune_range )
    from = 0;
  else
    from = ifft_idx - tune_range;
  to = ifft_idx + tune_range;
  if( to >= wfall_width ) to = wfall_width - 1;

  // Find max bin value around click point
  bin_max = 0;
  max_idx = ifft_idx;
  for( ifft_idx = from; ifft_idx < to; ifft_idx++ )
    if( bin_max < ifft_data->bin_ave[ifft_idx] )
    {
      bin_max = ifft_data->bin_ave[ifft_idx];
      max_idx = ifft_idx;
    }

  /* Audio frequency corresponding to ifft index. Since the
   * waterfall width has to be an odd number so that a center
   * line may be present, and since the FFT has to be a power
   * of 2 (e.g. even), the first (DC) output bin of the FFT is
   * not used and hence max_idx has to be increased by 1 and
   * the waterfall width also increased by one to correspond
   * with the FFT width. Only this way the audio frequency that
   * corresponds to FFT index can be calculated accurately. */
  audio_freq = ( 2 * (max_idx + 1) * center_freq ) / ( wfall_width + 1 );

  // Set Tuner Frequency
  int rx_freq = (int)TRx->rx_frequency;
  rx_freq += audio_freq - center_freq;
  if( rx_freq < 0 ) rx_freq = 0;
  TRx->rx_frequency = (uint32_t)rx_freq;
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

  // Link Tx frequency to Rx
  if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
  {
    TRx->tx_frequency = (uint32_t)rx_freq;
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
  }

} // Tune_to_Monitor()

//----------------------------------------------------------------------

/* Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */
  void
Display_Waterfall( pixbuffer_t *wfall, ifft_data_t *ifft_data )
{
  uint16_t
    idh, idv,   // Index to hor. and vert. position in warterfall
    idb,        // Index to average bin values array
    idf,        // Index to ifft output array
    temp;

  // Constant needed to draw white line in waterfall
  uint16_t center_line = wfall->width / 2;

  // Pointer to current pixel
  static guchar *pix;

  // Draw a vertical white line in waterfall at detector's freq.
  pix  = wfall->pixels + wfall->rowstride + wfall->n_channels;
  pix += wfall->n_channels * center_line;
  pix[0] = pix[1] = pix[2] = 0xff;

  // Copy each line of waterfall to next one
  temp = wfall->height - 2;
  for( idv = temp; idv > 0; idv-- )
  {
    pix = wfall->pixels + wfall->rowstride * idv + wfall->n_channels;

    for( idh = 0; idh < wfall->width; idh++ )
    {
      *pix = *( pix - wfall->rowstride ); pix++;
      *pix = *( pix - wfall->rowstride ); pix++;
      *pix = *( pix - wfall->rowstride );
      pix += wfall->n_channels - 2;
    }
  }

  // Got to top left of pixbuf
  pix = wfall->pixels;

  // First (DC) ifft bin not used
  ifft_data->bin_ave[0] = 0;

  // Do the iFFT on input array
  iFFT_Real( ifft_data );
  idb = 0;
  for( idf = 2; idf < ifft_data->data_len; idf += 2 )
  {
    // Calculate power of signal at each frequency "bin"
    uint8_t pixel_val = iFFT_Bin_Value( ifft_data, idf, False );

    // Calculate average bin values
    ifft_data->bin_ave[idb++] = pixel_val;

    // Color code signal strength
    Colorize( pix, pixel_val );
    pix += wfall->n_channels;

  } // for( idf = 2; idf < ifft_data_length; idf += 2 )

  // Reset function
  iFFT_Bin_Value( ifft_data, 0, True );

  // At last draw waterfall
  Queue_Draw( wfall->canvas );

} // Display_Waterfall()

//------------------------------------------------------------------------

/* Read_Station_Info()
 *
 * Reads the Station information and the Macros
 * from the Guest Mode's configuration file
 */
  gboolean
Read_Station_Info(
    FILE *rcfile_ptr,
    char **macro,
    const char *label_name,
    GtkBuilder *builder )
{
  uint16_t
    idx,
    lb_idx, // Label buffer index
    ln_len; // Line buffer length

  char line[READ_LINE_BUF_SIZE];  // Buffer for Read_Line


  // Read operator callsign, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Operator Callsign") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.call, line, sizeof(op_data.call) );

  // Read operator name, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Operator Name") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.name, line, sizeof(op_data.name) );

  // Read Zone, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("ITU Zone") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.zone, line, sizeof(op_data.zone) );

  // Read QTH, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("QTH Name") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.qth, line, sizeof(op_data.qth) );

  // Read QTH locator, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("QTH Locator") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.loc, line, sizeof(op_data.loc) );

  // Read Transmitter name, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Transmitter Name") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.tx, line, sizeof(op_data.tx) );

  // Read Transmitter power, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Transmitter Power") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.tx_power, line, sizeof(op_data.tx_power) );

  // Read Transmitter antenna, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Transmitter Antenna") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.tx_ant, line, sizeof(op_data.tx_ant) );

  // Read Receiver name, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Receiver Name") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.rx, line, sizeof(op_data.rx) );

  // Read Receiver N.Fig, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Receiver Noise Figure") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.rx_nfig, line, sizeof(op_data.rx_nfig) );

  // Read Receiver antenna, abort if EOF
  if( Read_Line(line, rcfile_ptr, _("Receiver Antenna") ) != SUCCESS )
    return( FALSE );
  Strlcpy( op_data.rx_ant, line, sizeof(op_data.rx_ant) );

  // Read propagation medium
  if( Read_Line(line, rcfile_ptr, _("Propagation Medium") ) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.via, line, sizeof(qso_record.via) );

  // Read default remarks on QSO
  if( Read_Line(line, rcfile_ptr, _("Default Remarks") ) != SUCCESS )
    return( FALSE );
  Strlcpy( qso_record.remarks, line, sizeof(qso_record.remarks) );

  // *** Read macro labels from rc file ***
  // Null buffer pointers just in case
  for( idx = 0; idx < NUM_OF_LABELS; idx++ )
    macro[idx] = NULL;

  // Preload a line
  if( Read_Line(line, rcfile_ptr, _("Macro Labels") ) != SUCCESS )
    return( FALSE );

  for( lb_idx = 0; lb_idx < NUM_OF_LABELS; lb_idx++ )
  {
    // For setting macro button labels
    GtkLabel *lbl;
    char lname[16];
    char label[LABEL_LENGTH+6];

    // Abort if first line is not a label
    if( line[0] != '[' )
    {
      fprintf( stderr,
          "First Macro line not a label\n%s\n", line );
      Close_File( &rcfile_ptr );
      Error_Dialog(
          _("Error reading configuration file\n"\
            "First Macro line not a label\n"\
            "Quit Guest Mode and correct"), HIDE_OK);
      return( FALSE );
    }

    // Look for closing ']'
    uint8_t ln_idx = 1;
    while( (ln_idx <= LABEL_LENGTH) && (line[ln_idx] != ']') )
      ln_idx++;

    // Abort if label string > LABEL_LENGTH char + []
    if( ln_idx > LABEL_LENGTH )
    {
      fprintf( stderr,
          "Macro label too long\n%s\n", line );
      Close_File( &rcfile_ptr );
      Error_Dialog(
          _("Error reading configuration file\n"\
            "Macro label too long\n"\
            "Quit Guest Mode and correct"), HIDE_OK);
      return( FALSE );
    }

    // Enter label string if all OK
    snprintf( label, sizeof(label), "[F%d] ", lb_idx+1 );
    line[strlen(line)-1] = '\0';
    Strlcat( label, &line[1], sizeof(label) );

    // Set macro button labels
    snprintf( lname, sizeof(lname), "%s%d", label_name, lb_idx+1 );
    lbl = GTK_LABEL( Builder_Get_Object(builder, lname) );
    gtk_label_set_text( lbl, (gchar *)label );

    // *** Read macros ***
    uint16_t mc_idx = 0; // Macro buffer index
    uint8_t  tg_cnt = 0; // Count '<' and '>'
    do
    {
      if( Read_Line(line, rcfile_ptr, _("Macros")) != SUCCESS )
        return( FALSE );

      // Read labels if line starts with '['
      if( line[0] == '[' ) break;

      // Allocate memory to macro buffer
      ln_len = (uint16_t)strlen( line );

      // Ignore | end-of-macro delimeter
      if( line[ln_len-1] == '|' ) ln_len--;
      Mem_Realloc((void **) &macro[lb_idx], (size_t)(ln_len + mc_idx + 2) );

      // Concatenate lines to macro
      for( ln_idx = 0; ln_idx < ln_len; ln_idx++ )
      {
        macro[lb_idx][mc_idx++] = line[ln_idx];

        // Count tag delimiters
        if( (line[ln_idx] == '<') ||
            (line[ln_idx] == '>') )
          tg_cnt++;
      }

      // Terminate Macro with LF and null
      macro[lb_idx][mc_idx++] = LF;
      macro[lb_idx][mc_idx] = '\0';
    } // do
    while( line[ln_len] != '|'  );

    // Abort if tag delimiters are odd number
    if( tg_cnt & 1 )
    {
      fprintf( stderr, "Malformed tag in Macro\n%s\n", line );
      Close_File( &rcfile_ptr );
      Error_Dialog(
          _("Error reading configuration file\n"\
            "Malformed tag in Macro\n"\
            "Quit Guest Mode and correct"), HIDE_OK );
      return( FALSE );
    }

  } // for( lb_idx = 0; lb_idx < NUM_OF_LABELS; lb_idx++ )

  return( TRUE );
} // Read_Station_Info()

//------------------------------------------------------------------------

/* Save_Window_Position()
 *
 * Gets and saves the positon of window at the end of given file
 */
  void
Save_Window_Position( GtkWidget *window, const char *fname )
{
  // Get window position
  gint x, y;
  gtk_window_get_position( GTK_WINDOW(window), &x, &y );

  // Open file
  FILE *fd = NULL;
  if( Open_File(&fd, fname, "r+") == FAILURE )
  {
    Error_Dialog( _("Save_Window_Position(): Open_File() failed"), SHOW_OK );
    Close_File( &fd );
    return;
  }

  // Seek back 14 bytes from end of file to write the position
  if( fseek(fd, -12, SEEK_END) != 0 )
  {
    Error_Dialog( _("Save_Window_Position(): fseek() failed"), SHOW_OK );
    Close_File( &fd );
    return;
  }

  // Print window position to file (13 bytes).
  if( fprintf(fd, "%4d,%4d\n#", x, y) != 11 )
  {
    Error_Dialog( _("Save_Window_Position(): fprintf() failed"), SHOW_OK );
    Close_File( &fd );
    return;
  }

  // Flush and close file
  Close_File( &fd );

} // Save_Window_Position()

//------------------------------------------------------------------------

/* Get_Window_Geometry()
 *
 * Gets a Window size and position
 */
  void
Get_Window_Geometry(
    GtkWidget *window,
    gint *x, gint *y,
    gint *width, gint *height )
{
  // Save window state for restoring later
  if( window )
  {
    if( width && height )
      gtk_window_get_size( GTK_WINDOW(window), width, height );
    if( x && y )
      gtk_window_get_position( GTK_WINDOW(window), x, y );
  }
  else // Disable window restoring
  {
    if( x && y )
    {
      *x = -1;
      *y = -1;
    }

    if( width && height )
    {
      *width  = 0;
      *height = 0;
    }
  }

} // Get_Window_Geometry()

//------------------------------------------------------------------------

/* Set_Window_Geometry()
 *
 * Sets the size and position of a window
 */
  void
Set_Window_Geometry(
    GtkWidget *window,
    gint x, gint y, gint width, gint height )
{
  if( window == NULL ) return;

  // Set size and position of window
  if( width && height )
    gtk_window_resize( GTK_WINDOW(window), width, height );
  gtk_window_move( GTK_WINDOW(window), x, y );

  // Wait for GTK to complete its tasks
  while( g_main_context_iteration(NULL, FALSE) );

} // Set_Window_Geometry()

//------------------------------------------------------------------------

/** Table of constant values **/
#define DEG2RAD  1.74532925E-2   /* Degrees to Radians */
#define ERADIUS  6.378135E3      /* Earth radius in km */

/* Gridloc_to_Position()
 *
 * Function to calculate longitude/latitude
 * given a position's grid locator
 */
  void
Gridloc_to_Position( const char *grid_locator, double *latitude, double *longitude )
{
  *longitude  = (grid_locator[0] - 'A') * 20.0;
  *longitude += (grid_locator[2] - '0') * 2.0;

  // If locator is only first 4 digits, use 'M' instead
  if( strlen(grid_locator) == 4 )
    *longitude += ('M' - 'A') / 12.0;
  else
    *longitude += (grid_locator[4] - 'A') / 12.0;

  *longitude -= 180.0;

  *latitude  = (grid_locator[1] - 'A') * 10.0;
  *latitude += (grid_locator[3] - '0') * 1.0;

  // If locator is only first 4 digits, use 'M' instead
  if( strlen(grid_locator) == 4 )
    *latitude += ('M' - 'A') / 12.0;
  else
    *latitude += (grid_locator[5] - 'A') / 24.0;

  *latitude -= 90.0;

} /* End of Gridloc_to_Position() */

/*--------------------------------------------------------------------*/

/* Bearing_Distance()
 *
 * Function to calculate and show bearing
 * and distance of Remote from Home
 */
  void
Bearing_Distance(
    double *bearing,   double *distance,
    double home_lat,   double home_long,
    double remote_lat, double remote_long )
{
  double
    gc_arc, cos_gc_arc,       /* Great circle arc and cos A to B */
    cos_bearing, sin_bearing, /* cos/sin of bearing  from A to B */
    lon_diff;                 /* Difference in longitude of B from A    */

  /* Temporary long/latd */
  double
    lon_a, lat_a,
    lon_b, lat_b;

  /* Convert to radians */
  lat_a = home_lat    * DEG2RAD;
  lon_a = home_long   * DEG2RAD;
  lat_b = remote_lat  * DEG2RAD;
  lon_b = remote_long * DEG2RAD;

  /* Longitude differnce of B from A */
  lon_diff = lon_b - lon_a;

  /* Calculate great circle distance A to B */
  cos_gc_arc = cos(lon_diff) * cos(lat_a) * cos(lat_b) + sin(lat_a) * sin(lat_b);
  gc_arc = acos( cos_gc_arc );

  /* Distance in km */
  *distance = ERADIUS * gc_arc;

  /* Calculate bearing A to B */
  cos_bearing  = sin(lat_b)    - sin(lat_a) * cos_gc_arc;
  sin_bearing  = sin(lon_diff) * cos(lat_a) * cos(lat_b);
  *bearing     = atan2(sin_bearing, cos_bearing);

  /* Correct negative (anticlockwise) bearings */
  if( *bearing < 0.0 ) *bearing += M_2PI;

  /* Convert to degrees */
  *bearing /= DEG2RAD;

} /* End of Bearing_Distance() */

/*--------------------------------------------------------------------*/

// Thread function to launch xplanet
  static void *
_Xplanet( void *command )
{
  // Kill xplanet
  int ret = system( "pkill xplanet" );
  if( ret != 0 ) perror( "!! hermes2: system(pkill xplanet)" );

  // Invoke xplanet
  ret = system( (char *)command );
  if( ret != 0 ) perror( "!! hermes2: system(xplanet)" );

  return( NULL );
} // _Xplanet()

//------------------------------------------------------------------

// Utility to launch xplanet using above thread function
  void
Launch_Xplanet( char *command )
{
  pthread_t thrd;
  if( !Pthread_Create(
        &thrd, NULL, _Xplanet, (void *)command, _("Failed to create xplanet thread")) )
    return;
} // Launch_Xplanet()

//------------------------------------------------------------------

// Utility to kill xplanet and unlink config files
  void
Kill_Xplanet( char *config_file, char *marker_file, char *arc_file  )
{
  int ret = system( "pkill xplanet" );
  if( ret == 0 ) perror( "!! hermes2: system(pkill xplanet)" );
  unlink( config_file );
  unlink( marker_file );
  unlink( arc_file );
} // Kill_Xplanet()

//------------------------------------------------------------------

