/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "filters.h"
#include "common.h"
#include "shared.h"
#include "transceiver.h"
#include "utils.h"
#include <math.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>

//----------------------------------------------------------------------

/* Low Pass Filter parameters for main filter.
 * The number of poles _must_be_even_ */
#define ROOF_FILTER_POLES    6
#define ROOF_FILTER_RIPPLE   1.0

//----------------------------------------------------------------------

/* Init_Chebyshev_Filter()
 *
 * Calculates Chebyshev recursive filter coefficients.
 * The filter_data_t struct is defined in common/transceiver.h
 */
  static void
Init_Chebyshev_Filter( filter_data_t *filter_data )
{
  double *ta = NULL, *tb = NULL;
  double sa, sb, gain;
  uint32_t i, p;
  double es, vx, kx, k, t, w;

  // Abort for incorrect input
  if( !filter_data->npoles ) return;

  // Suspend data I/O during init of filter
  Flag[HERMES2_SUSPEND_DATA_IO] = True;
  usleep( 1000 );

  // Allocate data arrays
  size_t mreq = (size_t)(filter_data->npoles + 3) * sizeof(double);
  Mem_Alloc( (void **) &ta, mreq );
  Mem_Alloc( (void **) &tb, mreq );

  // Allocate coefficient arrays
  Mem_Realloc( (void **) &(filter_data->a), mreq );
  Mem_Realloc( (void **) &(filter_data->b), mreq );

  // Allocate saved input and output arrays
  mreq = (size_t)(filter_data->npoles + 1) * sizeof(double);
  Mem_Realloc( (void **) &(filter_data->x), mreq );
  Mem_Realloc( (void **) &(filter_data->y), mreq );

  // Clear x and y arrays
  for( i = 0; i <= filter_data->npoles; i++ )
  {
    filter_data->x[i] = 0.0;
    filter_data->y[i] = 0.0;
  }

  // Clear coefficient arrays
  for( i = 0; i <= filter_data->npoles + 2; i++ )
  {
    filter_data->a[i] = 0.0;
    filter_data->b[i] = 0.0;
  }
  filter_data->a[2] = 1.0;
  filter_data->b[2] = 1.0;

  // S-domain to Z-domain conversion
  t = 2.0 * tan( 0.5 );

  // Cutoff frequency
  if( filter_data->cutoff >= 0.49 ) filter_data->cutoff = 0.49;
  w = M_2PI * filter_data->cutoff;

  // Low Pass to Low Pass or Low Pass to High Pass transform
  if( filter_data->type == FILTER_HIGHPASS )
    k = -cos( (w + 1.0) / 2.0 ) / cos( (w - 1.0) / 2.0 );
  else if( filter_data->type == FILTER_LOWPASS )
    k = sin( (1.0 - w) / 2.0 ) / sin( (1.0 + w) / 2.0 );
  else k = 1.0; // For compiler warnings */

  // Find coefficients for 2-pole filter for each pole pair
  for( p = 1; p <= filter_data->npoles / 2; p++ )
  {
    double a0, a1, a2, b1, b2, rp, ip, m;
    double xn0, xn1, xn2, yn1, yn2, tmp;

    // Calculate the pole location on the unit circle
    tmp  = M_PI / (double)filter_data->npoles / 2.0;
    tmp += (double)(p - 1) * M_PI / (double)filter_data->npoles;
    rp  = -cos( tmp );
    ip  =  sin( tmp );

    // Wrap from a circle to an ellipse
    if( filter_data->ripple > 0.0 )
    {
      tmp = 100.0 / ( 100.0 - filter_data->ripple );
      es  = sqrt( tmp * tmp - 1.0 );
      tmp = 1.0 / (double)filter_data->npoles;
      vx  = tmp * asinh( 1.0 / es );
      kx  = tmp * acosh( 1.0 / es );
      kx  = cosh( kx );
      rp *= sinh( vx ) / kx;
      ip *= cosh( vx ) / kx;
    }

    // S-domain to Z-domain conversion
    m = rp * rp + ip * ip;
    double d = 4.0 - 4.0 * rp * t + m * t * t;
    xn0 = t * t / d;
    xn1 = 2.0 * t * t / d;
    xn2 = t * t / d;
    yn1 = ( 8.0 - 2.0 * m * t * t ) / d;
    yn2 = ( -4.0 -4.0 * rp * t - m * t * t ) / d;

    // Low Pass to Low Pass or Low Pass to High Pass transform
    d  = 1.0 + yn1 * k -yn2 * k * k;
    a0 = ( xn0 - xn1 * k + xn2 * k * k ) / d;
    a1 = ( -2.0 * xn0 * k + xn1 + xn1 * k * k - 2.0 * xn2 * k ) / d;
    a2 = ( xn0 * k * k - xn1 * k + xn2 ) / d;
    b1 = ( 2.0 * k + yn1 + yn1 * k * k - 2.0 * yn2 * k ) / d;
    b2 = ( -k * k - yn1 * k + yn2 ) / d;

    if( filter_data->type == FILTER_HIGHPASS )
    {
      a1 = -a1;
      b1 = -b1;
    }

    // Add coefficients to the cascade
    for( i = 0; i <= filter_data->npoles + 2; i++ )
    {
      ta[i] = filter_data->a[i];
      tb[i] = filter_data->b[i];
    }

    for( i = 2; i <= filter_data->npoles + 2; i++ )
    {
      filter_data->a[i] = a0 * ta[i] + a1 * ta[i-1] + a2 * ta[i-2];
      filter_data->b[i] =      tb[i] - b1 * tb[i-1] - b2 * tb[i-2];
    }
  } // for( p = 1; p <= np / 2; p++ )

  // Finish combining coefficients
  filter_data->b[2] = 0.0;
  for( i = 0; i <= filter_data->npoles; i++ )
  {
    filter_data->a[i] =  filter_data->a[i+2];
    filter_data->b[i] = -filter_data->b[i+2];
  }

  // Normalize the gain
  double sign  = 1.0;
  sa = 0.0; sb = 0.0;
  for( i = 0; i <= filter_data->npoles; i++ )
  {
    if( filter_data->type == FILTER_LOWPASS )
    {
      sa += filter_data->a[i];
      sb += filter_data->b[i];
    }
    else if( filter_data->type == FILTER_HIGHPASS )
    {
      sa += filter_data->a[i] * sign;
      sb += filter_data->b[i] * sign;
      sign = -sign;
    }
  } // for( i = 0; <= filter_data->npoles; i++ )

  /* Normalize the filtered output. The '/ 2.0' term is my addition, since the
   * calculated gain seems to be high by a factor of 2 (filter o/p is 1/2 of input) */
  gain = sa / ( 1.0 - sb );
  for( i = 0; i <= filter_data->npoles; i++ )
  {
    filter_data->a[i] /= gain;
  }

  Mem_Free( (void **) &ta );
  Mem_Free( (void **) &tb );

  Flag[HERMES2_SUSPEND_DATA_IO] = False;
  return;

} // Init_Chebyshev_Filter()

//----------------------------------------------------------------------

// Free resources
  void
Deinit_Chebyshev_Filter( filter_data_t *filter_data )
{
  Mem_Free( (void **) &(filter_data->a) );
  Mem_Free( (void **) &(filter_data->b) );
  Mem_Free( (void **) &(filter_data->x) );
  Mem_Free( (void **) &(filter_data->y) );
}

//----------------------------------------------------------------------

/* DSP_Filter()
 *
 * DSP Recursive Filter, normally used as low pass
 */
  void
DSP_Filter( filter_data_t *filter_data )
{
  // Index to samples buffer
  uint32_t buf_idx, idx, npp1, len;
  double y;

  // Initialize filter if needed
  if( filter_data->init_filter == True )
  {
    Init_Chebyshev_Filter( filter_data );
    filter_data->init_filter = False;
  }

  // Filter samples in the buffer
  npp1 = filter_data->npoles + 1;
  len  = filter_data->samples_buffer_len;
  for( buf_idx = 0; buf_idx < len; buf_idx++ )
  {
    double yn0;

    // Calculate and save filtered samples
    yn0 = filter_data->samples_buffer[buf_idx] * filter_data->a[0];
    for( idx = 1; idx < npp1; idx++ )
    {
      // Summate contribution of past input samples
      y    = filter_data->a[idx];
      y   *= filter_data->x[filter_data->ring_idx];
      yn0 += y;

      // Summate contribution of past output samples
      y    = filter_data->b[idx];
      y   *= filter_data->y[filter_data->ring_idx];
      yn0 += y;

      // Advance ring buffers index
      filter_data->ring_idx++;
      if( filter_data->ring_idx >= npp1 )
        filter_data->ring_idx = 0;
    } // for( idx = 0; idx < npp1; idx++ )

    // Save new yn0 output to y ring buffer
    filter_data->y[filter_data->ring_idx] = yn0;

    // Save current input sample to x ring buffer
    filter_data->x[filter_data->ring_idx] =
      filter_data->samples_buffer[buf_idx];

    // Return filtered samples
    filter_data->samples_buffer[buf_idx] = yn0;

  } // for( buf_idx = 0; buf_idx < len; buf_idx++ )

} // DSP_Filter()

//----------------------------------------------------------------------

/* Init_Roofing_Filter()
 *
 * Initializes main roofing filter
 */
  void
Init_Roofing_Filter( Transceiver_t *TRx )
{
  // Init I Low Pass Filter
  TRx->roof_filter_i.cutoff   = (double)TRx->spectrum_data.fft_bandwidth / 2.0;
  TRx->roof_filter_i.cutoff  /= (double)hermes2_rc.ddc_sample_rate;
  TRx->roof_filter_i.cutoff  *= 0.95; // Anti alias guard
  TRx->roof_filter_i.ripple   = ROOF_FILTER_RIPPLE;
  TRx->roof_filter_i.npoles   = ROOF_FILTER_POLES;
  TRx->roof_filter_i.type     = FILTER_LOWPASS;
  TRx->roof_filter_i.ring_idx = 0;
  TRx->roof_filter_i.samples_buffer_len = hermes2_rc.ddc_buffer_length;
  TRx->roof_filter_i.init_filter = True;

  // Init Q Low Pass Filter
  TRx->roof_filter_q.cutoff   = (double)TRx->spectrum_data.fft_bandwidth / 2.0;
  TRx->roof_filter_q.cutoff  /= (double)hermes2_rc.ddc_sample_rate;
  TRx->roof_filter_q.cutoff  *= 0.95; // Anti alias guard
  TRx->roof_filter_q.ripple   = ROOF_FILTER_RIPPLE;
  TRx->roof_filter_q.npoles   = ROOF_FILTER_POLES;
  TRx->roof_filter_q.type     = FILTER_LOWPASS;
  TRx->roof_filter_q.ring_idx = 0;
  TRx->roof_filter_q.samples_buffer_len = hermes2_rc.ddc_buffer_length;
  TRx->roof_filter_q.init_filter = True;

} // Init_Roofing_Filter()

//----------------------------------------------------------------------

