/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_CONVERT_H
#define COMMON_CONVERT_H   1

#include <stdint.h>

//----------------------------------------------------------------------

/* Int16_to_String()
 *
 * Converts a 16-bit integer to a 2-byte Big Endian string.
 */
  static inline void
Int16_to_String( int16_t num, uint8_t *seq_str )
{
  seq_str[0] = ((uint8_t *) &num)[1];
  seq_str[1] = ((uint8_t *) &num)[0];
} // Int16_to_String()

//----------------------------------------------------------------------

/* Uint16_to_String()
 *
 * Converts a 16-bit unsigned integer to a 2-byte Big Endian string.
 */
  static inline void
Uint16_to_String( uint16_t num, uint8_t *seq_str )
{
  seq_str[0] = ((uint8_t *) &num)[1];
  seq_str[1] = ((uint8_t *) &num)[0];
} // Uint16_to_String()

//----------------------------------------------------------------------

/* String_to_Uint16()
 *
 * Converts a 2-byte Big Endian string to a 16-bit unsigned integer
 */
  static inline uint16_t
String_to_Uint16( const uint8_t *str )
{
  uint16_t num = 0;

  ((uint8_t *) &num)[1] = str[0];
  ((uint8_t *) &num)[0] = str[1];

  return( num );
} // String_to_Uint16()

//----------------------------------------------------------------------

/* String_to_Int16()
 *
 * Converts a 2-byte LITTLE ENDIAN string to a 16-bit integer
 */
  static inline int16_t
String_to_Int16( const uint8_t *str )
{
  int16_t num = 0;

  ((uint8_t *) &num)[0] = str[0];
  ((uint8_t *) &num)[1] = str[1];

  return( num );
} // String_to_Uint16()

//----------------------------------------------------------------------

/* Uint32_to_String()
 *
 * Converts a 32-bit integer to a 4-byte Big Endian string
 */
  static inline void
Uint32_to_String( uint32_t num, uint8_t *str )
{
  str[0] = ((uint8_t *) &num)[3];
  str[1] = ((uint8_t *) &num)[2];
  str[2] = ((uint8_t *) &num)[1];
  str[3] = ((uint8_t *) &num)[0];
} // Uint32_to_String()

//----------------------------------------------------------------------

/* String_to_Uint32()
 *
 * Converts a 4-byte Big Endian string to a 32-bit unsigned integer
 */
  static inline uint32_t
String_to_Uint32( const uint8_t *str )
{
  uint32_t num = 0;

  ((uint8_t *) &num)[3] = str[0];
  ((uint8_t *) &num)[2] = str[1];
  ((uint8_t *) &num)[1] = str[2];
  ((uint8_t *) &num)[0] = str[3];

  return( num );
} // String_to_Uint32()

//----------------------------------------------------------------------

/* String_to_Int32()
 *
 * Converts a 2-byte Big Endian string to a 32-bit integer
 */
  static inline int32_t
String_to_Int32( const uint8_t *str )
{
  int32_t num = 0;

  ((uint8_t *) &num)[3] = str[0];
  ((uint8_t *) &num)[2] = str[1];
  ((uint8_t *) &num)[1] = 0;
  ((uint8_t *) &num)[0] = 0;
  num /= 65536;

  return( num );
} // String_to_Int32()

//----------------------------------------------------------------------

/* String_to_Uint64()
 *
 * Converts a 8-byte sequence string to a 64-bit integer
  static inline uint64_t
String_to_Uint64( uint8_t *str )
{
  uint64_t num;

  ((uint8_t *) &num)[7] = str[0];
  ((uint8_t *) &num)[6] = str[1];
  ((uint8_t *) &num)[5] = str[2];
  ((uint8_t *) &num)[4] = str[3];
  ((uint8_t *) &num)[3] = str[4];
  ((uint8_t *) &num)[2] = str[5];
  ((uint8_t *) &num)[1] = str[6];
  ((uint8_t *) &num)[0] = str[7];

  return( num );
} */ // String_to_Uint64()

//----------------------------------------------------------------------

/* String_to_Sample()
 *
 * Converts a 3-byte Big Endian string to a 32-bit integer
 */
  static inline int32_t
String_to_Sample( const uint8_t *str )
{
  int32_t num;

  ((uint8_t *) &num)[3] = str[0];
  ((uint8_t *) &num)[2] = str[1];
  ((uint8_t *) &num)[1] = str[2];
  ((uint8_t *) &num)[0] = 0;
  num /= 256;

  return( num );
} // String_to_Sample()

//----------------------------------------------------------------------

/* Sample32_to_String()
 *
 * Converts a 32-bit integer to a 3-byte Big Endian string
 */
  static inline void
Sample32_to_String( int32_t num, uint8_t *str )
{
  str[0] = ((uint8_t *) &num)[2];
  str[1] = ((uint8_t *) &num)[1];
  str[2] = ((uint8_t *) &num)[0];
} // Sample_to_String()

//----------------------------------------------------------------------

#endif

