/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "cfft.h"
#include "common.h"
#include "shared.h"
#include "transceiver.h"
#include "utils.h"
#include <math.h>
#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

/* Initialize_FFT()
 *
 * Initializes the FFT() function
 */
  void
Initialize_FFT(
    spectrum_data_t *spectrum_data,
    uint16_t spectrum_width,
    uint16_t input_length,
    BOOLEAN forward )
{
  size_t mreq;

  // Forward or reverse flag
  spectrum_data->fft_forward = forward;

  // (Re) allocate input buffers
  if( spectrum_data->fft_in_length != input_length )
  {
    spectrum_data->fft_in_length = input_length;
    mreq = (size_t)input_length * sizeof(double);
    Mem_Realloc( (void **) &(spectrum_data->fft_in_i), mreq );
    Mem_Realloc( (void **) &(spectrum_data->fft_in_q), mreq );
    memset( spectrum_data->fft_in_i, 0, mreq );
    memset( spectrum_data->fft_in_q, 0, mreq );

    // Calculate the order of fft
    uint16_t l = input_length;
    for( spectrum_data->fft_order = 0; l != 1; spectrum_data->fft_order++ )
      l /= 2;
  }

  // (Re)allocate FFT output bins
  if( spectrum_data->fft_out_length != spectrum_width )
  {
    spectrum_data->fft_out_length = spectrum_width;
    mreq = (size_t)spectrum_width * sizeof(int);
    Mem_Realloc( (void **) &(spectrum_data->fft_bin_values), mreq );
    memset( spectrum_data->fft_bin_values, 0, mreq );
  }

  // Initialize the FFT function after allocations complete
  if( hermes2_rc.waterfall_width &&
      hermes2_rc.waterfall_height )
    spectrum_data->spectrum_init = True;
} // Initialize_FFT()

//----------------------------------------------------------------------

// Free resources
  void
Deinit_FFT( spectrum_data_t *spectrum_data )
{
  Mem_Free( (void **) &(spectrum_data->fft_in_i) );
  Mem_Free( (void **) &(spectrum_data->fft_in_q) );
  Mem_Free( (void **) &(spectrum_data->fft_bin_values) );
}

//----------------------------------------------------------------------

/* cFFT
 *
 * This computes an in-place complex-to-complex Radix-2 FFT.
 */
  void
cFFT(
    uint16_t fft_in_length,
    uint16_t fft_order,
    double  *fft_in_i,
    double  *fft_in_q )
{
  uint16_t a, b, c, step, dft, bfly;
  double cr, ci, tr, ti, z;

  // Decompose time domain signal (bit reversal)
  a = fft_in_length - 1;
  for( b = 1; b < a; b++ )
  {
    c = 0;
    for( uint8_t d = 0; d < fft_order; d++ )
    {
      // Bit reversal of indices
      c <<= 1;
      c |= (b >> d) & 0x01;
    }

    // Enter data to locations with bit reversed indices
    if( b < c )
    {
      tr = fft_in_i[b];
      ti = fft_in_q[b];
      fft_in_i[b] = fft_in_i[c];
      fft_in_q[b] = fft_in_q[c];
      fft_in_i[c] = tr;
      fft_in_q[c] = ti;
    }
  } // for( i = 0; i < l; i++ )

  // Compute the FFT
  cr = -1.0;
  ci = 0.0;
  a  = 1;

  /* Loop over the number of decomposition
   * (bit reversal) steps (the FFT order) */
  for( step = 0; step < fft_order; step++ )
  {
    b = a;
    a <<= 1;
    double ur = 1.0;
    double ui = 0.0;

    // Loop over sub-dft's
    for( dft = 0; dft < b; dft++ )
    {
      // Loop over each butterfly
      for( bfly = dft; bfly < fft_in_length; bfly += a )
      {
        c = bfly + b;
        tr = ur * fft_in_i[c] - ui * fft_in_q[c];
        ti = ur * fft_in_q[c] + ui * fft_in_i[c];
        fft_in_i[c] = fft_in_i[bfly] - tr;
        fft_in_q[c] = fft_in_q[bfly] - ti;
        fft_in_i[bfly] += tr;
        fft_in_q[bfly] += ti;
      } // for( bfly = dft; bfly < fft_width; bfly += a )

      /* Calculate the complex product of "twiddle"
       * sinusoid and the complex time domain point */
      z =  ur * cr - ui * ci;
      ui = ur * ci + ui * cr;
      ur = z;
    } // for( dft = 0; dft < b; dft++ )

    // Calculate the "twiddle" factors recursively
    ci = -sqrt( (1.0 - cr) / 2.0 );
    cr =  sqrt( (1.0 + cr) / 2.0 );
  } // for( step = 0; step < fft_order; step++ )

} // cFFT()

//----------------------------------------------------------------------

