/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "spectrum.h"
#include "demodulate.h"
#include "display.h"
#include "interface.h"
#include "../common/common.h"
#include "../common/cfft.h"
#include "../common/shared.h"
#include "../common/guest_utils.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <math.h>
#include <stdint.h>
#include <stdio.h>

//----------------------------------------------------------------------

// Cairo RGB values for colors in plotting
#define SCOPE_DEMODBW      0.0, 0.4, 0.0
#define SCOPE_RETICLE      0.7, 0.5, 0.0
#define SCOPE_SCALES       1.0, 0.9, 0.0
#define SCOPE_CENTERLINE   0.0, 1.0, 1.0
#define WATERFALL_SCALES   1.0, 0.9, 0.0
#define WATERFALL_RETICLE  1.0, 1.0, 1.0
#define WATERFALL_BGND     0x00400000

//----------------------------------------------------------------------

// The fraction of spectrum area height allocated to waterfall
#define WATERFALL_RATION  6 / 16

// Max number of steps in Frequency scale
#define MAX_FREQ_SCALE_STEPS  18

//----------------------------------------------------------------------

// Number of steps in the Level scale
#define NUM_LEVEL_STEPS  6
#define LEVEL_STEP       15

//----------------------------------------------------------------------

/* Size of pango layout for displaying the
 * vertical scales in the Spectrum Monitors */
static gint
  vertical_scale_width,
  vertical_scale_height;

//----------------------------------------------------------------------

/* Fit_to_Int_Scale()
 *
 * Finds a new range of values between upper_val and lower_val
 * so that the value of steps between scale points is a
 * multiple of 1, 2, 5 or 10, for easier interpolation
 */
  static void
Fit_to_Int_Scale(
    const double *upper_val,
    const double *lower_val,
    double *range,
    double *scale_step,
    uint32_t *num_points )
{
  int order;

  // Initial range of values between max and min
  *range = *upper_val - *lower_val;
  if( *range < 10.0 ) return;

  // Initial values of step between scale points
  *scale_step = *range / (double)( *num_points );

  // Order (log 10) of this step
  order = (int)( log10(*scale_step) );

  // Get this step in the range 1.0 - 10.0
  *scale_step /= pow( 10.0, (double)order );

  /* Chose a nearby step value that is easier to interpolate
   * between scale points, e.g. scale steps of 1, 2, 5, 10 */
  if( *scale_step < 1.0 )  *scale_step = 1.0;
  else if( *scale_step < 2.0 )  *scale_step = 2.0;
  else if( *scale_step < 5.0 )  *scale_step = 5.0;
  else if( *scale_step < 10.0 ) *scale_step = 10.0;

  // Raise this new step to its original order
  *scale_step *= pow( 10.0, (double)order );

  /* Find a new number of steps to match new step value.
   * The new range must be smaller than the original */
  *num_points = (uint32_t)( *range * 0.95 / *scale_step );

  // Calculate new range to be multiple of scale intervals
  *range = *scale_step * (double)( *num_points );

} // Fit_to_Int_Scale()

//-----------------------------------------------------------------------

/* Plot_Frequency_Scale()
 *
 * Draws out a horizontal scale, between the min
 * and max value of the FFT spectrum frequency range
 */
  static uint32_t
Plot_Frequency_Scale(
    spectrum_data_t *spectrum_data,
    cairo_t   *cr,
    double     pos_x,
    double     pos_y,
    double    *x_offset,
    double    *range,
    uint32_t  *num_points )
{
  uint32_t idx;
  char value[16];
  PangoLayout *layout;

  static int
    pl_width  = 0, // Pango Layout Width
    pl_height = 0; // Pango Layout Height

  // Lower and Upper value of scale and scale step
  double lower_val, upper_val, scale_step = 1.0;

  double
    center_freq,   // Center of Monitors frequency range
    khz_per_pixel, // Resolution of Signal Display in kHz/pixel
    offset,        // Offset in kHz between center freq and nearest scale point
    origin,        // The frequency of first scale point of the freq scale
    stride,        // The distance between signal display scale points in kHz
    margin;        // Margin in kHz between first scale point and lowest freq

  // Half the FFT bandwidth in kHz
  uint32_t fft_bw, fft_bw2;

  // Center frequency in kHz
  center_freq = (double)spectrum_data->center_freq / 1000.0;

  /* Number of values to use in the frequency scale is
   * calculated according to the order of the frequency */
  *num_points = MAX_FREQ_SCALE_STEPS - (uint32_t)( log10(center_freq) );

  // FFT bandwidth in kHz
  fft_bw = spectrum_data->fft_bandwidth / 1000;

  // Half the FFT bandwidth in kHz
  fft_bw2 = fft_bw / 2;

  /* This is the range of values
   * required for the frequency scale */
  lower_val = center_freq - fft_bw2;
  upper_val = center_freq + fft_bw2;

  /* Adjust the range and number of values in
   * frequency scale to make a legible display */
  Fit_to_Int_Scale(
      &upper_val, &lower_val, range, &scale_step, num_points );
  *range -= scale_step; // Reduce to accommodate scale numbers

  // Abort if not enough values to plot
  if( *num_points < 3 ) return( 0 );

  // Frequency step per pixel of Spectrum Monitors (kHz per pixel)
  khz_per_pixel = fft_bw / (double)spectrum_data->fft_out_length;

  // Margin in kHz between FFT B/W and Scale range, either side
  margin = ( fft_bw - *range ) / 2.0;

  // Frequency in kHz of first point in frequency scale
  origin = lower_val + margin;

  // Round origin to multiple of scale steps
  idx    = (uint32_t)( origin / scale_step + 0.5 );
  origin = (double)idx * scale_step;

  // Offset between Rx center freq and the center of scale
  offset = margin + origin + *range / 2.0 - center_freq;

  /* Offset in pixels for the position of the frequency
   * scale. It makes it possible to keep the Rx center
   * frequency in the middle of the signal display */
  *x_offset = offset / khz_per_pixel;

  // Create a pango layout and get sizes
  snprintf( value, sizeof(value), "%1.0f", center_freq );
  layout = gtk_widget_create_pango_layout(
      spectrum_data->spectrum_drawingarea, value );
  pango_layout_get_pixel_size( layout, &pl_width, &pl_height );

  /* The clear height of the signal display above the frequency scale.
   * It is rounded to a multiple of the Level scales number of steps to make
   * the stride of the Level scale values and scale lines an integer number. */
  hermes2_rc.scope_clear_height =
    ( hermes2_rc.scope_height - (uint16_t)pl_height ) / NUM_LEVEL_STEPS;
  hermes2_rc.scope_clear_height *= NUM_LEVEL_STEPS;

  // Clear the Frequency scale area
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle( cr,
      0.0, (double)hermes2_rc.scope_clear_height,
      (double)hermes2_rc.spectrum_width,
      (double)(hermes2_rc.scope_height - hermes2_rc.scope_clear_height) );
  cairo_fill( cr );

  // Align with center of Pango Layout rectangle
  pos_x  = *x_offset + (double)vertical_scale_width;
  pos_x -= (double)pl_width / 2.0;
  pos_y -= (double)pl_height;

  // Distance between printed scale values
  stride  =
    (double)spectrum_data->fft_out_length * *range / fft_bw;
  stride /= (double)(*num_points - 1);

  // Print horizontal scale values using Pango
  cairo_set_source_rgb( cr, SCOPE_SCALES );
  for( idx = 0; idx < *num_points; idx++ )
  {
    snprintf( value, sizeof(value), "%1.0f", origin );
    pango_layout_set_text( layout, value, -1 );
    cairo_move_to( cr, pos_x, pos_y );
    pango_cairo_show_layout( cr, layout );
    origin += scale_step;
    pos_x += stride;
  }

  g_object_unref( layout );

  return( (uint32_t) pl_height );

} // Plot_Frequency_Scale()

//-----------------------------------------------------------------------

/* Plot_Level_Scale()
 *
 * Plots a vertical scale to the left of the Scope
 * Display and horizontal reticle lines. Normally
 * this would be in dBm or dB relative as needed.
 */
  static void
Plot_Level_Scale( spectrum_data_t *spectrum_data, cairo_t *cr )
{
  int idx;
  double yps, stride, origin;
  char value[16];
  PangoLayout *layout;

  // Distance in pixels between horizontal reticle lines
  stride = (double)hermes2_rc.scope_clear_height / NUM_LEVEL_STEPS;

  // Cairo set up to draw lines
  cairo_set_line_width( cr, 1.0 );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SCOPE_RETICLE );

  // Draw horizontal reticle lines
  yps = stride;
  for( idx = 0; idx < NUM_LEVEL_STEPS; idx++ )
  {
    cairo_move_to( cr, (double)vertical_scale_width,  yps );
    cairo_line_to( cr, (double)hermes2_rc.scope_width, yps );
    yps += stride;
  }

  // Draw a horizontal line at the bottom of the Scope display
  cairo_move_to( cr,
      (double)vertical_scale_width,
      (double)hermes2_rc.scope_height - 1.0 );
  cairo_line_to( cr,
      (double)hermes2_rc.spectrum_width,
      (double)hermes2_rc.scope_height - 1.0 );

  // Draw left vertical border line of Scope display
  cairo_move_to( cr, (double)vertical_scale_width, 0.0 );
  cairo_line_to( cr,
      (double)vertical_scale_width,
      (double)hermes2_rc.scope_height );

  // Stroke cairo paths
  cairo_stroke( cr );

  // Print scale values before reticle lines
  yps    = stride - vertical_scale_height / 2.0;
  origin = 0.0;
  layout = gtk_widget_create_pango_layout(
      spectrum_data->spectrum_drawingarea, "000" );
  cairo_set_source_rgb( cr, SCOPE_SCALES );
  for( idx = 0; idx < NUM_LEVEL_STEPS; idx++ )
  {
    // Print scale values
    origin -= LEVEL_STEP;
    snprintf( value, sizeof(value), "%1.0f", origin );
    pango_layout_set_text( layout, value, -1 );
    cairo_move_to( cr, 1.0, yps );
    pango_cairo_show_layout( cr, layout );
    yps += stride;
  }

  g_object_unref( layout );

} // Plot_Level_Scale()

//-----------------------------------------------------------------------

/* Hermes2_Display_Waterfall()
 *
 * Displays the spectrum waterfall in response
 * to the draw event for the Spectrum drawingarea
 */
  static void
Hermes2_Display_Waterfall(
    cairo_t *cr,
    spectrum_data_t *spectrum_data,
    uint8_t rx_idx )
{
  uint32_t idx;
  int origin;
  double yps;
  char value[4];
  PangoLayout *layout;

  static uint32_t
    reticle_points[MAX_RECEIVERS],
    fame_rate[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  static gint
    reticle_stride[MAX_RECEIVERS],
    spectrum_height[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  double pixbuf_y =
    (double)( hermes2_rc.spectrum_height - hermes2_rc.waterfall_height );


  /* Paint the FFT spectrum as waterfall. waterfall_pixbuf
   * is prepared in Paint_Waterfall() in process.c */
  gdk_cairo_set_source_pixbuf( cr, spectrum_data->waterfall_pixbuf, 0.0, pixbuf_y );
  cairo_paint( cr );

  // Cairo set up to draw lines
  cairo_set_line_width( cr, 1.0 );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SCOPE_RETICLE );

  // Draw vertical border line at left of Waterfall
  cairo_move_to( cr, (double)vertical_scale_width, pixbuf_y );
  cairo_line_to( cr, (double)vertical_scale_width,
      (double)hermes2_rc.waterfall_height + pixbuf_y );

  // Stroke cairo paths
  cairo_stroke( cr );

  // Change of Spectrum Monitors height or rate
  if( (spectrum_height[rx_idx] != hermes2_rc.waterfall_height) ||
      (fame_rate[rx_idx] != spectrum_data->fft_frame_rate) )
  {
    /* Number of waterfall lines that are a multiple of frame rate
     * and not less than the height of pango's font pixel size */
    gint n = 1 +
      vertical_scale_height / spectrum_data->fft_frame_rate;

    /* The stride in lines of the reticle (horizontal lines)
     * in the waterfall display. They are seconds markers */
    reticle_stride[rx_idx] = n * spectrum_data->fft_frame_rate;

    // Prefered number of steps in time scale
    reticle_points[rx_idx] =
      (uint32_t)( hermes2_rc.waterfall_height / reticle_stride[rx_idx] );

    spectrum_height[rx_idx] = hermes2_rc.waterfall_height;
    fame_rate[rx_idx] = spectrum_data->fft_frame_rate;
  } // if( (height[rx_idx] != hermes2_rc.waterfall_height) ||

  /* Draw horizontal reticle lines.
   * Start from here on Y axis. */
  yps = pixbuf_y + (double)reticle_stride[rx_idx];
  cairo_set_source_rgb( cr, WATERFALL_RETICLE );
  for( idx = 0; idx < reticle_points[rx_idx]; idx++ )
  {
    cairo_move_to( cr, (double)vertical_scale_width, yps );
    cairo_line_to( cr, (double)hermes2_rc.waterfall_width, yps );
    yps += (double)reticle_stride[rx_idx];
  }

  // Stroke cairo paths
  cairo_stroke( cr );

  // Print scale values at start of reticle lines
  yps    = pixbuf_y + (double)reticle_stride[rx_idx] / 2.0;
  origin = 0; // Starting value of scale points
  layout = gtk_widget_create_pango_layout(
      spectrum_data->spectrum_drawingarea, "000" );
  cairo_set_source_rgb( cr, WATERFALL_SCALES );

  // Print scale values (secs)
  for( idx = 0; idx < reticle_points[rx_idx]; idx++ )
  {
    origin += reticle_stride[rx_idx] / spectrum_data->fft_frame_rate;
    snprintf( value, sizeof(value), "%3d", origin );
    pango_layout_set_text( layout, value, -1 );
    cairo_move_to( cr, 1.0, yps );
    pango_cairo_show_layout( cr, layout );
    yps += (double)reticle_stride[rx_idx];
  }

  g_object_unref( layout );

} // Hermes2_Display_Waterfall()

//----------------------------------------------------------------------

// Points to plot
static GdkPoint *points[MAX_RECEIVERS] = { NULL, NULL, NULL, NULL };

/* Hermes2_Display_Scope()
 *
 * Displays amplitude of the FFT Spectrum bins and draws
 * the verical and horizontal reticle lines, in response
 * to the draw event for the Spectrum Monitors drawingarea
 */
  static void
Hermes2_Display_Scope(
    GtkWidget *widget,
    cairo_t *cr,
    spectrum_data_t *spectrum_data,
    uint8_t rx_idx )
{
  static uint32_t
    points_idx[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    num_points[MAX_RECEIVERS];

  static double
    range[MAX_RECEIVERS] = { 10.0, 10.0, 10.0, 10.0 },
    x_offset[MAX_RECEIVERS];

  GdkDrawingContext *gdc;
  GdkWindow *window;

  /* X co-ordinate and width of the Demodulator
   * bandwidth rectangle in the signal signal display */
  double
    dembw_x,     // X co-ordinate of Demodulator bandwidth rectangle
    dembw_width, // Width of Demodulator bandwidth rectangle
    bkgnd_width, // Width of signal background areas, outside above
    xps, yps,    // X and Y co-ordinates for Cairo graphics
    stride;      // Distance in pixels between vertical scale lines

  // Index to fft output array
  uint32_t fft_idx, idx, pl_height; // Pango layout height

  // Initialize on first call
  if( points[rx_idx] == NULL )
  Mem_Alloc( (void **) &points[rx_idx], SPECTRUM_WIDTH * sizeof(GdkPoint) );

  // Initialize drawing
  window = gtk_widget_get_window( widget );
  cairo_region_t *region = cairo_region_create();
  gdc = gdk_window_begin_draw_frame( window, region );
  cairo_region_destroy( region );
  cairo_set_line_join( cr, CAIRO_LINE_JOIN_ROUND );
  cairo_set_line_cap(  cr, CAIRO_LINE_CAP_ROUND );

  // Plot the frequency scale under the signal
  pl_height =
    Plot_Frequency_Scale(
        spectrum_data,
        cr, 0.0,
        (double)hermes2_rc.scope_height,
        &x_offset[rx_idx],
        &range[rx_idx],
        &num_points[rx_idx] );

  // Save values to be plotted (scaled to fit display)
  points_idx[rx_idx] = 0;
  for( fft_idx = 0; fft_idx < spectrum_data->fft_out_length; fft_idx++ )
  {
    points[rx_idx][points_idx[rx_idx]].y =
      hermes2_rc.scope_clear_height - spectrum_data->fft_bin_values[fft_idx];
    if( points[rx_idx][points_idx[rx_idx]].y < 0 )
      points[rx_idx][points_idx[rx_idx]].y = 0;
    points[rx_idx][points_idx[rx_idx]].x = (gint)points_idx[rx_idx];
    points_idx[rx_idx]++;
  } // for( fft_idx = 0; fft_idx < ... )

  // Draw demodulator bandwidth background
  dembw_width  = (double)spectrum_data->fft_out_length;
  dembw_width *=
    (double)spectrum_data->rx_bandwidth /
    (double)spectrum_data->fft_bandwidth;

  // Make bandwidth background width odd so it has a center
  uint16_t iw = (uint16_t)dembw_width;
  iw |= 1;
  dembw_width = (double)iw;
  dembw_x  = ( (double)spectrum_data->fft_out_length - 1 - dembw_width ) / 2.0;
  dembw_x += (double)vertical_scale_width;

  cairo_set_source_rgb( cr, SCOPE_DEMODBW );
  cairo_rectangle( cr,
      dembw_x, 0.0, dembw_width, (double)hermes2_rc.scope_clear_height );
  cairo_fill( cr );

  // Draw signal background, right of bandwidth area
  bkgnd_width = ( (double)spectrum_data->fft_out_length - 1 - dembw_width ) / 2.0;
  cairo_set_source_rgb( cr, SCOPE_BACKGND );
  cairo_rectangle( cr,
      dembw_x + dembw_width, 0.0,
      bkgnd_width, (double)hermes2_rc.scope_clear_height );
  cairo_fill( cr );

  // Draw signal background, left of bandwidth area
  bkgnd_width += (double)vertical_scale_width;
  cairo_rectangle( cr, 0.0, 0.0,
      bkgnd_width, (double)hermes2_rc.scope_clear_height );
  cairo_fill( cr );

  // Plot signal graph
  cairo_set_line_width( cr, 1.5 );
  cairo_set_source_rgb( cr, SCOPE_FOREGND );
  cairo_move_to( cr,
      (double)points[rx_idx][0].x + (double)vertical_scale_width + 0.5,
      (double)points[rx_idx][0].y );
  for( idx = 1; idx < points_idx[rx_idx]; idx++ )
    cairo_line_to( cr,
        (double)points[rx_idx][idx].x + (double)vertical_scale_width + 0.5,
        (double)points[rx_idx][idx].y );

  // Stroke paths
  cairo_stroke( cr );

  // Cairo set up to draw vertical divisions
  cairo_set_line_width( cr, 1.0 );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SCOPE_RETICLE );

  // Distance between vertical reticle lines, in pixels
  stride = (double)spectrum_data->fft_out_length * range[rx_idx] * 1000.0;
  stride /=
    (double)spectrum_data->fft_bandwidth *
    (double)(num_points[rx_idx] - 1);

  /* Starting X position and Y position
   * of vertical reticle lines, in pixels */
  xps = x_offset[rx_idx] + (double)vertical_scale_width;
  yps = (double)hermes2_rc.scope_clear_height + (double)pl_height / 3.0;

  // Draw vertical reticle lines
  for( idx = 0; idx <= num_points[rx_idx]; idx++ )
  {
    cairo_move_to( cr, xps, 0.0 );
    cairo_line_to( cr, xps, yps );
    xps += stride;
  }

  // Stroke paths
  cairo_stroke( cr );

  // Draw horizontal reticle lines
  Plot_Level_Scale( spectrum_data, cr );

  // Draw vertical centerline
  cairo_set_source_rgb( cr, SCOPE_CENTERLINE );
  xps =
    (double)spectrum_data->fft_out_length / 2.0 +
    (double)vertical_scale_width;
  yps = (double)hermes2_rc.scope_clear_height;
  cairo_move_to( cr, xps, 0.0 );
  cairo_line_to( cr, xps, yps );

  // Stroke paths
  cairo_stroke( cr );

  // Wait for GTK to complete its tasks
  gdk_window_end_draw_frame( window, gdc );

} // Hermes2_Display_Scope

//----------------------------------------------------------------------

  void
Free_Scope_Points( uint8_t idx )
{
  Mem_Free( (void **) &points[idx] );
}

//----------------------------------------------------------------------

/* Hermes2_Spectrum_Draw()
 *
 * Handles the on_spectrum_drawingarea_draw() callback
 */
  void
Hermes2_Spectrum_Draw( GtkWidget *widget, cairo_t *cr, gpointer user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data) );
  title = strstr( title, "TRX" );
  uint8_t rx_idx = (uint8_t)atoi( title + 3 );
  Transceiver_t *TRx = Transceiver[rx_idx];

  if( !TRx->spectrum_data.spectrum_drawingarea ||
      !TRx->spectrum_data.spectrum_init )
    return;

  /* Enter the center frequency and demodulator
   * bandwidth for the Spectrum monitors */
  if( TRx->spectrum_data.receiver_spectrum )
  {
    TRx->spectrum_data.center_freq = TRx->rx_frequency + (uint32_t)TRx->rx_weaver_offset;
    TRx->spectrum_data.rx_bandwidth = TRx->rx_bandwidth;
  }

  // Display spectrum as oscilloscope and waterfall
  Hermes2_Display_Scope( widget, cr, &(TRx->spectrum_data), rx_idx );
  Hermes2_Display_Waterfall( cr, &(TRx->spectrum_data), rx_idx );

  // Cairo set up to draw vertical divisions
  cairo_set_line_width( cr, 1.0 );
  cairo_set_antialias(  cr, CAIRO_ANTIALIAS_NONE );
  cairo_set_source_rgb( cr, SCOPE_RETICLE );

  // Draw a frame around the drawingarea
  cairo_rectangle( cr,
      1.0, 1.0,
      (double)(hermes2_rc.spectrum_width  - 1),
      (double)(hermes2_rc.spectrum_height - 1) );

  // Stroke paths
  cairo_stroke( cr );

} // Hermes2_Spectrum_Draw()

//----------------------------------------------------------------------

/* Hermes2_Spectrum_Size_Alloc()
 *
 * Calculates widget sizes of spectrum display
 */
  void
Hermes2_Spectrum_Size_Alloc( GtkWidget *widget )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Pango layout size for vertical scales
  PangoLayout *layout;
  layout = gtk_widget_create_pango_layout( TRx->tcvr_window, "000" );
  pango_layout_get_pixel_size( layout, &vertical_scale_width, &vertical_scale_height );
  g_object_unref( layout );

  /* Widen the default size of the spectrum displays
   * to accomodate the time and level scales on the left */
  uint16_t width = SPECTRUM_WIDTH + (uint16_t)vertical_scale_width;
  gtk_widget_set_size_request( widget, width, SPECTRUM_HEIGHT );

  // Initialize for Spectrum allocation
  // Destroy existing pixbuff
  if( TRx->spectrum_data.waterfall_pixbuf != NULL )
  {
    g_object_unref( G_OBJECT(TRx->spectrum_data.waterfall_pixbuf) );
    TRx->spectrum_data.waterfall_pixbuf = NULL;
  }

  // Create Spectrum Monitors pixbuf
  // Use a little less than half the available height
  TRx->spectrum_data.waterfall_pixbuf = gdk_pixbuf_new(
      GDK_COLORSPACE_RGB, FALSE, 8, width, SPECTRUM_HEIGHT * WATERFALL_RATION );
  if( TRx->spectrum_data.waterfall_pixbuf == NULL )
  {
    TRx->spectrum_data.waterfall_pixels = NULL;
    hermes2_rc.waterfall_width      = 0;
    hermes2_rc.waterfall_height     = 0;
    hermes2_rc.waterfall_rowstride  = 0;
    hermes2_rc.waterfall_n_channels = 0;

    Error_Dialog( _("Failed to create pixbuf for Spectrum Monitors"), HIDE_OK );
    return;
  }

  // Get Spectrum Monitors pixbuf details and clear it
  hermes2_rc.waterfall_n_channels =
    (uint16_t)gdk_pixbuf_get_n_channels( TRx->spectrum_data.waterfall_pixbuf );
  TRx->spectrum_data.waterfall_pixels =
    gdk_pixbuf_get_pixels( TRx->spectrum_data.waterfall_pixbuf );
  TRx->spectrum_data.waterfall_pixels +=
    vertical_scale_width * hermes2_rc.waterfall_n_channels;
  hermes2_rc.waterfall_width =
    (uint16_t)gdk_pixbuf_get_width ( TRx->spectrum_data.waterfall_pixbuf );
  hermes2_rc.waterfall_height =
    (uint16_t)gdk_pixbuf_get_height( TRx->spectrum_data.waterfall_pixbuf );
  hermes2_rc.waterfall_rowstride =
    (uint16_t)gdk_pixbuf_get_rowstride( TRx->spectrum_data.waterfall_pixbuf );
  gdk_pixbuf_fill( TRx->spectrum_data.waterfall_pixbuf, WATERFALL_BGND );

  // Allocate rest of available height to Scope
  hermes2_rc.scope_height = SPECTRUM_HEIGHT - hermes2_rc.waterfall_height;
  hermes2_rc.scope_width  = width;

} // Hermes2_Spectrum_Size_Alloc()

//----------------------------------------------------------------------

/* Hermes2_Tune_to_Monitor()
 *
 * Tunes the hermes2 tuner to the frequency of the strongest
 * signal near a mouse click in the Spectrum or Signal window
 */
  void
Hermes2_Tune_to_Monitor( double pos_x, gpointer user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data) );
  title = strstr( title, "TRX" );
  uint8_t rx_idx = (uint8_t)atoi( title + 3 );
  Transceiver_t *TRx = Transceiver[rx_idx];

  int
    from, to,   // Range to scan for a max bin value
    bin_max,    // Max bin value found in this range
    max_idx,    // fft idx at which the max is found
    fft_idx,    // Idx used to search fft bin values
    center_freq = (int)TRx->spectrum_data.center_freq;

  // Frequency at click position in Spectrum Monitors
  double click_freq;

  // Abort if hermes2 not ready
  if( !Flag[HERMES2_INITIALIZED] ||
      !TRx->spectrum_data.spectrum_init ||
      !TRx->spectrum_data.spectrum_running )
    return;

  /* Calculate fft index corresponding
   * to pointer x in Spectrum Monitors */
  fft_idx = (int)pos_x - vertical_scale_width;

  // Look above and below click point for max bin val
  from = fft_idx - CLICK_TUNE_RANGE;
  if( from < 0 ) from = 0;
  to = fft_idx + CLICK_TUNE_RANGE;
  if( to >= (int)TRx->spectrum_data.fft_out_length )
    to = SPECTRUM_WIDTH - 1;

  // Find max bin value around click point
  bin_max = 0; max_idx = fft_idx;
  for( fft_idx = from; fft_idx < to; fft_idx++ )
    if( bin_max < TRx->spectrum_data.fft_bin_values[fft_idx] )
    {
      bin_max = TRx->spectrum_data.fft_bin_values[fft_idx];
      max_idx = fft_idx;
    }

  // Find frequency offset from center of spectrum
  double mid = (double)TRx->spectrum_data.fft_out_length / 2.0;
  click_freq = (double)max_idx - mid;
  click_freq *=
    (double) TRx->spectrum_data.fft_bandwidth /
    (double)(TRx->spectrum_data.fft_out_length );

  // Set Tuner Frequency
  center_freq += (int)click_freq;
  if( center_freq < 0 ) center_freq = 0;
  TRx->rx_frequency = (uint32_t)center_freq;
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

  // Link Tx frequency to Rx
  if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
  {
    TRx->tx_frequency = (uint32_t)center_freq;
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
  }

  return;
} // Hermes2_Tune_to_Monitor()

//----------------------------------------------------------------------

