/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "demodulate.h"
#include "interface.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include <alsa/asoundlib.h>
#include <gtk/gtk.h>
#include <math.h> /////////////////// temp
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

//----------------------------------------------------------------------

#define SND_PERIOD_LENGTH   2048    // PCM period size
#define SND_NUM_PERIODS   4      // Number of periods
#define SND_BUF_STRIDE    4096   // Sound buffer stride. Must be 2 * PERIOD_LENGTH
#define SND_EXACT_VALUE   0
#define SND_PCM_MODE      SND_PCM_ASYNC
#define SND_PCM_ACCESS    SND_PCM_ACCESS_RW_INTERLEAVED
#define SND_PCM_FORMAT    SND_PCM_FORMAT_S16_LE

//----------------------------------------------------------------------

// ALSA pcm capture playback and mixer handles
static snd_pcm_t   *playback_handle = NULL;
static snd_pcm_t   *capture_handle  = NULL;
static snd_mixer_t *mixer_handle    = NULL;

// Simple mixer elements for setting up capture volume
static snd_mixer_elem_t *capture_elem = NULL;

// Hardware parameters type
static snd_pcm_hw_params_t *hw_params = NULL;

static uint32_t sound_num_writes;
static int
  recv_buf_size, // Recv DSP signal samples buffer
  recv_buf_idx;  // Index to Rx signal samples buffer

// Receive samples ring buffer
#define NUM_SND_RING_BUFS    8
static short *recv_buffer[NUM_SND_RING_BUFS];

//----------------------------------------------------------------------

/* Xrun_Recovery()
 *
 * Recover from underrrun (broken pipe) and suspend
 */
  static BOOLEAN
Xrun_Recovery( snd_pcm_t *phandle, int error )
{
  if( error == -EPIPE )
  {
    error = snd_pcm_prepare( phandle );
    if( error < 0 )
    {
      fprintf( stderr,
          _("Cannot recover from underrun, prepare failed\n"\
            "Error: %s\n"), snd_strerror(error) );
      return( False );
    }
  }
  else if( error == -ESTRPIPE )
  {
    while( (error = snd_pcm_resume(phandle)) == -EAGAIN )
      sleep(1);
    if( error < 0 )
    {
      error = snd_pcm_prepare( phandle );
      if( error < 0 )
      {
        fprintf( stderr,
            _("Cannot recover from suspend, prepare failed\n"\
              "Error: %s\n"), snd_strerror(error) );
        return( False );
      }
    }
  }

  return( True );
} // Xrun_Recovery()

//----------------------------------------------------------------------

/* Open_PCM()
 *
 * Opens a pcm device for a given handle
 */
  static BOOLEAN
Open_PCM( snd_pcm_t **handle, snd_pcm_stream_t stream, char *mesg )
{
   int error;

  // Open Sound Card PCM device
  error = snd_pcm_open( handle, hermes2_rc.pcm_device, stream, SND_PCM_ASYNC );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot open sound device "), MESG_STRING_SIZE );
    Strlcat( mesg, hermes2_rc.pcm_device, MESG_STRING_SIZE );
    return( FALSE );
  }

  // Allocate memory to hardware parameters structure
  error = snd_pcm_hw_params_malloc( &hw_params );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot allocate hw_params struct"), MESG_STRING_SIZE );
    return( FALSE );
  }

  // Initialize hardware parameter structure
  error = snd_pcm_hw_params_any( *handle, hw_params );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot initialize hw_params struct"), MESG_STRING_SIZE );
    return( FALSE );
  }

  // Set access type
  error = snd_pcm_hw_params_set_access( *handle, hw_params, SND_PCM_ACCESS );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot set PCM access type"), MESG_STRING_SIZE );
    return( FALSE );
  }

  // Set sample format
  error = snd_pcm_hw_params_set_format( *handle, hw_params, SND_PCM_FORMAT );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot set sample format"), MESG_STRING_SIZE );
    return( FALSE );
  }

  // Set sample rate
  error = snd_pcm_hw_params_set_rate( *handle, hw_params, SND_DSP_RATE, SND_EXACT_VALUE );
  if( error < 0 )
  {
    snprintf( mesg, MESG_STRING_SIZE, _("Cannot set sample rate to %d"), SND_DSP_RATE );
    return( FALSE );
  }

  // Set channel count
  error = snd_pcm_hw_params_set_channels( *handle, hw_params, SND_NUM_CHANNELS );
  if( error < 0 )
  {
    snprintf( mesg, MESG_STRING_SIZE, _("Cannot set channel count to %d"), SND_NUM_CHANNELS );
    return( FALSE );
  }

  // Set number of periods
  error = snd_pcm_hw_params_set_periods( *handle, hw_params, SND_NUM_PERIODS, SND_EXACT_VALUE );
  if( error < 0)
  {
    snprintf( mesg, MESG_STRING_SIZE,
        _("Cannot set number periods to %d"), SND_NUM_PERIODS );
    return( FALSE );
  }

  // Set period size
  error = snd_pcm_hw_params_set_period_size(
      *handle, hw_params, SND_PERIOD_LENGTH, SND_EXACT_VALUE );
  if( error < 0)
  {
    snprintf( mesg, MESG_STRING_SIZE,
        _("Cannot set period size to %d"), SND_PERIOD_LENGTH );
    return( FALSE );
  }

  // Set parameters
  error = snd_pcm_hw_params( *handle, hw_params );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot set capture parameters"), MESG_STRING_SIZE );
    return( FALSE );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  // Prepare audio interface for use
  error = snd_pcm_prepare( *handle );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot prepare audio interface"), MESG_STRING_SIZE );
    return( FALSE );
  }

  return( TRUE );
} // Open_PCM()

//----------------------------------------------------------------------

/* Open_PCSound()
 *
 * Opens sound card for PCSound
 */
  BOOLEAN
Open_PCSound( void )
{
  char err_mesg[MESG_STRING_SIZE];
  err_mesg[0] = '\0';

  // Abort if PC Sound Playback already open
  if( Flag[HERMES2_PCSOUND_SETUP] ) return( True );

  // Open & setup pcm for PCSound
  if( !Open_PCM(&playback_handle, SND_PCM_STREAM_PLAYBACK, err_mesg) )
  {
    char mesg[ MESG_STRING_SIZE ];
    snprintf( mesg, MESG_STRING_SIZE,
        _("%s\nCreate a new configuration file for hermes2?"), err_mesg );
    Message_Dialog( mesg );
    Flag[HERMES2_CREATE_CONFIG] = True;
    Flag[HERMES2_PCSOUND_SETUP] = False;
    return( False );
  }

  sound_num_writes = SOUND_OP_BUF_SIZE / SND_PERIOD_LENGTH;

  // Allocate memory to PCSound ring buffers
  if( !sound_ring_buf )
  {
    // Allocate ring buffers array
    Mem_Alloc( (void **) &sound_ring_buf, sizeof(short *) * (size_t)NUM_SOUND_RING_BUFFS );

    // Allocate sound ring buffers
    size_t buf_size  = SND_NUM_CHANNELS * SOUND_OP_BUF_SIZE * sizeof(short);
    for( uint32_t idx = 0; idx < NUM_SOUND_RING_BUFFS; idx++ )
    {
      sound_ring_buf[idx] = NULL;
      Mem_Alloc( (void **) &sound_ring_buf[idx], buf_size );
    }
  }

  Flag[HERMES2_PCSOUND_SETUP] = True;

  return( True );
} // Open_PCSound()

//----------------------------------------------------------------------

/*  DSP_Write()
 *
 *  Write a samples buffer to DSP
 */
  BOOLEAN
DSP_Write( short *buffer, uint32_t buf_len )
{
  snd_pcm_sframes_t error;
  uint32_t idx, buf_idx;

  // Abort if sound card not ready
  if( playback_handle == NULL ) return( False );

  // Write samples buffer to DSP in chunks of PERIOD_SIZE
  buf_idx = 0;
  for( idx = 0; idx < sound_num_writes; idx++ )
  {
    // Write samples buffer to DSP
    error = snd_pcm_writei( playback_handle, &buffer[buf_idx], SND_PERIOD_LENGTH );
    if( error != SND_PERIOD_LENGTH )
    {
      if( error < 0 )
        fprintf( stderr, _("hermes2: DSP_Write(): %s\n"),
            snd_strerror((int)error) );
      else
        fprintf( stderr,
            _("hermes2: DSP_Write(): Written only %d frames out of %u\n"),
            (int)error, buf_len );

      // Try to recover from error
      if( !Xrun_Recovery(playback_handle, (int)error) ) return( False );
    } // if( error != PERIOD_SIZE )

    buf_idx += SND_BUF_STRIDE;
  } // for( idx = 0; idx < sound_num_writes; idx++ )

  return( True );
} // DSP_Write()

//----------------------------------------------------------------------

/* Close_PCSound()
 *
 * Close the PC Sound set up
 */
  void
Close_PCSound( void )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Wait for async write to terminate
  Flag[HERMES2_PCSOUND_SETUP] = False;
  if( Flag[HERMES2_PCSOUND_RUNNING] )
    pthread_join( TRx->sound_thread, NULL );

  // Free startrx buffer
  if( sound_ring_buf )
  {
    for( uint32_t idx = 0; idx < NUM_SOUND_RING_BUFFS; idx++ )
      Mem_Free( (void **) &sound_ring_buf[idx] );
    Mem_Free( (void **) &sound_ring_buf );
  }

  if( hw_params != NULL )
  {
    snd_pcm_hw_params_free( hw_params );
    hw_params = NULL;
  }

  if( playback_handle != NULL )
  {
    snd_pcm_close( playback_handle );
    playback_handle = NULL;
  }

  if( capture_handle != NULL )
  {
    snd_pcm_close( capture_handle );
    capture_handle = NULL;
  }

} // Close_PCSound()

//----------------------------------------------------------------------

static pthread_t snd_readi_thrd;

  static void *
Readi_Thread( void *arg )
{
  while( Flag[HERMES2_CAPTURE_SETUP] )
  {
    static uint8_t thrd_idx = NUM_SND_RING_BUFS / 2;

    // Read audio samples from DSP, abort on error
    snd_pcm_sframes_t error;
    error = snd_pcm_readi( capture_handle, recv_buffer[thrd_idx], SND_PERIOD_LENGTH );
    if( error != SND_PERIOD_LENGTH )
    {
      // Try to recover from error
      fprintf( stderr, "hermes2: Signal_Sample(): %s\n", snd_strerror((int)error) );
      if( !Xrun_Recovery(capture_handle, (int)error) )
        return( NULL );
    } // if( error  )

    // Generates a single tone for testing only
/*    static double dW1 = M_2PI * 1500 / 48000;  // Delta Omega of first signal of 300Hz
    static double  A1 = 32000;                 // Amplitude of first signal
    static double  f1 = 0;
    int k = hermes2_rc.use_snd_chan;
    for( int i = 0; i < recv_buf_size / 2; i++ )
    {
      recv_buffer[thrd_idx][k] = (short)( A1 * sin(f1) );
      k  += SND_NUM_CHANNELS;
      f1 += dW1; CLAMP_2PI(f1);
    }
*/

    // Increment and control thread ring buffer index
    thrd_idx++;
    if( thrd_idx >= NUM_SND_RING_BUFS ) thrd_idx = 0;
  } // while( Flag[HERMES2_CAPTURE_SETUP] )

  return( NULL );
} // Readi_Thread()

//----------------------------------------------------------------------

/* Open_Capture()
 *
 * Opens sound card for Capture
 */
  BOOLEAN
Open_Capture( void )
{
  char err_mesg[MESG_STRING_SIZE];
  err_mesg[0] = '\0';


  // Return if Capture is setup
  if( Flag[HERMES2_CAPTURE_SETUP] ) return( TRUE );

  // Open & setup pcm for Capture
  if( !Open_PCM( &capture_handle, SND_PCM_STREAM_CAPTURE, err_mesg) )
   {
    char mesg[ MESG_STRING_SIZE ];
    snprintf( mesg, MESG_STRING_SIZE,
        _("%s\nCreate a new configuration file for hermes2?"), err_mesg );
    Message_Dialog( mesg );
    Flag[HERMES2_CREATE_CONFIG] = True;
    Flag[HERMES2_CAPTURE_SETUP] = False;
    return( False );
  }

  // Size of receive samples buffer in 'shorts'
  recv_buf_size = SND_PERIOD_LENGTH * SND_NUM_CHANNELS;

  // Index to recv samples buffer (set to end)
  recv_buf_idx = recv_buf_size;

  // Allocate memory to receive samples buffers
  for( uint8_t idx = 0; idx < NUM_SND_RING_BUFS; idx++ )
  {
    recv_buffer[idx] = NULL;
    Mem_Alloc( (void **)&recv_buffer[idx], (size_t)recv_buf_size * sizeof(short) );
    memset( recv_buffer[idx], 0, (size_t)recv_buf_size * sizeof(short) );
  }

  // Open Mixer, abort on failure
  if( !Open_Mixer(&mixer_handle, err_mesg) )
  {
    char mesg[ MESG_STRING_SIZE ];
    snprintf( mesg, MESG_STRING_SIZE,
        _("%s\nCreate a new configuration file for hermes2?"), err_mesg );
    Message_Dialog( mesg );
    Flag[HERMES2_CREATE_CONFIG] = True;
    Flag[HERMES2_CAPTURE_SETUP] = False;
    return( FALSE );
  }

  // Set Capture level. Failure is not considered fatal
  if( !Set_Capture_Level(hermes2_rc.capture_level, err_mesg) )
    Message_Dialog( err_mesg );

  // Flag needed in the Readi_Thread()
  Flag[HERMES2_CAPTURE_SETUP] = True;

  // Start snd_pcm_readi() thread
  if( !Pthread_Create(
        &snd_readi_thrd, NULL, Readi_Thread, NULL,
        "Failed to create Readi_Thread() thread") )
  {
    Flag[HERMES2_CAPTURE_SETUP] = False;
    return( False );
  }

  return( TRUE );
} // Open_Capture()

//------------------------------------------------------------------------

/* Signal_Sample()
 *
 * Gets the next DSP sample of the signal input.
 */
  BOOLEAN
Signal_Sample( short *sample_val )
{
  static uint8_t use_idx = 0;

  // Refill recv DSP samples buffer when needed
  if( recv_buf_idx >= recv_buf_size )
  {
    // Start buffer index according to stereo/mono mode
    recv_buf_idx = hermes2_rc.use_snd_chan;

    // Increment and control in-use ring buffer index
    use_idx++;
    if( use_idx >= NUM_SND_RING_BUFS ) use_idx = 0;
  } // End of if( recv_buf_idx >= recv_buf_size )

  // Take audio sample and increment according to mono/stereo mode
  *sample_val   = recv_buffer[use_idx][recv_buf_idx];
  recv_buf_idx += SND_NUM_CHANNELS;

  return( TRUE );
} // End of Signal_Sample()

//------------------------------------------------------------------------

// Close Mic Capture
  void
Close_Capture( void )
{
  // Stop Readi_Thread() and wait for it to exit
  Flag[HERMES2_CAPTURE_SETUP] = False;
  pthread_join( snd_readi_thrd, NULL );

  if( capture_handle != NULL )
    snd_pcm_close( capture_handle );
  capture_handle = NULL;

  if( hw_params != NULL )
    snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

} // Close_Capture()

/*------------------------------------------------------------------------*/

/* Open_Mixer()
 *
 * Opens Mixer interface
 */
  BOOLEAN
Open_Mixer( snd_mixer_t **handle, char *mesg )
{
  snd_mixer_elem_t     *elem;
  snd_mixer_selem_id_t *sid;
  int error;

  // Abort if mixer already setup
  if( Flag[HERMES2_MIXER_SETUP] ) return( TRUE );
  Flag[HERMES2_MIXER_SETUP] = False;

  // Open mixer handle
  error = snd_mixer_open( &mixer_handle, 0 );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot open Mixer handle"), MESG_STRING_SIZE );
    return( False );
  }

  // Attach mixer
  error = snd_mixer_attach( mixer_handle, hermes2_rc.hwctl_device );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot attach Mixer to PCM device"), MESG_STRING_SIZE );
    return( False );
  }

  // Register Mixer
  error = snd_mixer_selem_register( mixer_handle, NULL, NULL );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot register Mixer"), MESG_STRING_SIZE );
    return( False );
  }

  // Load Mixer
  error = snd_mixer_load( mixer_handle );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot load Mixer"), MESG_STRING_SIZE );
    return( False );
  }

  // Allocate selem_id structure
  error = snd_mixer_selem_id_malloc( &sid );
  if( error < 0 )
  {
    Strlcat( mesg, _("Cannot allocate selem_id struct"), MESG_STRING_SIZE );
    return( False );
  }

  // Find capture source selem
  snd_mixer_selem_id_set_index( sid, 0 );
  snd_mixer_selem_id_set_name( sid, hermes2_rc.capture_src );
  elem = snd_mixer_find_selem( mixer_handle, sid );
  if( elem == NULL )
  {
    snd_mixer_selem_id_free( sid );
    Strlcat( mesg, _("Cannot find Capture source element"), MESG_STRING_SIZE );
    return( False );
  }

  // Set capture switch for capture source
  if( snd_mixer_selem_has_capture_switch(elem) )
  {
    error = snd_mixer_selem_set_capture_switch( elem, hermes2_rc.capture_chan, 1 );
    if( error < 0 )
    {
      snd_mixer_selem_id_free( sid );
      Strlcat( mesg, _("Cannot set Capture source"), MESG_STRING_SIZE );
      return( False );
    }
  }
  else
  {
    snd_mixer_selem_id_free( sid );
    Strlcat( mesg, _("PC Sound does not have Capture capability"), MESG_STRING_SIZE );
    return( FALSE );
  }

  // Find capture volume selem if not --
  if( strcmp(hermes2_rc.capture_vol, "--") != 0 )
  {
    snd_mixer_selem_id_set_index( sid, 0 );
    snd_mixer_selem_id_set_name( sid, hermes2_rc.capture_vol );
    capture_elem = snd_mixer_find_selem( mixer_handle, sid );
    if( !capture_elem )
    {
      Strlcat( mesg, _("Cannot find Volume element"), MESG_STRING_SIZE );
      Error_Dialog( mesg, SHOW_OK );
      return( FALSE );
    }
  }

  snd_mixer_selem_id_free( sid );
  Flag[HERMES2_MIXER_SETUP] = True;
  return( TRUE );

} // Open_Mixer()

//------------------------------------------------------------------------

/* Set_Capture_Level()
 *
 * Sets Capture Control level
 */
  BOOLEAN
Set_Capture_Level( int level, char *mesg )
{
  long cmin, cmax;

  // Abort with no error if Mixer not setup
  if( mixer_handle == NULL ) return( TRUE );

  // Set capture volume
  if( capture_elem != NULL )
  {
    if( snd_mixer_selem_has_capture_volume(capture_elem) )
    {
      // Change from % volume to sound card value
      long lev;
      snd_mixer_selem_get_capture_volume_range( capture_elem, &cmin, &cmax );
      lev = cmin + ((cmax - cmin) * level) / 100;

      // Set capture volume. Channel is 0 for Left, 1 for Right
      uint8_t chn = hermes2_rc.capture_chan & 0x01;
      int error = snd_mixer_selem_set_capture_volume( capture_elem, chn, lev );
      if( error < 0 )
      {
        snprintf( mesg, MESG_STRING_SIZE,
            _("Cannot set capture volume to %d\nError: %s"), level, snd_strerror(error) );
        return( FALSE );
      }
    }
  } // if( capture_elem != NULL )

  return( TRUE );
} // Set_Capture_Level()

//------------------------------------------------------------------------

  void
Close_Mixer( void )
{
  if( mixer_handle != NULL ) snd_mixer_close( mixer_handle );
  mixer_handle = NULL;
  Flag[HERMES2_MIXER_SETUP] = False;
}

//------------------------------------------------------------------------

