/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "process.h"
#include "demodulate.h"
#include "display.h"
#include "sound.h"
#include "spectrum.h"
#include "../common/common.h"
#include "../common/cfft.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../Hermes2/callback_func.h"
#include "../hpsdr/hpsdr_init.h"
#include "../hpsdr/settings.h"
#include <gtk/gtk.h>
#include <math.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>

//----------------------------------------------------------------------

// Number of steps in the Level scale
#define LEVEL_RANGE      90.0 // LEVEL_STEP * NUM_LEVEL_STEPS

//----------------------------------------------------------------------

// Pointer to Time Station receiver function
void (*Receive_Time_Station)( Transceiver_t *TRx );

//----------------------------------------------------------------------

/* Hermes2_Run()
 *
 * Controls the flow of hermes2 in its own thread
 */
  void
Hermes2_Run( Transceiver_t *TRx )
{
  // Low pass filter I and Q data
  DSP_Filter( &(TRx->roof_filter_i) );
  DSP_Filter( &(TRx->roof_filter_q) );

  // Display FFT Spectrum and Signal Monitors
  if( TRx->spectrum_data.receiver_spectrum &&
      TRx->spectrum_data.spectrum_init )
  {
    static uint32_t
      fft_in_idx = 0,         // Index to FFT input buffers
      samples_buffer_idx = 0; // Incoming IQ samples buffer index

    // Decimation of IQ samples for the FFT
    uint32_t fft_decim;

    // Samples decimation factor for different FFT bandwidths
    fft_decim = hermes2_rc.ddc_sample_rate / TRx->spectrum_data.fft_bandwidth;

    // Make a copy of filtered samples for the FFT
    // Continue while FFT buffers not yet full
    samples_buffer_idx = 0;
    while( (samples_buffer_idx < hermes2_rc.ddc_buffer_length) &&
           (TRx->spectrum_data.fft_in_count < TRx->spectrum_data.fft_in_length) )
    {
      if( (samples_buffer_idx % fft_decim) == 0 )
      {
        // Clear fft input buffer
        fft_in_idx = TRx->spectrum_data.fft_in_count;
        TRx->spectrum_data.fft_in_i[fft_in_idx] = 0.0;
        TRx->spectrum_data.fft_in_q[fft_in_idx] = 0.0;
      } // if( (samples_buffer_idx % fft_decim) == 0 )

      // Enter FFT IQ data by summating (decimating) input samples
      TRx->spectrum_data.fft_in_i[fft_in_idx] +=
        TRx->roof_filter_i.samples_buffer[samples_buffer_idx];
      TRx->spectrum_data.fft_in_q[fft_in_idx] +=
        TRx->roof_filter_q.samples_buffer[samples_buffer_idx];
      samples_buffer_idx++;

      // Normalize FFT input buffers at fft_decim counts
      if( (samples_buffer_idx % fft_decim) == 0 )
      {
        // Normalize FFT input data
        TRx->spectrum_data.fft_in_i[fft_in_idx] /= (double)fft_decim;
        TRx->spectrum_data.fft_in_q[fft_in_idx] /= (double)fft_decim;

        /* Increment FFT data in count. When full, it
         * signals the display of the Spectrum Monitors */
        TRx->spectrum_data.fft_in_count++;

      } // if( (samples_buffer_idx % fft_decim) == 0 )
    } // while( samples_buffer_idx < hermes2_rc.ddc_buffer_length
  } // if( TRx->spectrum_status && Flag[SPECTRUM_INIT]

  // Decode time and date from Time Stations
  if( TRx->timedec_active )
  {
    if( Receive_Time_Station != NULL )
      Receive_Time_Station( TRx );
  }

  // Stream data from Device and demodulate
  if( TRx->receive_active && !TRx->receiver_muted )
  {
    // Call appropriate demodulator function
    if( !TRx->Demodulator(TRx, SOUND_OP_BUF_SIZE) )
      pthread_exit( NULL );

    // Calculate relative signal strength for ADAGC
    Hpsdr_ADAGC_Smeter( TRx );

    // Display tuner gain and signal level bar
    gdk_threads_add_idle( Hermes2_Display_S_meter, TRx );
  } // if( Flag[SOUND_OP_START] )

} // Hermes2_Run()

//----------------------------------------------------------------------

/* Hermes2_Stop()
 *
 * Runs down and stops every running process of Hermes2
 * and waits for every relevant running thread to exit
 */
  void
Hermes2_Stop( void )
{
  // Repeat for every open Transceiver window and instance
  for( int8_t idx = (int8_t)Indices.Num_of_TRXs - 1; idx >= 0; idx-- )
  {
    Transceiver_t *TRx = Transceiver[idx];

    // Signal Guest Rx or Tx threads to quit
     Flag[GUEST_QUIT] = True;

    // Disable Transmitter
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->mox_togglebutton), FALSE );
    while( g_main_context_iteration(NULL, FALSE) );

    // Disable Spectrum monitors
    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON( TRx->spectrum_off_radiobutton), TRUE );
    while( g_main_context_iteration(NULL, FALSE) );

    // Disable Receiving
    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON( TRx->startrx_togglebutton), FALSE );
    gtk_toggle_button_set_active(
        GTK_TOGGLE_BUTTON( TRx->afc_checkbutton), FALSE );

    // Wait for GTK to exit from any idle added threads
    while( g_main_context_iteration(NULL, FALSE) );

    /* Post to semaphore to prevent blocking
     * and wait for AFC thread if running */
    sem_post( &(TRx->afc_semaphore) );

    // Terminate RSID if running
    TRx->rx_rsid_enable = False;
    TRx->tx_rsid_enable = False;

    // Post to  RSID semaphore to free it and join RSID thread
    int sval;
    sem_getvalue( &rsid_semaphore, &sval );
    if( !sval ) sem_post( &rsid_semaphore );
    if( TRx->rx_rsid_running )
      pthread_join( TRx->rsid_thread, NULL );

    /* Post to semaphore to prevent blocking
     * and wait for RSID thread if running */
    sem_post( &rsid_semaphore );

  } // for( idx = (int8_t)Num_of_TRXs - 1; idx >= 0; idx-- )

} // Hermes2_Stop()

//----------------------------------------------------------------------

/* Sound_Write_Async()
 *
 * Thread to asynchronously write sound buffer for replay
 */
  void *
Sound_Write_Async( void *arg )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  Flag[HERMES2_PCSOUND_RUNNING] = True;
  while( Flag[HERMES2_PCSOUND_SETUP] )
  {
    // Write Demodulator buffer to sound device if not muted
    if( !TRx->receiver_muted )
    {
      if( !DSP_Write(sound_ring_buf[sound_ring_buf_idx], SOUND_OP_BUF_SIZE) )
        break;
    }
    else usleep( 42000 ); // Sleep about the time it takes to play a sound buffer

    // Advance sound startrx buffer index and keep in limits
    sound_ring_buf_idx++;
    if( sound_ring_buf_idx >= NUM_SOUND_RING_BUFFS )
      sound_ring_buf_idx = 0;
  } // while( Flag[HERMES2_PCSOUND_SETUP] )

  Flag[HERMES2_PCSOUND_RUNNING] = False;
  return( NULL );

} // Sound_Write_Async()

//----------------------------------------------------------------------

/* FFT_of_DDC()
 *
 * FFT function. It covers positive and negative
 * frequencies either side of the bandwidth center.
 */
  static void
FFT_of_DDC( spectrum_data_t *spectrum_data )
{
  uint32_t
    op_bin_idx,  // Index to FFT decimated ouput bin
    fft_bin_mid, // Index to middle of FFT's output bin
    fft_bin_idx; // Index to FFT's output bins

  double val_max = -1000.0, magn;

  // Value of FFT decimated output bins
  static double op_bin_val[MAX_RECEIVERS] = { 200, 200, 200, 200 };

  // Maximum value of fft bins
  static double bin_max[MAX_RECEIVERS];

  // Transceiver object index
  uint8_t rx_idx = spectrum_data->trx_index;

  // Perform FFT on input samples
  cFFT( spectrum_data->fft_in_length, spectrum_data->fft_order,
      spectrum_data->fft_in_i, spectrum_data->fft_in_q );

  // Calculate output FFT bins for positive frequencies
  op_bin_idx  = 0;
  fft_bin_mid = spectrum_data->fft_out_length / 2 + 1;
  fft_bin_idx = fft_bin_mid;
  while( fft_bin_idx-- )
  {
    // Calculate signal power at each frequency (output bin)
    spectrum_data->fft_in_i[fft_bin_idx] /= spectrum_data->fft_scale;
    spectrum_data->fft_in_q[fft_bin_idx] /= spectrum_data->fft_scale;
    magn = hypot(
        spectrum_data->fft_in_i[fft_bin_idx],
        spectrum_data->fft_in_q[fft_bin_idx] );

    // Keep previous value if magnitude is zero
    if( magn != 0.0 ) op_bin_val[rx_idx] = 20.0 * log10( magn );

    // Record maximum bin value
    if( val_max < op_bin_val[rx_idx] ) val_max = op_bin_val[rx_idx];

    // Offset bin values to below max bin value (0 dBm relative)
    op_bin_val[rx_idx] -= bin_max[rx_idx];

    // Maintain bin values in range 0 to -range (100) dBm
    if( op_bin_val[rx_idx] > 0.0 )
      op_bin_val[rx_idx] = 0.0;
    else if( op_bin_val[rx_idx] < -LEVEL_RANGE )
      op_bin_val[rx_idx] = -LEVEL_RANGE;

    // Convert bin values to clear height in Spectrum Monitor
    spectrum_data->fft_bin_values[op_bin_idx] =
      (int)( (LEVEL_RANGE + op_bin_val[rx_idx]) / LEVEL_RANGE *
          (double)hermes2_rc.scope_clear_height );

    op_bin_idx++;
  } // while( fft_bin_idx-- )

  // Calculate output FFT bins for negative frequencies
  fft_bin_idx = spectrum_data->fft_out_length;
  while( fft_bin_idx > fft_bin_mid )
  {
    // Calculate signal power at each frequency (output bin)
    fft_bin_idx--;
    spectrum_data->fft_in_i[fft_bin_idx] /= spectrum_data->fft_scale;
    spectrum_data->fft_in_q[fft_bin_idx] /= spectrum_data->fft_scale;
    magn = hypot(
        spectrum_data->fft_in_i[fft_bin_idx],
        spectrum_data->fft_in_q[fft_bin_idx] );

    // Keep previous value if magnitude is zero
    if( magn != 0.0 ) op_bin_val[rx_idx] = 20.0 * log10( magn );

    // Record maximum bin value
    if( val_max < op_bin_val[rx_idx] ) val_max = op_bin_val[rx_idx];

    // Offset bin values to below max bin value (0 dBm relative)
    op_bin_val[rx_idx] -= bin_max[rx_idx];

    // Maintain bin values in range 0 to -range (100) dBm
    if( op_bin_val[rx_idx] > 0.0 )
      op_bin_val[rx_idx] = 0.0;
    else if( op_bin_val[rx_idx] < -LEVEL_RANGE )
      op_bin_val[rx_idx] = -LEVEL_RANGE;

    // Convert bin values to clear height in Spectrum Monitor
    spectrum_data->fft_bin_values[op_bin_idx] =
      (int)( (LEVEL_RANGE + op_bin_val[rx_idx]) / LEVEL_RANGE *
          (double)hermes2_rc.scope_clear_height );

    op_bin_idx++;
  } // for( fft_bin_idx = 0; fft_bin_idx < fft_bin_mid; fft_bin_idx++ )

  // Reset bin_max variable
  bin_max[rx_idx] = val_max;

} // FFT_of_DDC()

//----------------------------------------------------------------------

/* FFT_of_Wideband()
 *
 * FFT function. It covers positive and negative
 * frequencies either side of the bandwidth center.
 */
  static void
FFT_of_Wideband( spectrum_data_t *spectrum_data )
{
  // Index to FFT's bins
  uint32_t fft_bin_idx;

  double val_max = -1000.0;

  // Value of FFT decimated output bins
  static double op_bin_val[MAX_RECEIVERS] = { 200, 200, 200, 200 };

  // Maximum value of fft bins
  static double bin_max[MAX_RECEIVERS];

  // Transceiver object index
  uint8_t rx_idx = spectrum_data->trx_index;

  // Perform FFT on input samples
  cFFT( spectrum_data->fft_in_length, spectrum_data->fft_order,
      spectrum_data->fft_in_i, spectrum_data->fft_in_q );

  // Calculate output FFT bins, ignore 'DC' first term (index 0)
  spectrum_data->fft_bin_values[0] = hermes2_rc.scope_clear_height / 2;
  for( fft_bin_idx = 1;
      fft_bin_idx < spectrum_data->fft_out_length;
      fft_bin_idx++ )
  {
    // Calculate signal power at each frequency (output bin)
    spectrum_data->fft_in_i[fft_bin_idx] /= spectrum_data->fft_scale;
    spectrum_data->fft_in_q[fft_bin_idx] /= spectrum_data->fft_scale;
    double magn = hypot(
        spectrum_data->fft_in_i[fft_bin_idx],
        spectrum_data->fft_in_q[fft_bin_idx] );

    // Keep previous value if magnitude is zero
    if( magn != 0.0 ) op_bin_val[rx_idx] = 20.0 * log10( magn );

    // Record maximum bin value
    if( val_max < op_bin_val[rx_idx] )
      val_max = op_bin_val[rx_idx];

    // Offset bin values to below max bin value (0 dBm relative)
    op_bin_val[rx_idx] -= bin_max[rx_idx];

    // Maintain bin values in range 0 to -range (100) dBm
    if( op_bin_val[rx_idx] > 0.0 )
      op_bin_val[rx_idx] = 0.0;
    else if( op_bin_val[rx_idx] < -LEVEL_RANGE )
      op_bin_val[rx_idx] = -LEVEL_RANGE;

    // Convert bin values to clear height in Spectrum Monitor
    spectrum_data->fft_bin_values[fft_bin_idx] =
      (int)( (LEVEL_RANGE + op_bin_val[rx_idx]) / LEVEL_RANGE *
          (double)hermes2_rc.scope_clear_height );

  } // for( fft_bin_idx = fft_bin_mid; ...

  // Reset bin_max variable
  bin_max[rx_idx] = val_max;

} // FFT_of_Wideband()

//----------------------------------------------------------------------

/* Paint_Waterfall()
 *
 * Paints the FFT Spectrum as a Waterfall in waterfall_pixbuf.
 * This function is run in its own thread while the Spectrum
 * togglebutton is active. It is also responsible for carrying
 * out the FFT on incoming IQ data prepared by Hermes2_Run()
 */
  void *
Paint_Waterfall( void *data )
{
  uint32_t fft_idx; // Index to fft output array

  int
    vert_lim,  // Limit of vertical index for copying lines
    idh, idv,  // Index to hor. and vert. position in Spectrum display
    pixel_val; // Greyscale value of pixel derived from fft o/p

  // Pointer to current pixel
  static guchar *pix;

  // Spectrum data buffer
  spectrum_data_t *spectrum_data = (spectrum_data_t *)data;

  // Loop while Spectrum display is enabled
  spectrum_data->spectrum_running = True;
  while( spectrum_data->spectrum_active && spectrum_data->spectrum_init )
  {
    // If not initialized wait
    if( !hermes2_rc.scope_clear_height )
    {
      usleep( 10000 );
      continue;
    }

    // Do the FFT on DDC or Wideband samples input
    if( spectrum_data->receiver_spectrum )
      FFT_of_DDC( spectrum_data );
    else if( spectrum_data->wideband_spectrum )
      FFT_of_Wideband( spectrum_data );

    // Copy each line of spectrum to next one
    vert_lim = hermes2_rc.waterfall_height - 2;
    for( idv = vert_lim; idv > 0; idv-- )
    {
      pix =
        spectrum_data->waterfall_pixels +
        hermes2_rc.waterfall_rowstride * idv;

      for( idh = 0; idh < (int)(spectrum_data->fft_out_length); idh++ )
      {
        pix[0] = pix[-hermes2_rc.waterfall_rowstride];
        pix[1] = pix[1-hermes2_rc.waterfall_rowstride];
        pix[2] = pix[2-hermes2_rc.waterfall_rowstride];
        pix += hermes2_rc.waterfall_n_channels;
      }
    } // for( idv = vert_lim; idv > 0; idv-- )

    // Go to top left +1 hor. +1 vert. of pixbuf
    pix = spectrum_data->waterfall_pixels + hermes2_rc.waterfall_rowstride;

    // Calculate pixel values after FFT
    for( fft_idx = 0; fft_idx < spectrum_data->fft_out_length; fft_idx++ )
    {
      // Color code signal strength (rescale to max = 255 )
      pixel_val  = spectrum_data->fft_bin_values[fft_idx] * 255;
      pixel_val /= hermes2_rc.scope_clear_height;
      Colorize( pix, (uint8_t)pixel_val );
      pix += hermes2_rc.waterfall_n_channels;
    } // for( fft_idx = 0; fft_idx < buf->fft_width; fft_idx++ )

    // Draw Spectrum waterfall and Signal Monitors
    if( spectrum_data->spectrum_drawingarea )
      Queue_Draw( spectrum_data->spectrum_drawingarea );
    spectrum_data->fft_in_count = 0;

    // FFT frame refresh rate
    usleep( 1000000 / spectrum_data->fft_frame_rate );

  } // while( spectrum_status && ... )

  spectrum_data->spectrum_running = False;
  return( NULL );

} // Paint_Waterfall()

//----------------------------------------------------------------------

