/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "modulate.h"
#include "sound.h"
#include "callback_func.h"
#include "../common/common.h"
#include "../common/convert.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/pc_to_hw.h"
#include "../hpsdr/hw_to_pc.h"
#include "../olivia/mfsk.h"
#include <gtk/gtk.h>
#include <math.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

// Blackman-Harris Window constants
#define BHW_A0   0.35875
#define BHW_A1   0.48829
#define BHW_A2   0.14128
#define BHW_A3   0.01168

/* Length of Blackman-Harris Window's rising or falling
 * edge as a multiple of DUC packet length. Overall Window
 * length is 8 packets so that each edge is 5 mSec long. */
#define BHW_EDGE_LENGTH   4

//----------------------------------------------------------------------

// Microphone Compressor O/P peak value reference
#define MIC_COMP_PEAK   0x7F00

//----------------------------------------------------------------------

// Filter poles must be even
#define TX_SSB_FILTER_POLES_4    4
#define TX_SSB_FILTER_POLES_6    6
#define TX_SSB_FILTER_RIPPLE     1.0
#define TX_SSB_FILTER_NARROW     500

//----------------------------------------------------------------------

static int16_t
duc_buf_i[DUC_NUM_IQ_SAMPLES],  // Buffer for I samples
duc_buf_q[DUC_NUM_IQ_SAMPLES];  // Buffer for Q samples

// Sidetone samples buffer
static int16_t *samples_buf = NULL;

//----------------------------------------------------------------------

/* Clear_DUC()
 *
 * Clears the hardware DUC I/Q samples buffer
 */
  void
Clear_DUC( void )
{
  // Fill the USB frame I/Q buffer with zero's to clear DUC
  uint16_t buf_idx1 = 16, buf_idx2 = 528;
  memset( &Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx1], 0, 504 );
  memset( &Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx2], 0, 504 );
} // Clear_DUC()

//----------------------------------------------------------------------

/* Generate_CW()
 *
 * Generates a pure CW signal by filling the DUC's I buffer with a
 * constant DUC_SAMPLE_MAX value and the Q buffer with 0. Leading
 * and trailing edges are ramped using a Blackman-Harris window.
 */
  BOOLEAN
Generate_CW( BOOLEAN up_flag )
{
  static uint8_t
    bh_cnt = 0,  // Count of Blackman-Harris buffer sections used
    iq_cnt = 0;  // Count of DUC IQ samples added to buffer

  // Signal initialization of function on first call
  static BOOLEAN init = True, done = True;

  // Carrier wave samples = DUC_SAMPLE_MAX
  static int16_t cwav_buf[DUC_NUM_IQ_SAMPLES];

  // Blackman-Harris Window buffer
  static int16_t BHw[2 * BHW_EDGE_LENGTH * DUC_NUM_IQ_SAMPLES];


  // Fill in DUC buffers on first call
  if( init )
  {
    // Prepare IQ buffers for steady Carrier at full level
    for( uint16_t idx = 0; idx < DUC_NUM_IQ_SAMPLES; idx++ )
    {
      /* We send a "DC" or zero IF sample stream to
       * the DUC since we want just a pure CW signal */
      cwav_buf[idx] = IQ_SAMPLE_MAX;
    }

    // Prepare a buffer for the Blackman-Harris window
    uint16_t N = 2 * BHW_EDGE_LENGTH * DUC_NUM_IQ_SAMPLES;
    for( uint16_t n = 0; n < N; n++ )
    {
      double w, a;
      a = M_2PI * (double)n / N;
      w =
        BHW_A0 -
        BHW_A1 * cos( a ) +
        BHW_A2 * cos( 2.0 * a ) -
        BHW_A3 * cos( 4.0 * a );
      BHw[n] = (int16_t)( w * IQ_SAMPLE_MAX );
    }

    init = False;
  } // if( init )

  for( uint8_t mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; mic_cnt++ )
  {
    // DUC samples buffers
    const static int16_t *duc_i, *duc_q;

    // Generate carrier
    if( up_flag )
    {
      done = False;
      iq_cnt++;
      if( iq_cnt >= DUC_NUM_IQ_SAMPLES )
      {
        /* Point the I buffer to the B-H buffer
         * till all sections are sent to DUC */
        if( bh_cnt < BHW_EDGE_LENGTH )
        {
          duc_i = &BHw[bh_cnt * DUC_NUM_IQ_SAMPLES];
          duc_q = duc_i;
          bh_cnt++;
        }
        else // After the rising edge, send steady carrier samples
        {
          duc_i = cwav_buf;
          duc_q = duc_i;
        }

        // Pack the SSB sample stream into the C&C USB Buffer
        Pack_DUC_Tx_Data( duc_i, duc_q, duc_i, duc_q );
        iq_cnt = 0;
      } // if( iq_cnt >= DUC_NUM_IQ_SAMPLES )
    } // if( up_flag )
    else if( !done )
    {
      iq_cnt++;
      if( iq_cnt >= DUC_NUM_IQ_SAMPLES )
      {
        // Send the last (falling) sections of the B-H window
        if( bh_cnt < 2 * BHW_EDGE_LENGTH )
        {
          duc_i = &BHw[bh_cnt * DUC_NUM_IQ_SAMPLES];
          duc_q = duc_i;
          bh_cnt++;
        } // if( bh_cnt < 2 * BHW_EDGE_LENGTH )
        else // Ramp down finished
        {
          // Clear DUC buffers when finished
          Clear_DUC();
          bh_cnt = 0;
          done = True;
        } // else of if( bh_cnt < 2 * BHW_EDGE_LENGTH )

        // Pack the SSB sample stream into the C&C USB Buffer
        Pack_DUC_Tx_Data( duc_i, duc_q, duc_i, duc_q );

        iq_cnt = 0;
      } // if( iq_cnt >= DUC_NUM_IQ_SAMPLES )
    } // else of if( up_flag )
  } // for( uint8_t mic_cnt = 0; mic_cnt < MIC_NUM_SAMPLES; mic_cnt++ )

  return( True );
} // Generate_CW()

//----------------------------------------------------------------------

/* Modulate_CW()
 *
 * Modulates the Transmitter with Morse
 * Code keyed by a manual straight keyer
 */
  void
Modulate_CW( void *data )
{
  // Cleanup of hardware buffers done
  static BOOLEAN done = False;
  //const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // CW keyer state
  uint8_t keyer = ddv->dot_dash_ptt & (DOT_REPORT | DASH_REPORT);

  // Keyer ON, generate CW and Side tone
  if( keyer )
  {
    if( !Generate_CW(True) ) Modulator = NULL;
    done = False;
  } // if( keyer_status )
  else if( !done )
  {
    // Send CW ramp down samples to the DUC
    if( !Generate_CW(False) ) Modulator = NULL;
    done = True;
  } // if( keyer_status )

} // Modulate_CW()

//----------------------------------------------------------------------

/* Modulate_SSB()
 *
 * Processes Microphone samples to produce SSB modulation
 */
  void
Modulate_SSB( void *data )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  static double
    weaver_buf_i[DUC_NUM_IQ_SAMPLES],  // Buffer for weaver I samples
    weaver_buf_q[DUC_NUM_IQ_SAMPLES];  // Buffer for weaver Q samples

  // Modulator Filter data
  static filter_data_t mod_filter_data_i, mod_filter_data_q;

  // DUC samples buffer cleared
  static BOOLEAN cleared  = False;

  // Saves current modulation mode
  static uint8_t mode = 100;

  // Saved values of modulator bandwidth and weaver frequency
  static uint32_t bwidth = 0, weaver = 1;

  // Delta of Weaver method phase angle per IQ sample of 48kHz
  static double dphi = 0.0;

  // Weaver method phase angle for SSB modulator
  static double phi_wvr = 0.0;

  const double *mic_data = (double *)data;


  // Abort if no weaver frequency selected
  if( !TRx->tx_weaver_frequency ) return;

  // Clear DUC buffer and return if no PTT
  if( !(ddv->dot_dash_ptt & PTT_REPORT) && !ddv->transmit_on )
  {
    if( !cleared )
    {
      // Send zero IQ samples packet to DUC to clear
      Clear_DUC();
      cleared = True;
    }
    return;
  } // if( !(hermes2_rc.dot_dash_ptt & PTT_REPORT) && ...

  // Initialize on first call or change of params
  if( (mode   != TRx->tx_modulation_mode)  ||
      (weaver != TRx->tx_weaver_frequency) )
  {
    mode   = TRx->tx_modulation_mode;
    weaver = TRx->tx_weaver_frequency;

    /* Calculate Weaver phase increment. "Negative" Weaver
     * frequency is used for lower sideband modulation */
    switch( mode )
    {
      case TX_MODE_USBW:    case TX_MODE_USBM:    case TX_MODE_USBN:
      case TX_MODE_32_1000: case TX_MODE_16_1000: case TX_MODE_16_500:
      case TX_MODE_8_500:   case TX_MODE_8_250:   case TX_MODE_4_250:
      case TX_MODE_4_125:   case TX_MODE_FTX:
        dphi = M_2PI * (double)TRx->tx_weaver_frequency / DUC_SAMPLE_RATE;
        break;

      case TX_MODE_LSBW: case TX_MODE_LSBM: case TX_MODE_LSBN:
        dphi = -M_2PI * (double)TRx->tx_weaver_frequency / DUC_SAMPLE_RATE;
        break;
    }

    return;
  } // if( (mode != TRx->t=p-[x_modulation_mode) || ...

  // Prepare LP filters on B/W change
  if( bwidth != TRx->tx_bandwidth )
  {
    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    double cutoff;
    bwidth  = TRx->tx_bandwidth;
    cutoff  = (double)bwidth / 2.0 / DUC_SAMPLE_RATE;

    // Initialize Modulator I filter
    mod_filter_data_i.cutoff   = cutoff;
    mod_filter_data_i.ripple   = TX_SSB_FILTER_RIPPLE;
    if( bwidth < TX_SSB_FILTER_NARROW )
      mod_filter_data_i.npoles = TX_SSB_FILTER_POLES_4;
    else
      mod_filter_data_i.npoles = TX_SSB_FILTER_POLES_6;
    mod_filter_data_i.type     = FILTER_LOWPASS;
    mod_filter_data_i.ring_idx = 0;
    mod_filter_data_i.samples_buffer = weaver_buf_i;
    mod_filter_data_i.samples_buffer_len = DUC_NUM_IQ_SAMPLES;
    mod_filter_data_i.init_filter = True;

    // Initialize Modulator Q filter
    mod_filter_data_q.cutoff   = cutoff;
    mod_filter_data_q.ripple   = TX_SSB_FILTER_RIPPLE;
    if( bwidth < TX_SSB_FILTER_NARROW )
      mod_filter_data_q.npoles = TX_SSB_FILTER_POLES_4;
    else
      mod_filter_data_q.npoles = TX_SSB_FILTER_POLES_6;
    mod_filter_data_q.type     = FILTER_LOWPASS;
    mod_filter_data_q.ring_idx = 0;
    mod_filter_data_q.samples_buffer = weaver_buf_q;
    mod_filter_data_q.samples_buffer_len = DUC_NUM_IQ_SAMPLES;
    mod_filter_data_q.init_filter = True;

    return;
  } // if( bwidth != TRx->tx_bandwidth )

  // For all microphone samples in packet (2x63 sample pairs)
  for( uint8_t mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; mic_cnt++ )
  {
    // Generates test tones, debugging only
    /*(static double dM1 = M_2PI * 1550 / DUC_SAMPLE_RATE;
    static double M1  = 0, m1;
    static double dM2 = M_2PI * 620 / DUC_SAMPLE_RATE;
    static double M2  = 0, m2;
    m1  = 0.8 * S16_SCALE * sin( M1 );
    M1 += dM1;
    CLAMP_2PI( M1 );
    m2  = 0.8 * S16_SCALE * sin( M2 );
    M2 += dM2;
    CLAMP_2PI( M2 );
    weaver_buf_i[mic_cnt] = ( m1 + m2 ) * cos( phi_wvr );
    weaver_buf_q[mic_cnt] = ( m1 + m2 ) * sin( phi_wvr );*/

    // Generate SSB signal with the Weaver method
    weaver_buf_i[mic_cnt] = mic_data[mic_cnt] * cos( phi_wvr );
    weaver_buf_q[mic_cnt] = mic_data[mic_cnt] * sin( phi_wvr );

    // Step and limit Weaver phase angle
    phi_wvr += dphi;
    CLAMP_N2PI( phi_wvr );
  } // for( mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; ...

  // Low-pass filter AFC samples buffers
  DSP_Filter( &mod_filter_data_i );
  DSP_Filter( &mod_filter_data_q );

  /* Save float format IQ samples to int16_t for the DUC. The filtered
   * SSB signal needs to be doubled in amplitude to give full power output */
  for( uint8_t idx = 0; idx < DUC_NUM_IQ_SAMPLES; idx++ )
  {
    duc_buf_i[idx] = (int16_t)( weaver_buf_i[idx] * 2 );
    duc_buf_q[idx] = (int16_t)( weaver_buf_q[idx] * 2 );
  }

  // Pack the SSB sample stream into the C&C USB Buffer
  Pack_DUC_Tx_Data( duc_buf_i, duc_buf_q, duc_buf_i, duc_buf_q );
  cleared = False;

} // Modulate_SSB()

//----------------------------------------------------------------------

// Ratio of Max DUC sample value to AM Carrier value
#define AM_CARRIER_SCALE    2

/* Modulate_AM()
 *
 * Processes Microphone samples to produce AM modulation
 */
  void
Modulate_AM( void *data )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // The DUC sample size for the AM carrier with no mod
  static int16_t am_carrier;

  static BOOLEAN
    init    = True,   // Initialize on first call
    cleared = False;  // DUC samples buffer cleared

  const double *mic_data = (double *)data;

  // Initialize on first call or change of parameters
  if( init )
  {
    memset( duc_buf_q, 0, DUC_NUM_IQ_SAMPLES * sizeof(int16_t) );
    am_carrier = DUC_SAMPLE_MAX / AM_CARRIER_SCALE;
    init       = False;
    return;
  }

  // Clear DUC buffer and return if no PTT
  if( !(ddv->dot_dash_ptt & PTT_REPORT) && !ddv->transmit_on )
  {
    if( !cleared )
    {
      // Send zero IQ samples packet to DUC to clear
      Clear_DUC();
      cleared = True;
    }
    return;
  } // if( !(hermes2_rc.dot_dash_ptt & PTT_REPORT) && ...

  // For all microphone samples in packet (2x63 sample pairs)
  for( uint8_t mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; mic_cnt++ )
  {
    // Generates a test tone, for testing only
    /* static double dM = M_2PI * 1050 / DUC_SAMPLE_RATE;
    static double M  = 0, m;
    m  = am_carrier * sin(M);
    M += dM;
    CLAMP_2PI( M );
    duc_buf_i[mic_cnt] = am_carrier + (int16_t)m;
    duc_buf_q[mic_cnt] = 0; */

    // Generate IQ samples in proportion to mic data
    duc_buf_i[mic_cnt] = am_carrier + (int16_t)( mic_data[mic_cnt] / AM_CARRIER_SCALE );
    duc_buf_q[mic_cnt] = 0;

  } // for( mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; ...

  // Pack the AM sample stream into the C&C USB Buffer
  Pack_DUC_Tx_Data( duc_buf_i, duc_buf_q, duc_buf_i, duc_buf_q );
  cleared = False;

} // Modulate_AM()

//----------------------------------------------------------------------

/* Modulate_FM()
 *
 * Processes Microphone samples to produce FM modulation
 */
  void
Modulate_FM( void *data )
{
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // DUC samples buffer cleared
  static BOOLEAN cleared = False;

  // Change of phase angle of FM deviation per unit of Microphone samples level
  static double dW;
  static uint32_t fm_dev = 0;

  const double *mic_data = (double *)data;

  // Initialize of first call or change of parameters
  if( fm_dev != TRx->tx_fm_deviation )
  {
    dW = M_2PI * (double)TRx->tx_fm_deviation / MIC_MAX_VALUE / DUC_SAMPLE_RATE;
    fm_dev = TRx->tx_fm_deviation;
    return;
  }

  // Clear DUC buffer and return if no PTT or MOX
  if( !(ddv->dot_dash_ptt & PTT_REPORT) && !ddv->transmit_on  )
  {
    if( !cleared )
    {
      // Send zero IQ samples packet to DUC to clear
      Clear_DUC();
      cleared = True;
    }
    return;
  } // if( !(hermes2_rc.dot_dash_ptt & PTT_REPORT) && ...

  // For all microphone samples in packet (2x63 sample pairs)
  for( uint8_t mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; mic_cnt++ )
  {
    static double W = 0.0;

    // Generates a test tone, for testing only
    /* static double dM = M_2PI * 1050 / DUC_SAMPLE_RATE;
    static double M  = 0, m;
    m  = MIC_MAX_VALUE * sin(M);
    M += dM;
    CLAMP_2PI( M ); */

    // Generate IQ samples for the instantaneous value of W
    duc_buf_i[mic_cnt] = (int16_t)( S16_SCALE * cos(W) );
    duc_buf_q[mic_cnt] = (int16_t)( S16_SCALE * sin(W) );

    // Vary W according to the microphone level, to produce FM
    W -= mic_data[mic_cnt] * dW;
    CLAMP_N2PI( W );

  } // for( mic_cnt = 0; mic_cnt < MIC_NUM_OF_SAMPLES; ...

  // Pack the SSB sample stream into the C&C USB Buffer
  Pack_DUC_Tx_Data( duc_buf_i, duc_buf_q, duc_buf_i, duc_buf_q );
  cleared = False;

} // Modulate_FM()

//----------------------------------------------------------------------

// Definitions for ascii to Morse code conversion
#define NUM_MORSE_CHARS     55

/* ASCII equivalents to Morse hex code. Last
 * one (*) used for unrecognized characters. */
#define ASCII_CHAR \
{ \
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',  'I', 'J', 'K', \
  'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S',  'T', 'U', 'V', \
  'W', 'X', 'Y', 'Z', '1', '2', '3', '4',  '5', '6', '7', \
  '8', '9', '0', '.', ',', ':', '?', '\'', '-', '/', '(', \
  '"', ';', '$', '#', '<', '!', '@', ']',  '=', '~', ' '                                           \
}

/* Hex equivalents to Morse code chars above. Formed
 * by starting with a 1 and following with a 0 for
 * dash and 1 for dit e.g: A = ditdah = 101 = Hex 0x06 */
#define MORSE_CODE \
{ \
  0x06, 0x17, 0x15, 0x0b, 0x03, 0x1d, 0x09, 0x1f, 0x07, 0x18, 0x0a, \
  0x1b, 0x04, 0x05, 0x08, 0x19, 0x12, 0x0d, 0x0f, 0x02, 0x0e, 0x1e, \
  0x0c, 0x16, 0x14, 0x13, 0x30, 0x38, 0x3c, 0x3e, 0x3f, 0x2f, 0x27, \
  0x23, 0x21, 0x20, 0x6a, 0x4c, 0x47, 0x73, 0x61, 0x5e, 0x2d, 0x52, \
  0x6d, 0x55, 0xf6, 0x35, 0x7a, 0x2a, 0x37, 0x29, 0x2e, 0xff, 0x01  \
}

//-------------------------------------------------------------------

/* Morse_Symbol()
 *
 * Transmits a Morse symbol (a Dit or Dah) as a mark signal
 * followed by a space signal, both of specified length
 */
  static BOOLEAN
Morse_Symbol( uint16_t mark_len, uint16_t space_len )
{
  static uint16_t
    mark_timer  = 0,
    space_timer = 0;

  // Send a Mark signal over a mark_len count
  if( mark_timer < mark_len )
  {
    if( !Generate_CW(True) ) Modulator = NULL;
    mark_timer++;
    return( False );
  }

  // Send a Space signal over a space_len count
  if( space_timer < space_len )
  {
    if( !Generate_CW(False) ) Modulator = NULL;
    space_timer++;
    return( False );
  }

  // Reset and signal completion of Mark/Space
  mark_timer  = 0;
  space_timer = 0;

  return( True );
} // Morse_Symbol()

//-------------------------------------------------------------------

#define ELEM_LENGTH    20    // Length of Morse element (dit)
#define DASH_LENGTH    60    // Length of Morse dash (3 elements)
#define CHAR_SPACE     40    // Space between characters (2 elements)
#define WORD_SPACE     120   // Space between words (6 elements)

/* Morse_Transmit()
 *
 * Trasnsmits a character string in Morse code
 */
  void
Morse_Transmit( void *data )
{
  const char *mesg = (char *)data;

  // New message is to be sent
  static BOOLEAN new_mesg = True;

  static int
    mesg_len,   // String length of message to send
    mesg_idx,   // Index into the message string
    morse_idx;  // Index into the Morse Hex code

  // Tables of Morse and ascii characters
  static const int morse_code[] = MORSE_CODE;

  // Initialize on new message
  if( new_mesg )
  {
    new_mesg  = False;
    mesg_len  = (int)strlen( mesg );
    mesg_idx  = 0;
    morse_idx = -2;
  }

  // Convert ascii characters to Morse hex code
  if( mesg_idx < mesg_len )
  {
    static int morse_chr;

    /* Find equivalent Hex morse code if new
     * mesg or previous character finished */
    if( morse_idx == -2 )
    {
      int16_t idx;
      static const int ascii_char[] = ASCII_CHAR;

      for( idx = 0; idx < NUM_MORSE_CHARS; idx++ )
        if( mesg[mesg_idx] == ascii_char[idx] )
          break;

      /* In case of unknown character
       * in mesg, transmitt space */
      if( idx < NUM_MORSE_CHARS )
        morse_chr = morse_code[idx];
      else
        morse_chr = 0x01; // Space as default

      // Find beginning of Morse code (1st '1')
      for( idx = 8; idx >= 0; idx-- )
        if( morse_chr & (1 << idx) )
          break;
      morse_idx = idx - 1;

    } // if( morse_idx == -2 )

    // If character is word space, transmit space
    if( morse_chr == 0x01 )
    {
      if( Morse_Symbol(0, WORD_SPACE) )
      {
        // Signal for new character
        morse_idx = -2;
        mesg_idx++;
      }
      return;
    }

    // Transmit dots (1's) or dashes (0's)
    if( morse_idx >= 0 )
    {
      // If element is dot ('1')
      if( morse_chr & (1 << morse_idx) )
      {
        if( Morse_Symbol(ELEM_LENGTH, ELEM_LENGTH) ) morse_idx--;
        return;
      }
      else // If element is dash ('0')
      {
        if( Morse_Symbol(DASH_LENGTH, ELEM_LENGTH) ) morse_idx--;
        return;
      }
    } // if( morse_idx >= 0 )

    // Send inter-character space
    if( !Morse_Symbol(0, CHAR_SPACE) ) return;

    // Signal for new character
    morse_idx = -2;
    mesg_idx++;
    return;
  } // if( mesg_idx < mesg_len )

  Modulator = NULL;
  new_mesg  = True;
  Flag[TRANSMIT_MORSE_MESG] = False;

  // Signal Morse message transmission complete
  int sval;
  sem_getvalue( &duc_send_semaphore, &sval );
  if( !sval ) sem_post( &duc_send_semaphore );

} // Morse_Transmit()

//------------------------------------------------------------------------

/* DUC_Buffer_Transmit()
 *
 * Transmits the supplied samples buffer to the DUC
 */
  void
DUC_Buffer_Transmit( void *data )
{
  // Count of input buffer samples used
  static uint32_t buf_cnt = 0;

  // Count of microphone samples used for timing
  static uint8_t mic_cnt = 0;

  // Abort if samples buffer empty
  if( !xmit_buffer.xmit_buf_len ) return;

  // The mic data packet is used to time samples uploads to the DUC
  while( mic_cnt < MIC_NUM_OF_SAMPLES )
  {
    // Enter input data to DUC buffer. (MIC_NUM_OF_SAMPLES = DUC_NUM_IQ_SAMPLES)
    duc_buf_i[mic_cnt] = xmit_buffer.xmit_buf_i[buf_cnt];
    duc_buf_q[mic_cnt] = xmit_buffer.xmit_buf_q[buf_cnt];
    buf_cnt++;

    // Post to semaphore and reset buf_cnt if all data copied
    if( buf_cnt >= xmit_buffer.xmit_buf_len )
    {
      xmit_buffer.status = True;
      int sval;
      sem_getvalue( &duc_send_semaphore, &sval );
      if( !sval ) sem_post( &duc_send_semaphore );
      buf_cnt = 0;
      return;
    }

    mic_cnt++;
  } // while( mic_cnt > 0 )
  mic_cnt = 0;

  // Pack the SSB sample stream into the C&C USB Buffer
  Pack_DUC_Tx_Data( duc_buf_i, duc_buf_q, duc_buf_i, duc_buf_q );

  return;
} // DUC_Buffer_Transmit()

//------------------------------------------------------------------------

  void
Free_Modulator_Buffers( void )
{
  Mem_Free( (void **) &samples_buf );
}

//------------------------------------------------------------------------

