/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "demodulate.h"
#include "interface.h"
#include "modulate.h"
#include "process.h"
#include "spectrum.h"
#include "../common/common.h"
#include "../common/cfft.h"
#include "../common/filters.h"
#include "../common/hermes2_rc.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/hpsdr_init.h"
#include "../rsid/rsid.h"
#include <gtk/gtk.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>

//----------------------------------------------------------------------

#define HERMES2_GLADE_FILE "/.hermes2/hermes2/hermes2.glade"

// Signal handler
static void sig_handler( int signal );

/* main()
 *
 * Hermes2's main() function
 */
  int
main( int argc, char *argv[] )
{
  // New and old actions for sigaction()
  struct sigaction sa_new, sa_old;

  // Command line option returned by getopt()
  int option;

  // Initialize new actions
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  // Register function to handle signals
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );
  sigaction( SIGALRM, &sa_new, 0 );

  // Clear all flags
  for( uint16_t idx = 0; idx < NUM_OF_FLAGS; idx++ )
    Flag[idx] = False;

  // Process command line options
  while( (option = getopt(argc, argv, "vV") ) != -1 )
    switch( option )
    {
      case 'v': // Enable Error messages
        Flag[HERMES2_ERROR_MESSAGES] = True;
        break;

      case 'V': // Enable Verbose messages
        Flag[HERMES2_VERBOSE_MESSAGES] = True;
        Flag[HERMES2_ERROR_MESSAGES]   = True;
        break;

    } // End of switch( option )

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_init (&argc, &argv);

  /* Initialize some objects */
  audio_ring_buff.buff_left  = NULL;
  audio_ring_buff.buff_right = NULL;
  hermes2_gui.guest_window   = NULL;
  hermes2_gui.window         = NULL;
  hermes2_gui.window_builder = NULL;
  qso_record.qso_record_fp   = NULL;
  hermes2_gui.config_dialog_builder = NULL;
  hermes2_gui.bmk_window_builder    = NULL;
  Flag[HERMES2_INITIALIZED]    = False;
  Flag[HERMES2_RCCONFIG_SETUP] = False;
  Flag[GUEST_TRANSMITTING]     = False;
  Flag[GUEST_RECEIVING]        = False;
  Flag[HERMES2_CAPTURE_SETUP]  = False;
  Flag[HERMES2_PCSOUND_SETUP]  = False;

  // Initialize the Command and Control function
  Init_Cmnd_Ctrl_Audio_IQ_Data();

  // Create main hermes2 window's glade file path
  Strlcpy( hermes2_rc.hermes2_glade, getenv("HOME"), sizeof(hermes2_rc.hermes2_glade) );
  Strlcat( hermes2_rc.hermes2_glade, HERMES2_GLADE_FILE, sizeof(hermes2_rc.hermes2_glade) );

  // Check for the Glade config file
  FILE *fp = fopen( hermes2_rc.hermes2_glade, "r" );
  if( fp == NULL )
  {
    fprintf( stderr, "hermes2: cannot open hermes2 Glade GUI Description file.\n" );
    perror( hermes2_rc.hermes2_glade );
    fprintf( stderr, "hermes2: trying to create hermes2 config directory "
        "from the installation prefix file tree.\n" );

    // Find the binary's path (location in file system)
    char exe_path[256], file_path[288];

    // Read the file path to hermes2 executable
    size_t len = sizeof( exe_path );
    int bytes = (int)readlink( "/proc/self/exe", exe_path, len );
    if( bytes <= 0 )
    {
      fprintf( stderr, "hermes2: cannot read hermes2 binary's location.\n" );
      perror( "/proc/self/exe" );
      exit( -1 );
    }

    // Remove "/bin/hermes2" from the path with room for termination
    bytes -= sizeof( "/bin/hermes2" ) - 1;
    if( bytes < 1 )
    {
      fprintf( stderr, "hermes2: cannot create file path to examples/hermes2.\n" );
      exit( -1 );
    }

    // Create file path to hermes2 examples directory
    exe_path[bytes] = '\0';
    Strlcpy( file_path, exe_path, sizeof(file_path) );
    Strlcat( file_path, "/share/examples/hermes2", sizeof(file_path) );
    fprintf( stderr, "hermes2: creating hermes2 config directory from: %s\n", file_path );

    // Create system command to copy examples/hermes2 to ~/.hermes2
    char syscmnd[512];
    Strlcpy( syscmnd, "cp -r ", sizeof(syscmnd) );
    Strlcat( syscmnd, file_path, sizeof(syscmnd) );
    Strlcat( syscmnd, " ", sizeof(syscmnd) );
    Strlcat( syscmnd, getenv("HOME"), sizeof(syscmnd) );
    Strlcat( syscmnd, "/.hermes2",   sizeof(syscmnd) );
    int ret = system( syscmnd );
    if( ret == -1 )
    {
      fprintf( stderr,"hermes2: cannot create hermes2's working directory.\n" );
      perror( file_path );
      exit( -1 );
    }
  } // if( fp == NULL )
  else Close_File( &fp );

  // Do discovery and load the hermes2_rc file
  g_idle_add( Do_Discovery, NULL );

  gtk_main();

  Free_Modulator_Buffers();
  Free_Xmit_Buffers();
  Free_Hermes2_Buffer();

  return( 0 );
} // main()

//----------------------------------------------------------------------

/* sig_handler()
 *
 * Signal Action Handler function
 */
void sig_handler(int signal)  __attribute__ ((noreturn));
  static void
sig_handler( int signal )
{
  Hermes2_Stop();
  switch( signal )
  {
    case SIGINT :
      fprintf( stderr, "\n!!%s\n", _("hermes2: Exiting via User Interrupt") );
      exit(-1);

    case SIGSEGV :
      fprintf( stderr, "\n!!%s\n", _("hermes2: Segmentation Fault") );
      exit(-1);

    case SIGFPE :
      fprintf( stderr, "\n!!%s\n", _("hermes2: Floating Point Exception") );
      exit(-1);

    case SIGABRT :
      fprintf( stderr, "\n!!%s\n", _("hermes2: Abort Signal received") );
      exit(-1);

    case SIGTERM :
      fprintf( stderr, "\n!!%s\n", _("hermes2: Termination Request received") );
      exit(-1);

    default:
      fprintf( stderr, "\n!!%s\n", _("hermes2: Quit in default") );
      exit(-1);
  }

} // End of sig_handler()

//----------------------------------------------------------------------

