/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "interface.h"
#include "callback_func.h"
#include "display.h"
#include "process.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/hw_to_pc.h"
#include "../hpsdr/discovery.h"
#include "../time/interface.h"
#include <gtk/gtk.h>
#include <pthread.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <semaphore.h>

//----------------------------------------------------------------------

#define CONFIG_DIALOG_IDS \
  "hermes2_config_dialog", \
  "cfg_buf_adjustment", \
  "cfg_cw_hang_adjustment", \
  "cfg_ptt_hang_adjustment", \
  "cfg_rfdelay_adjustment", \
  "cfg_tune_adjustment", \
  "cfg_volume_adjustment", \
  NULL

//----------------------------------------------------------------------

#define DISCOVERY_DIALOG_IDS \
  "discovery_dialog", \
  NULL

//----------------------------------------------------------------------

#define MESSAGE_DIALOG_IDS \
  "hermes2_message_dialog", \
  "hermes2_message_label", \
  NULL

//----------------------------------------------------------------------

#define QUIT_DIALOG_IDS \
  "hermes2_quit_dialog", \
  NULL

//----------------------------------------------------------------------

#define SAVE_DIALOG_IDS \
  "guest_save_dialog", \
  NULL

//----------------------------------------------------------------------

/* Hermes2_Window()
 *
 * Creates the Main hermes2 window if not open
 * The Gtk Builder is created in Read_Config()
 */
  void
Hermes2_Window( void )
{
  if( hermes2_gui.window != NULL ) return;

  hermes2_gui.window = Builder_Get_Object( hermes2_gui.window_builder, "hermes2_window" );
  gtk_window_set_title( GTK_WINDOW(hermes2_gui.window), PACKAGE_STRING );
  gtk_widget_show( hermes2_gui.window );
  hermes2_gui.device_combobox =
    Builder_Get_Object( hermes2_gui.window_builder, "device_combobox" );
  hermes2_gui.sample_rate_combobox =
    Builder_Get_Object( hermes2_gui.window_builder, "sample_rate_combobox" );
  hermes2_gui.lna_gain_spinbutton =
    Builder_Get_Object( hermes2_gui.window_builder, "lna_gain" );
  hermes2_gui.lna_gain_auto =
    Builder_Get_Object( hermes2_gui.window_builder, "lna_gain_auto" );
  hermes2_gui.lna_gain_hw =
    Builder_Get_Object( hermes2_gui.window_builder, "lna_gain_hw" );
  hermes2_gui.tx_pa_enable =
    Builder_Get_Object( hermes2_gui.window_builder, "tx_pa_enable" );
  hermes2_gui.tx_enable_icon =
    Builder_Get_Object( hermes2_gui.window_builder, "tx_enable_icon" );
  hermes2_gui.tx_overload_icon =
    Builder_Get_Object( hermes2_gui.window_builder, "tx_overload_icon" );
  hermes2_gui.tx_rx_duplex =
    Builder_Get_Object( hermes2_gui.window_builder, "tx_rx_duplex" );
  hermes2_gui.rx_antenna1  =
    Builder_Get_Object( hermes2_gui.window_builder, "adc_antenna1" );
  hermes2_gui.rx_antenna2  =
    Builder_Get_Object( hermes2_gui.window_builder, "adc_antenna2" );
  hermes2_gui.adc_ol_icon  =
    Builder_Get_Object( hermes2_gui.window_builder, "adc_ol_icon" );
  hermes2_gui.overflow_icon =
    Builder_Get_Object( hermes2_gui.window_builder, "overflow_icon" );
  hermes2_gui.underflow_icon =
    Builder_Get_Object( hermes2_gui.window_builder, "underflow_icon" );
  hermes2_gui.tx_fifo_label =
    Builder_Get_Object( hermes2_gui.window_builder, "tx_fifo_label" );
  hermes2_gui.pa_cur_label =
    Builder_Get_Object( hermes2_gui.window_builder, "pa_cur_label" );
  hermes2_gui.temp_label =
    Builder_Get_Object( hermes2_gui.window_builder, "temp_label" );

  // Append HPSDR device names
  Append_Device_Names( hermes2_gui.device_combobox );

} // Hermes2_Window()

//----------------------------------------------------------------------

/* Message_Dialog()
 *
 * Opens a message dialog box
 */
  void
Message_Dialog( gchar *mesg )
{
  if( !hermes2_gui.message_dialog )
  {
    GtkBuilder *builder;
    gchar *object_ids[] = { MESSAGE_DIALOG_IDS };

    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.message_dialog = Builder_Get_Object( builder, "hermes2_message_dialog" );
    GtkLabel *label = GTK_LABEL( Builder_Get_Object( builder, "hermes2_message_label") );
    gtk_label_set_text( label, mesg );
    gtk_widget_show( hermes2_gui.message_dialog );
    g_object_unref( builder );
  }

} // Message_Dialog()

//----------------------------------------------------------------------

/* Quit_Dialog()
 *
 * Opens the quit dialog box
 */
  void
Quit_Dialog( void )
{
  Flag[GUEST_QUIT] = True;

  // Free any digimode semaphore blocks
  if( Flag[GUEST_RECEIVING] )
  {
    Flag[GUEST_RECEIVE_MODE] = False;
    int sval;
    sem_getvalue( &digimode_semaphore, &sval );
    if( !sval ) sem_post( &digimode_semaphore );
  }

  if( !hermes2_gui.quit_dialog )
  {
    GtkBuilder *builder;
    gchar *object_ids[] = { QUIT_DIALOG_IDS };
    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.quit_dialog = Builder_Get_Object( builder, "hermes2_quit_dialog" );
    gtk_widget_show( hermes2_gui.quit_dialog );
    g_object_unref( builder );
  }

} // Quit_Dialog()

//----------------------------------------------------------------------

/* Guest_Save_Dialog()
 *
 * Creates the Save Records dialog
 */
  void
Guest_Save_Dialog( const gchar *mesg )
{
  if( !hermes2_gui.save_dialog )
  {
    GtkBuilder *builder;
    gchar *object_ids[] = { SAVE_DIALOG_IDS };
    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.save_dialog = Builder_Get_Object( builder, "guest_save_dialog" );
    GtkWidget *lbl = Builder_Get_Object( builder, "guest_save_message" );
    gtk_label_set_text( GTK_LABEL(lbl), mesg );
    gtk_widget_show( hermes2_gui.save_dialog );
    g_object_unref( builder );
  }

} // Guest_Save_Dialog()

//------------------------------------------------------------------

static GtkTextIter    iter;
static GtkTextView   *text_view;
static GtkTextBuffer *text_buffer;

/* Discovery_Dialog_Text()
 *
 * Idle callback to write markup text to the Discovery dialog box
 */
  void
Discovery_Dialog_Text( const gchar *markup )
{
  // Insert markup and autoscroll to end of textview
  gtk_text_buffer_insert_markup( text_buffer, &iter, markup, -1 );
  gtk_text_view_scroll_mark_onscreen(
      text_view, gtk_text_buffer_create_mark(text_buffer, "end", &iter, FALSE) );
  while( g_main_context_iteration(NULL, FALSE) );
} // Discovery_Dialog_Text()

//----------------------------------------------------------------------

/* Discovery_Dialog()
 *
 * Opens the Discovery dialog box
 */
  void
Discovery_Dialog( int8_t flags )
{
  static GtkWidget *button_box, *close_button;

  // Create discovery dialog if not already done
  if( !hermes2_gui.discovery_dialog )
  {
    GtkBuilder *builder;
    gchar *object_ids[] = { DISCOVERY_DIALOG_IDS };
    gchar mesg[MESG_STRING_SIZE];

    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.discovery_dialog = Builder_Get_Object( builder, "discovery_dialog" );
    text_view   = GTK_TEXT_VIEW( Builder_Get_Object(builder, "discovery_textview") );
    text_buffer = gtk_text_view_get_buffer( text_view );
    gtk_text_buffer_get_start_iter( gtk_text_view_get_buffer(text_view), &iter );
    button_box   = Builder_Get_Object( builder, "discovery_button_box" );
    close_button = Builder_Get_Object( builder, "discovery_close_button" );
    gtk_widget_set_sensitive( button_box, FALSE );
    gtk_widget_show( hermes2_gui.discovery_dialog );

    g_object_unref( builder );

    // Display Welcome message
    snprintf( mesg, sizeof(mesg),
        _("<span foreground=\"green\" font_weight=\"bold\">"
          "Welcome to Hermes2 v%s\n\n</span>"), VERSION );
    Discovery_Dialog_Text( mesg );

  } // if( !discovery_dialog )

  // Set sensitivity of buttons according to flags
  if( flags & SHOW_BUTTON_BOX )
    gtk_widget_set_sensitive( button_box, TRUE );
  else
    gtk_widget_set_sensitive( button_box, FALSE );

  if( flags & SHOW_CLOSE_BUTTON )
    gtk_widget_set_sensitive( close_button, TRUE );
  else
    gtk_widget_set_sensitive( close_button, FALSE );

} // Discovery_Dialog()

//----------------------------------------------------------------------

/* Config_Dialog()
 *
 * Opens the hermes2 config dialog box
 */
  void
Config_Dialog( void )
{
  gchar *object_ids[] = { CONFIG_DIALOG_IDS };
  gchar *hermes2_buf[] = { HERMES2_BUF_SIZE };
  GtkWidget *widget;
  gchar str[16];
  int8_t idx;

  // Abort if already open
  if( hermes2_gui.config_dialog ) return;

  // Stop operations
  Hermes2_Stop();
  Guest_Quit_Activate( hermes2_gui.guest_window );

  // Create config dialog
  Gtk_Builder( &hermes2_gui.config_dialog_builder,
      (gchar *)hermes2_rc.hermes2_glade, object_ids );
  hermes2_gui.config_dialog =
    Builder_Get_Object( hermes2_gui.config_dialog_builder, "hermes2_config_dialog" );
  gtk_widget_show( hermes2_gui.config_dialog );

  // Disable Time decoding before reconfiguring
  if( time_window )
  {
    widget = Builder_Get_Object( time_window_builder, "time_receive_togglebutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), FALSE );
  }

  // Append HPSDR device names
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_device_combobox" );
  Append_Device_Names( widget );
  gtk_combo_box_set_active( GTK_COMBO_BOX(widget), (gint)hermes2_rc.device_index );

  // Enter current TCXO frequency error (PPM)
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_freq_error_entry" );
  snprintf( str, sizeof(str), "%+0.4f", hermes2_rc.device_freq_error );
  gtk_entry_set_text( GTK_ENTRY(widget), str );

  // Enter current ALSA HWCTL device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_hwctl_device_entry" );
  gtk_entry_set_text( GTK_ENTRY(widget), hermes2_rc.hwctl_device );

  // Enter current ALSA PCM device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_pcm_device_entry" );
  gtk_entry_set_text( GTK_ENTRY(widget), hermes2_rc.pcm_device );

  // Enter the ALSA Capture device name
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_capture_dev_entry" );
  gtk_entry_set_text( GTK_ENTRY(widget), hermes2_rc.capture_src );

  // Enter the ALSA Capture Volume device
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_capture_vol_entry" );
  gtk_entry_set_text( GTK_ENTRY(widget), hermes2_rc.capture_vol );

  // Enter the ALSA Capture Channel
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_cap_chn_combobox" );
  gtk_combo_box_set_active( GTK_COMBO_BOX(widget), hermes2_rc.capture_chan );

  // Set state of Local/Remote Audio check button
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_pc_sound_checkbutton" );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(widget), hermes2_rc.sound_pc_local );

  // Clear existing entries from DDC buffer combobox
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ddc_buff_combobox" );
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(widget) );

  // Append buffer sizes and set active
  for( idx = 0; idx < HERMES2_BUF_SIZE_NUM; idx++ )
    gtk_combo_box_text_append_text( GTK_COMBO_BOX_TEXT(widget), hermes2_buf[idx] );
  gtk_combo_box_set_active( GTK_COMBO_BOX(widget), (gint)hermes2_rc.ddc_buffer_idx );

  // Set the Beep Volume entry widget
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_beep_volume_spinbutton" );
  snprintf( str, sizeof(str), "%u", hermes2_rc.beep_volume );
  gtk_entry_set_text( GTK_ENTRY(widget), str );

  // We use settings from the primary Transceiver window, if open. Else we use defaults.
  Transceiver_t *TRx = Transceiver[0];
  if( TRx != NULL )
  {
    // Set the Sound Volume value
    widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_volume_spinbutton");
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), (gdouble)TRx->snd_vol_scale );

    // Set the Rx Frequency Band Index
    widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_rx_band_combobox");
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), TRx->rx_bands_idx );

    // Set the default DDC sample rate index
    widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ddc_rate_combobox");
    Append_Sample_Rates( widget );
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), hermes2_rc.sample_rate_idx );

    // Set the FFT frame rate
    widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_fft_rate_combobox" );
    gtk_combo_box_set_active( GTK_COMBO_BOX(widget), TRx->fft_rate_idx );

    // Set the Tx Frequency Band Index
    widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_band_combobox");
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), TRx->tx_bands_idx );
  }
  else // Default values
  {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), 30 );
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), 5 );
    gtk_combo_box_set_active( GTK_COMBO_BOX(widget), 3 );
    gtk_combo_box_set_active(GTK_COMBO_BOX(widget), 5 );
  }

  // Set the TX Tune Power %
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_tune_power_spinbutton" );
  gtk_spin_button_set_value(
      GTK_SPIN_BUTTON(widget), (gdouble)hermes2_rc.tx_tune_power / MAX_PA_POWER * 100.0 );

  // Set the PTT Hang Delay setting
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_ptt_hang_time_spinbutton" );
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), (gdouble)hermes2_rc.ptt_hang_time );

  // Set the CW Hang Delay setting
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_cw_hang_time_spinbutton" );
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), (gdouble)hermes2_rc.cw_hang_time );

  // Set the TX Buffer Latency in mSec
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_tx_buf_latency_spinbutton");
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), (gdouble)hermes2_rc.tx_buf_latency );

  // Set the RF Delay spinbutton value in mS
  widget = Builder_Get_Object( hermes2_gui.config_dialog_builder, "cfg_rf_delay_spinbutton" );
  gtk_spin_button_set_value( GTK_SPIN_BUTTON(widget), (gdouble)hermes2_rc.rf_delay );

} // Config_Dialog()

//----------------------------------------------------------------------

