/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "demodulate.h"
#include "callback_func.h"
#include "interface.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/settings.h"
#include <gtk/gtk.h>
#include <math.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

#define HERMES2_SAMPLE_RATES \
  "DDC 48 kS/s",  "DDC 96 kS/s",  "DDC 192 kS/s", "DDC 384 kS/s"
#define HERMES2_SAMPLE_RATES_NUM     4

//----------------------------------------------------------------------

// S-meter range in dBm (0 to -135 dBm)
#define SMETER_RANGE_DBM  135.0

/* HERMES2 S-meter offset to raise
 * ADAGC dBm range to -135 - 0 dBm */
#define HERMES2_SMETER_OFFSET  82.0

//----------------------------------------------------------------------

// Number of frequency labels in frequency display
#define NUM_FREQ_LABELS  9

//----------------------------------------------------------------------

// Order of maximum frequency to be used
#define MAX_FREQ_ORDER   100000000

//----------------------------------------------------------------------

/* Pointer to the center frequency "spin button"
 * label that has the pointer enter or leave it */
GtkLabel *pointer_cross_label;

//----------------------------------------------------------------------

/* Set_Rx_Band()
 *
 * Sets the Receiver Band combobox
 * according to the dial frequency
 */
  void
Set_Rx_Band( Transceiver_t *TRx, uint32_t freq )
{
  gint idx;

  // Don't change bands combobox if working in GENeral band
  if( TRx->rx_bands_idx == IDX_GEN ) return;

  // Find the apropriate Bands index
  if(      (freq >= LO_160) && (freq <= HI_160) ) idx = IDX_160;
  else if( (freq >= LO_80)  && (freq <= HI_80) )  idx = IDX_80;
  else if( (freq >= LO_60)  && (freq <= HI_60) )  idx = IDX_60;
  else if( (freq >= LO_40)  && (freq <= HI_40) )  idx = IDX_40;
  else if( (freq >= LO_30)  && (freq <= HI_30) )  idx = IDX_30;
  else if( (freq >= LO_20)  && (freq <= HI_20) )  idx = IDX_20;
  else if( (freq >= LO_17)  && (freq <= HI_17) )  idx = IDX_17;
  else if( (freq >= LO_15)  && (freq <= HI_15) )  idx = IDX_15;
  else if( (freq >= LO_12)  && (freq <= HI_12) )  idx = IDX_12;
  else if( (freq >= LO_10)  && (freq <= HI_10) )  idx = IDX_10;
  else                                            idx = IDX_GEN;

  // Set the Receiver combobox
  gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->rx_bands_combobox), idx );

} // Set_Rx_Band()

//----------------------------------------------------------------------

/* Display_Frequency_Spin_Dial()
 *
 * Sets the Rx or Tx frequency and displays it in the spin dial
 */
  void
Display_Frequency_Spin_Dial(
    Transceiver_t *TRx,
    GtkLabel *label,
    int32_t  data )
{
  uint8_t idx;
  int32_t mult;
  int32_t freq;
  gchar lbl[5], new_txt[HERMES2_MARKUP_SIZE];
  const gchar *name;
  GtkLabel *index_label = NULL;
  BOOLEAN zero_flag = False, rx_flag = False;

  // Set new value to label
  if( label == pointer_cross_label )
  {
    snprintf( new_txt, sizeof(new_txt), POINTER_ENTER_MARKUP, data );
  }
  else
  {
    snprintf( new_txt, sizeof(new_txt), POINTER_LEAVE_MARKUP, data );
  }
  gtk_label_set_markup( label, new_txt );

  // Determine Rx or Tx label
  name = gtk_widget_get_name( GTK_WIDGET(label) );
  if( strstr(name, "rx") )
    rx_flag = True;
  else if( strstr(name, "tx") )
    rx_flag = False;

  // Read frequency from labels
  mult = 1; freq = 0;
  for( idx = 1; idx <= NUM_FREQ_LABELS; idx++ )
  {
    const gchar *txt;
    int32_t value;

    // Get the frequency label at current index
    if( rx_flag )
      snprintf( lbl, sizeof(lbl), "rf%u", idx );
    else
      snprintf( lbl, sizeof(lbl), "tf%u", idx );

    index_label = GTK_LABEL( Builder_Get_Object(TRx->builder, lbl) );
    txt = gtk_label_get_text( index_label );

    /* If label at current index not the label changed
     * by user then zero it if Zero Right is enabled */
    if( label == index_label ) zero_flag = True;
    value = (int32_t)atoi( txt );
    if( !zero_flag &&
        ((TRx->rx_zero_right &&  rx_flag) ||
         (TRx->tx_zero_right && !rx_flag)) )
      value = 0;

    freq += value * mult;
    mult *= 10;
  } // for( idx = 1; idx <= NUM_FREQ_LABELS; idx++ )

  // Set Rx or Tx center frequency
  if( freq >= 0 )
  {
    // Set Receiver frequency
    if( rx_flag )
    {
      // Set the Rx Band combobox first
      Set_Rx_Band( TRx, (uint32_t)freq );

      // Enter the Receiver frequency
      TRx->rx_frequency = (uint32_t)freq;

      // Link Tx frequency to Rx
      if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
      {
        TRx->tx_frequency = TRx->rx_frequency;
        Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
      }
    }
    else if( !TRx->tx_freq_lock ) // Set Transmitter frequency
    {
      TRx->tx_frequency = (uint32_t)freq;
    }

    // Set the Center Frequency of the Transceiver
    Hermes2_Set_Center_Frequency( TRx, rx_flag );
  } // if( freq >= 0 )
  else
  {
    // Zero label if data goes negative
    if( label == pointer_cross_label )
    {
      snprintf( new_txt, sizeof(new_txt), POINTER_ENTER_MARKUP, 0 );
    }
    else
    {
      snprintf( new_txt, sizeof(new_txt), POINTER_LEAVE_MARKUP, 0 );
    }

    gtk_label_set_markup( label, new_txt );
  }  // else of if( freq >= 0 )

  // Save last used frequency to Rx or Tx Bands list
  if( rx_flag )
    hermes2_rc.rx_bands[TRx->rx_bands_idx] = TRx->rx_frequency;
  else
    hermes2_rc.tx_bands[TRx->tx_bands_idx] = TRx->tx_frequency;

} // Display_Frequency_Spin_Dial()

//----------------------------------------------------------------------

/* Update_Spin_Dial()
 *
 * Updates the Frequency spin dial with latest center frequency
 */
  void
Update_Spin_Dial( Transceiver_t *TRx, BOOLEAN rx_flag )
{
  uint32_t mult;
  int idx;
  gchar lbl[5], new_txt[HERMES2_MARKUP_SIZE];
  uint32_t freq;

  // Enter new frequency to the Frequency spin dial
  if( rx_flag )
    freq = TRx->rx_frequency;
  else
    freq = TRx->tx_frequency;

  mult = MAX_FREQ_ORDER;
  for( idx = NUM_FREQ_LABELS; idx >= 1; idx-- )
  {
    uint32_t value;
    GtkLabel *label = NULL;

    value = freq / mult;
    if( rx_flag )
      snprintf( lbl, sizeof(lbl), "rf%d", idx );
    else
      snprintf( lbl, sizeof(lbl), "tf%d", idx );
    label = GTK_LABEL( Builder_Get_Object(TRx->builder, lbl) );

    // If label has the pointer over it
    if( label == pointer_cross_label )
    {
      snprintf( new_txt, sizeof(new_txt), POINTER_ENTER_MARKUP, value );
    }
    else
    {
      snprintf( new_txt, sizeof(new_txt), POINTER_LEAVE_MARKUP, value );
    }

    // Set the new label
    if( label )
    {
      new_txt[HERMES2_MARKUP_SIZE-1] = '\0';
      gtk_label_set_markup( label, new_txt );
    }

    freq -= value * mult;
    mult /= 10;
  } // for( idx = NUM_FREQ_LABELS; idx > 0; idx-- )

} // Update_Spin_Dial()

//----------------------------------------------------------------------

/* Append_Sample_Rates()
 *
 * Appends sample rates to sample rates combobox
 */
  void
Append_Sample_Rates( GtkWidget *rate_combobox )
{
  int idx;

  // Clear existing rates
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(rate_combobox) );

  // Append sample rates
  gchar *hpsdr_srate[] = { HERMES2_SAMPLE_RATES };
  for( idx = 0; idx < HERMES2_SAMPLE_RATES_NUM; idx++ )
    gtk_combo_box_text_append_text(
        GTK_COMBO_BOX_TEXT(rate_combobox), hpsdr_srate[idx] );

} // Append_Sample_Rates()

//----------------------------------------------------------------------

/* Append_Device_Names()
 *
 * Appends the names of available HPSDR devices
 */
  void
Append_Device_Names( GtkWidget *combobox )
{
  // Clear existing entries
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(combobox) );

  // Append discovered device types
  uint32_t idx;
  for( idx = 0; idx < hermes2_rc.number_of_devices; idx++ )
  {
    gtk_combo_box_text_append_text(
        GTK_COMBO_BOX_TEXT(combobox), Device[idx].device_name );
  }

} // Append_Device_Names()

//----------------------------------------------------------------------

/* Append_BWidth_Weaver()
 *
 * Appends text into the demodulator and
 * modulator bandwidth and weaver comboboxes
 */
  void
Append_BWidth_Weaver( Transceiver_t *TRx )
{
  gchar *rx_bw[] = { RX_BAND_WIDTHS };
  gchar *tx_bw[] = { TX_BAND_WIDTHS };
  gchar *rx_weaver_freqs[] = { RX_WEAVER_FREQS };
  gchar *tx_weaver_freqs[] = { TX_WEAVER_FREQS };
  int idx;

  // Clear existing entries
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(TRx->rx_bw_combobox) );

  // Append Demodulator bandwidths
  for( idx = 0; idx < RX_BANDWIDTH_NUM; idx++ )
    gtk_combo_box_text_append_text(
        GTK_COMBO_BOX_TEXT(TRx->rx_bw_combobox), rx_bw[idx] );

  // Clear existing entries
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(TRx->rx_weaver_combobox) );

  // Append Weaver frequencies
  for( idx = 0; idx < RX_WEAVER_FREQ_NUM; idx++ )
    gtk_combo_box_text_append_text(
        GTK_COMBO_BOX_TEXT(TRx->rx_weaver_combobox), rx_weaver_freqs[idx] );

  // Clear existing entries
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(TRx->tx_bw_combobox) );

  // Append Modulator bandwidths
  for( idx = 0; idx < TX_BANDWIDTH_NUM; idx++ )
    gtk_combo_box_text_append_text( GTK_COMBO_BOX_TEXT(TRx->tx_bw_combobox), tx_bw[idx] );

  // Clear existing entries
  gtk_combo_box_text_remove_all( GTK_COMBO_BOX_TEXT(TRx->tx_weaver_combobox) );

  // Append Weaver frequencies
  for( idx = 0; idx < TX_WEAVER_FREQ_NUM; idx++ )
    gtk_combo_box_text_append_text(
        GTK_COMBO_BOX_TEXT(TRx->tx_weaver_combobox), tx_weaver_freqs[idx] );

} // Append_BWidth_Weaver()

//----------------------------------------------------------------------

/* Hermes2_Display_S_meter()
 *
 * Displays the S-meter bar and signal strength
 */
  gboolean
Hermes2_Display_S_meter( gpointer data )
{
  Transceiver_t *TRx = (Transceiver_t *)data;

  if( TRx->S_meter < 0.0 ) TRx->S_meter = 0.0;
  if( TRx->S_meter > 1.0 ) TRx->S_meter = 1.0;

  // Set Signal Strength progress bar
  gtk_level_bar_set_value( GTK_LEVEL_BAR(TRx->smeter_levelbar), (gdouble)TRx->S_meter );

  return( FALSE );
} // Hermes2_Display_S_meter()

//----------------------------------------------------------------------

/* Hermes2_Display_Mode()
 *
 * Shows the modulation mode on the mode menu button
 */
  void
Hermes2_Display_Mode( Transceiver_t *TRx, uint8_t mode, BOOLEAN rx_flag )
{
  GtkButton *button = NULL;

  if( TRx ) // Activated from Transceiver window
  {
    if( rx_flag )
    {
      gchar *modes[RX_MODE_ITEMS] = { RX_MODE_MENU_ITEMS };
      button = GTK_BUTTON( Builder_Get_Object( TRx->builder, "rx_modes_menu_button") );
      gtk_button_set_label( button, modes[mode] );
    }
    else
    {
      gchar *modes[TX_MODE_ITEMS] = { TX_MODE_MENU_ITEMS };
      button = GTK_BUTTON( Builder_Get_Object( TRx->builder, "tx_modes_menu_button") );
      gtk_button_set_label( button, modes[mode] );
    }
  } // if( TRx )

} // Hermes2_Display_Mode()

//----------------------------------------------------------------------

/* Hpsdr_ADAGC_Smeter()
 *
 * Calculates the S-Meter reading of the Audio AGC
 */
  void
Hpsdr_ADAGC_Smeter( Transceiver_t *TRx )
{
  // ADAGC scale factor in relative dBm
  double log_adagc_scale = 20.0 * log10( TRx->adagc_scale );

  // Total offset value for correct S-meter indication
  TRx->smeter_offset = HERMES2_SMETER_OFFSET;
  TRx->S_meter = log_adagc_scale + TRx->smeter_offset;

  // Substract any LNA Gain setting
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  TRx->S_meter -= ddv->lna_gain;

  // Scale Smeter to range 0.0 - 1.0
  TRx->S_meter /= SMETER_RANGE_DBM;

} // Hpsdr_ADAGC_Smeter()

//----------------------------------------------------------------------

/* Hermes2_Frequency_Status()
 *
 * Displays the status of center frequency setting
 */
  gboolean
Hermes2_Frequency_Status( gpointer data )
{
  Transceiver_t *TRx = (Transceiver_t *)data;

  // Indicate hermes2 Rx Frequency status
  GtkWidget *image = Builder_Get_Object( TRx->builder, "rx_status_image" );
  if( TRx->set_rx_freq_status )
  {
    gtk_image_set_from_icon_name(
        GTK_IMAGE(image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
  }
  else
  {
    gtk_image_set_from_icon_name(
        GTK_IMAGE(image), "gtk-no", GTK_ICON_SIZE_BUTTON );
  }

  // Indicate hermes2 Tx Frequency status
  image = Builder_Get_Object( TRx->builder, "tx_status_image" );
  if( TRx->set_tx_freq_status )
  {
    gtk_image_set_from_icon_name(
        GTK_IMAGE(image), "gtk-yes", GTK_ICON_SIZE_BUTTON );
  }
  else
  {
    gtk_image_set_from_icon_name(
        GTK_IMAGE(image), "gtk-no", GTK_ICON_SIZE_BUTTON );
  }

  return( FALSE );
} // Hermes2_Frequency_Status()

//----------------------------------------------------------------------

  static gboolean
RSID_Mode_idle_cb( gpointer data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  gtk_button_set_label( GTK_BUTTON(TRx->rsid_apply_button), (gchar *)data );
  gtk_image_set_from_icon_name( GTK_IMAGE(TRx->rsid_image), "gtk-ok", GTK_ICON_SIZE_BUTTON );
  return( FALSE );
}

/* RSID_Mode()
 *
 * Displays the RSID Mode and frequency
 */
  void
RSID_Mode( const char *mode, uint32_t freq )
{
  uint16_t mhz, khz, hz;
  gchar str[64];

  mhz = (uint16_t)( freq / 1000000 );
  khz = (uint16_t)( ( freq - mhz * 1000000 ) / 1000 );
  hz  = (uint16_t)( ( freq - mhz * 1000000 - khz * 1000 ) );
  snprintf( str, sizeof(str), "%s - %d %03d %03d Hz", mode, mhz, khz, hz );
  g_idle_add( RSID_Mode_idle_cb, (gpointer)str );
} // RSID_Mode()

//----------------------------------------------------------------------

/* RSID_Status()
 *
 * Displays the status of the RSID listening function
 */
  gboolean
RSID_Status( gpointer data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  if( TRx->rx_rsid_enable )
  {
    // Show search for RSID is active
    gtk_image_set_from_icon_name(
        GTK_IMAGE(TRx->rsid_image), "gtk-find", GTK_ICON_SIZE_BUTTON );
    if( strlen(TRx->RSID_select) == 0 )
    {
      gtk_button_set_label( GTK_BUTTON(TRx->rsid_apply_button), "Listening for ANY MODE" );
    }
    else
    {
      char str[32] = { "Listening for " };
      Strlcat( str, TRx->RSID_select, sizeof(str) );
      gtk_button_set_label( GTK_BUTTON(TRx->rsid_apply_button), (gchar *)str );
    }
  }
  else
  {
    // Show search for RSID is stopped
    gtk_image_set_from_icon_name(
        GTK_IMAGE(TRx->rsid_image), "gtk-stop", GTK_ICON_SIZE_BUTTON );
    gtk_button_set_label( GTK_BUTTON(TRx->rsid_apply_button), "Not Listening" );
  }

  return( FALSE );
} // RSID_Status()

//----------------------------------------------------------------------

