/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "demodulate.h"
#include "callback_func.h"
#include "display.h"
#include "sound.h"
#include "../common/common.h"
#include "../common/filters.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/pc_to_hw.h"
#include "../hpsdr/settings.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <semaphore.h>

//----------------------------------------------------------------------

// Max AFC correction range in Hz
#define MAX_AFC_OFFSET  500.0

// AFC filter bandwidth in SSB/AM-SYNC mode
#define AFC_BANDWIDTH   300.0

// AFC asymptotic convergence factor
#define AFC_CONVERGE  0.7

//----------------------------------------------------------------------

// Length and multiplier for AM averaging window
#define AM_AVE_WINDOW_LENGTH  5000.0
#define AM_AVE_WINDOW_MUL  4999.0   // above -1 always

//----------------------------------------------------------------------

/* Low Pass Filter parameters for FM filters.
 * The number of poles _must_be_even_ */
#define RX_FM_FILTER_POLES     6
#define RX_FM_FILTER_RIPPLE    1.0

//----------------------------------------------------------------------

/* Low Pass Filter parameters for AM filters.
 * The number of poles _must_be_even_ */
#define RX_AM_FILTER_POLES     6
#define RX_AM_FILTER_RIPPLE    1.0

//----------------------------------------------------------------------

/* Low Pass Filter parameters for the AFC and SSB
 * demodulator. The number of poles _must_be_even_ */
#define AFC_FILTER_POLES       4
#define AFC_FILTER_RIPPLE      1.0
#define RX_SSBW_FILTER_POLES   8
#define RX_SSBN_FILTER_POLES   6
#define RX_SSB_FILTER_RIPPLE   1.0

//----------------------------------------------------------------------

/* Sound PCSound Buffer has to be a ring
 * buffer to allow for the differences in the
 * SDR ADC and Sound card ADC sampling rates */
short  **sound_ring_buf;
uint32_t sound_ring_buf_idx;
uint32_t demod_ring_buf_idx;

// I/Q Samples buffers for the AFC LP Filters
static double
*afc_id[MAX_RECEIVERS] = { NULL, NULL, NULL, NULL },
*afc_qd[MAX_RECEIVERS] = { NULL, NULL, NULL, NULL };

//----------------------------------------------------------------------

#define DURATION    16000
#define TONE_FREQ   600.0

/* Demodulated_Signal_Control()
 *
 * Controls the level of the demodulated signal
 * (in Audio AGC mode) and/or saves it in the sound buffer
 */
  static void
Demodulated_Signal_Control(
    Transceiver_t *TRx,
    double demod_signal,
    uint32_t snd_buf_idx )
{
  double signal_ratio;
  uint32_t idx;
  int16_t audio, digi_signal;
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  /* Scale demodulated signal according to
   * modulation mode and SDR device type */
  demod_signal *= TRx->demod_scale;

  // Protect the AGC from stray demod output during beep
  if( Flag[BEEP_ACTIVE] || TRx->receiver_muted )
    demod_signal = 0.0;

  // *** Apply audio derived AGC ***
  /* Ratio of demodulated signal level to reference
   * level. This is about 2/3 of max level the sound
   * system can handle (+/- 32384 for 16-bit audio) */
  signal_ratio = fabs( demod_signal ) / ADAGC_REF_LEVEL;

  // This is the AGC "attack" function
  if( signal_ratio > TRx->adagc_scale )
    TRx->adagc_scale = signal_ratio;
  else // This the AGC "decay" function
    TRx->adagc_scale *= TRx->adagc_decay;

  // Scale demodulated signal as needed (AGC)
  if( TRx->adagc_scale != 0.0 )
    demod_signal /= TRx->adagc_scale;

  /* Save demod signal for digi modes
   * before applying Rx volume control */
  digi_signal = (int16_t)(demod_signal);

  // Apply Rx volume setting
  demod_signal *= TRx->sound_volume;

  // Generate a beep tone if requested, before volume control
  if( Flag[BEEP_ACTIVE] )
  {
    static uint32_t tone = DURATION, gap1 = DURATION, gap2 = DURATION;

    if( gap1 ) // Generate silent gap
    {
      demod_signal = 0.0;
      gap1--;
    }
    if( tone ) // Generate tone
    {
      static double w  = 0;
      static double dw = M_2PI / SND_DSP_RATE * TONE_FREQ;
      demod_signal = hermes2_rc.beep_volume * sin( w );
      w += dw;
      CLAMP_2PI( w );
      tone--;
    }
    else if( gap2 ) // Generate silent gap
    {
      demod_signal = 0.0;
      gap2--;
    }
    else // Reset for next beep
    {
      tone = DURATION;
      gap1 = DURATION;
      gap2 = DURATION;

      // (Re)-Mute Receiver if enabled
      if( TRx->mute_receiver && ddv->transmit_on )
        TRx->receiver_muted = True;

      Flag[BEEP_ACTIVE] = False;
    }
  } // if( Flag[BEEP_ACTIVE] )

  // Convert demodulated signal to sound format (16-bit samples)
  audio = (int16_t)demod_signal;

  // Find first Transceiver that is receiving
  for( idx = 0; idx < Indices.Num_of_TRXs; idx++ )
    if( Transceiver[idx]->receive_active )
      break;

  /* If active Transceiver index is equal
   * to TRx index, queue out audio samples */
  if( TRx->index == idx )
  {
    // Enable local (PC) or remote (Device) sound
    if( hermes2_rc.sound_pc_local )
    {
      // Fill sound buffer frames
      for( idx = 0; idx < SND_NUM_CHANNELS; idx++ )
        sound_ring_buf[demod_ring_buf_idx][snd_buf_idx++] = audio;
    }
  } // if( TRx->index == idx )

  // If any digital mode is active, save audio samples for it
  if( Flag[GUEST_RECEIVING] || TRx->rx_rsid_enable )
  {
    static uint16_t signal_index = 0;

    // Save audio sample for the digital mode function
    digimode_buffer[digi_buf_input][signal_index] = digi_signal;
    signal_index++;

    // Post to semaphore that audio sample is ready
    if( signal_index >= DIGIMODE_BUFFER_SIZE )
    {
      signal_index = 0;

      // Increment current audio sample save buffer index
      digi_buf_input++;
      if( digi_buf_input >= NUM_DIGIMODE_BUFFERS )
        digi_buf_input = 0;

      // Post to digimode semaphore
      int sval;
      sem_getvalue( &digimode_semaphore, &sval );
      if( !sval ) sem_post( &digimode_semaphore );

      // Post to RSID semaphore
      sem_getvalue( &rsid_semaphore, &sval );
      if( !sval ) sem_post( &rsid_semaphore );
    }

  } // if( Flag[GUEST_RECEIVING] || TRx->rx_rsid_enable )

} // Demodulated_Signal_Control()

//----------------------------------------------------------------------

/* Calculate_Frequency_Offset()
 *
 * Calculates the frequency offset between
 * signal carrier and SDR center frequency
 */
  static inline double
Calculate_Frequency_Offset( const Transceiver_t *TRx, double i, double q )
{
  double freq_offset, phi1, dphi;
  static double phi0[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  // Phase angle of current I/Q sample
  phi1 = atan2( i, q );

  // Phase difference from previous to current I/Q sample
  dphi = phi1 - phi0[TRx->index];

  // Save current phase angle as the next previous
  phi0[TRx->index] = phi1;

  // Correct for discontinuity at +-180 deg phase angle
  CLAMP_PI( dphi );

  // Measured signal frequency
  freq_offset = SND_DSP_RATE * dphi / M_2PI;

  return( freq_offset );
} // Calculate_Frequency_Offset()

//----------------------------------------------------------------------

/* Apply_Frequency_Offset()
 *
 * Gdk idle callback function to apply and display
 * corrections by the AFC of the frequency offset
 */
  static gboolean
Apply_Frequency_Offset( gpointer data )
{
  Transceiver_t *TRx = (Transceiver_t *)data;

  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );
  Update_Spin_Dial( TRx, RX_FLAG );

  return( FALSE );
} // Apply_Frequency_Offset()

//----------------------------------------------------------------------

/* Correct_Frequency_Offset()
 *
 * Corrects offset between Signal frequency and
 * Center frequency of tuner, e.g. AFC of sorts
 */
  void *
Correct_Frequency_Offset( void *data )
{
  Transceiver_t *TRx = (Transceiver_t *)data;

  // Loop while AFC frequency correction is enabled
  TRx->afc_thread_running = True;
  while( TRx->afc_enable_status )
  {
    // Wait for AFC data to be ready
    sem_wait( &(TRx->afc_semaphore) );
    if( !TRx->afc_enable_status ) break;

    // Limit max frequency offset correction
    if( TRx->freq_offset_data.norm_offset > MAX_AFC_OFFSET )
      TRx->freq_offset_data.norm_offset = MAX_AFC_OFFSET;
    if( TRx->freq_offset_data.norm_offset < -MAX_AFC_OFFSET )
      TRx->freq_offset_data.norm_offset = -MAX_AFC_OFFSET;

    /* Round frequency offset to AFC_CONVERGE of its
     * value for asymptotic frequency correction */
    TRx->freq_offset_data.ave_offset =
      round( TRx->freq_offset_data.norm_offset * AFC_CONVERGE );

    // Correct for frequency offset if not 0
    if( TRx->freq_offset_data.ave_offset != 0.0 )
    {
      int center_freq = (int)TRx->rx_frequency;
      center_freq += (int)TRx->freq_offset_data.norm_offset;
      if( center_freq < 0 ) center_freq = 0;
      TRx->rx_frequency = (uint32_t)center_freq;
      gdk_threads_add_idle( Apply_Frequency_Offset, TRx );
    }
  } // while( TRx->afc_enable_status )
  TRx->afc_thread_running = False;

  // Exit thread and destroy AFC semaphore
  Init_Semaphore( &(TRx->afc_semaphore), False );

  return( NULL );
} // Correct_Frequency_Offset()

//----------------------------------------------------------------------

/* PCSound_Control()
 *
 * Controls the rotation of demodulator
 * and PC sound write buffers
 */
  static void
PCSound_Control( void )
{
  // Keep some distance between demod and sound buffer indices
  demod_ring_buf_idx++;
  int diff = (int)(demod_ring_buf_idx - sound_ring_buf_idx);
  if( (diff >= -NUM_SOUND_RING_BUFFS / 4) &&
      (diff <=  NUM_SOUND_RING_BUFFS / 4) )
    demod_ring_buf_idx = sound_ring_buf_idx + NUM_SOUND_RING_BUFFS / 2;

  // Keep demod buffer index in range
  if( demod_ring_buf_idx >= NUM_SOUND_RING_BUFFS )
    demod_ring_buf_idx -= NUM_SOUND_RING_BUFFS;

} // PCSound_Control()

//----------------------------------------------------------------------

/* Alloc_Demod_Buffers()
 *
 * Allocates memory to demodulator
 * buffers used for local filtering
 */
  static void
Alloc_Demod_Buffers( Transceiver_t *TRx, uint32_t buf_len )
{
  // Allocate I/Q double buffers
  demod_iq_buf_siz = (size_t)buf_len * sizeof(double);
  Mem_Alloc( (void **) &(TRx->demod_id), demod_iq_buf_siz );
  Mem_Alloc( (void **) &(TRx->demod_qd), demod_iq_buf_siz );
} // Alloc_Demod_Buffers()

//----------------------------------------------------------------------

/* Alloc_Demod_Buf_Copies()
 *
 * Allocates memory to demodulator buffers copies
 */
  void
Alloc_Demod_Buf_Copies( uint32_t buf_len )
{
  // Allocate IQ data copy buffers
  demod_iq_buf_siz = (size_t)buf_len * sizeof(double);
  Mem_Alloc( (void **) &demod_id_cpy, demod_iq_buf_siz );
  Mem_Alloc( (void **) &demod_qd_cpy, demod_iq_buf_siz );
} // Alloc_Demod_Buf_Copies()

//----------------------------------------------------------------------

/* Demodulate_FM()
 *
 * Demodulates FM Signals
 */
  BOOLEAN
Demodulate_FM( Transceiver_t *TRx, uint32_t sound_buf_len )
{
  static uint32_t
    // ADC samples to go into one sound sample
    decimate_cnt[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index of samples buffer for the sound card
    snd_buf_idx[MAX_RECEIVERS]   = { 0, 0, 0, 0 },
    // Index to hermes2's ADC samples buffer
    ddc_buf_idx[MAX_RECEIVERS]   = { 0, 0, 0, 0 },
    // Index to i and q buffers
    iqd_buf_idx[MAX_RECEIVERS]   = { 0, 0, 0, 0 },
    // Saves current bandwidth value
    bandwidth[MAX_RECEIVERS]     = { 0, 0, 0, 0 },
    // Noise Squelch threshold
    squelch_thrld[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  // Measured frequency deviation of sampled IF signal
  static double freq_dev[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  // Last measured phase angle of sampled IF signal
  static double phi_0[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  // Used in squelch code and FM detector
  static double
    dev_diff[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    last_dev[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    delta_f[MAX_RECEIVERS]  = { 0, 0, 0, 0 };

  double
    phi_1,  // Current Phase angle of sampled IF signal
    dphi,   // Change in Phase angle of sampled IF signal
    ifreq;  // Instantaneous frequency of IF signal

  // Demodulated Signal (Audio)
  double demod_signal;

  // Initialize on new Transceiver object and window
  if( TRx->new_receiver )
  {
    bandwidth[TRx->index] = 0;
    TRx->new_receiver = 0;
    return( True );
  }

  // Prepare LP filters on B/W change
  if( TRx->demodulator_init || (bandwidth[TRx->index] != TRx->rx_bandwidth) )
  {
    // Allocate local filter buffers
    Alloc_Demod_Buffers( TRx, sound_buf_len );

    // Noise Squelch threshold
    squelch_thrld[TRx->index] = SQUELCH_THRESHOLD * SQUELCH_RANGE;
    squelch_thrld[TRx->index] = ( squelch_thrld[TRx->index] * 3 ) / 4;

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    double cutoff;
    bandwidth[TRx->index] = TRx->rx_bandwidth;
    cutoff  = (double)bandwidth[TRx->index];
    cutoff /= SND_DSP_RATE * 2.0;
    delta_f[TRx->index] = SND_DSP_RATE / M_2PI;

    // Initialize Demodulator I filter
    TRx->demod_filter_data_i.cutoff = cutoff;
    TRx->demod_filter_data_i.ripple = RX_FM_FILTER_RIPPLE;
    TRx->demod_filter_data_i.npoles = RX_FM_FILTER_POLES;
    TRx->demod_filter_data_i.type = FILTER_LOWPASS;
    TRx->demod_filter_data_i.ring_idx = 0;
    TRx->demod_filter_data_i.samples_buffer = TRx->demod_id;
    TRx->demod_filter_data_i.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_i.init_filter = True;

    // Initialize Demodulator Q filter
    TRx->demod_filter_data_q.cutoff = cutoff;
    TRx->demod_filter_data_q.ripple = RX_FM_FILTER_RIPPLE;
    TRx->demod_filter_data_q.npoles = RX_FM_FILTER_POLES;
    TRx->demod_filter_data_q.type = FILTER_LOWPASS;
    TRx->demod_filter_data_q.ring_idx = 0;
    TRx->demod_filter_data_q.samples_buffer = TRx->demod_qd;
    TRx->demod_filter_data_q.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_q.init_filter = True;

    TRx->freq_offset_data.limit = SND_DSP_RATE;
    TRx->freq_offset_data.ave_offset = 0.0;
    TRx->freq_offset_data.count = 0;
    TRx->demodulator_init = False;

    return( True );
  } // if( Flag[DEMODULATOR_INIT] || )

  // Fill buffer from hermes2 device. buf_len is in snd frames
  sound_buf_len *= SND_NUM_CHANNELS;
  while( !TRx->demodulator_exit )
  {
    while( snd_buf_idx[TRx->index] < sound_buf_len )
    {
      /* Use hermes2_rc.sound_decimate samples to
       * produce one carrier amplitude sample */
      TRx->demod_id[iqd_buf_idx[TRx->index]] = 0.0;
      TRx->demod_qd[iqd_buf_idx[TRx->index]] = 0.0;
      while( decimate_cnt[TRx->index] < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( ddc_buf_idx[TRx->index] >= hermes2_rc.ddc_buffer_length )
        {
          ddc_buf_idx[TRx->index] = 0;
          if( !TRx->demodulator_exit ) return( True );
        }

        // Sum I/Q values as floats
        TRx->demod_id[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_i.samples_buffer[ddc_buf_idx[TRx->index]];
        TRx->demod_qd[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_q.samples_buffer[ddc_buf_idx[TRx->index]];
        ddc_buf_idx[TRx->index]++;

        decimate_cnt[TRx->index]++;
      } // while( decimate_cnt < hermes2_rc.sound_decimate )
      decimate_cnt[TRx->index] = 0;

      iqd_buf_idx[TRx->index]++;
      snd_buf_idx[TRx->index] += SND_NUM_CHANNELS;
    } // While( snd_buf_idx < sound_buf_len )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    // Demodulate filtered I/Q buffers
    DSP_Filter( &(TRx->demod_filter_data_i) );
    DSP_Filter( &(TRx->demod_filter_data_q) );
    for( uint32_t idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
    {
      // Phase angle of current I/Q sample
      phi_1 = atan2(
          TRx->demod_id[iqd_buf_idx[TRx->index]],
          TRx->demod_qd[iqd_buf_idx[TRx->index]] );

      // Phase difference from previous to current I/Q sample
      dphi = phi_1 - phi_0[TRx->index];

      // Save current phase angle as the next previous
      phi_0[TRx->index] = phi_1;

      // Correct for discontinuity at +-180 deg phase angle
      CLAMP_PI( dphi );

      // Measured frequency deviation
      ifreq = delta_f[TRx->index] * dphi;
      freq_dev[TRx->index] = ifreq;

      // Do Noise Squelch calculations
      if( TRx->squelch_value <= squelch_thrld[TRx->index] )
      {
        double dev = freq_dev[TRx->index];
        dev_diff[TRx->index] *= SQUELCH_WINDOW_MULT;
        dev_diff[TRx->index] += fabs( dev - last_dev[TRx->index] );
        dev_diff[TRx->index] /= SQUELCH_WINDOW_LENGTH;
        last_dev[TRx->index] = dev;
      }
      else dev_diff[TRx->index] = 0.0;

      // Suppress output if squelch threshold passed
      if( dev_diff[TRx->index] > (double)TRx->squelch_value )
        demod_signal = 0.0;
      else
        demod_signal = freq_dev[TRx->index];

      // Process and save demodulated signal in startrx buffer
      Demodulated_Signal_Control( TRx, demod_signal, idx );

      // Will correct frequency offset only if enabled
      if( TRx->afc_enable_status )
      {
        // Frequency deviation is averaged over 1 sec
        /* Summate freq offset and count total length
         * (number of samples) */
        TRx->freq_offset_data.ave_offset += ifreq;

        // Gives a rate of 1 AFC per second
        TRx->freq_offset_data.count += 1;

        // Correct center frequency at limit intervals (default 1 sec)
        if( TRx->freq_offset_data.count >= TRx->freq_offset_data.limit )
        {
          // Normalize frequency offset
          TRx->freq_offset_data.norm_offset =
            TRx->freq_offset_data.ave_offset /
            (double)TRx->freq_offset_data.count;
          TRx->freq_offset_data.count = 0;

          // Post to semaphore that AFC data is ready
          int sval;
          sem_getvalue( &(TRx->afc_semaphore), &sval );
          if( !sval ) sem_post( &(TRx->afc_semaphore) );
        }
      } // if( Flag[ENABLE_AFC] )

      iqd_buf_idx[TRx->index]++;
    } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHN )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    /* Advance demodulator and
     * sound ring buffers index */
    if( hermes2_rc.sound_pc_local ) PCSound_Control();

  } // while( !TRx->demodulator_exit )

  TRx->demodulator_exit = False;
  TRx->Demodulator = TRx->New_Demodulator;
  return( True );
} // Demodulate_FM()

//----------------------------------------------------------------------

/* Demodulate_AM()
 *
 * Demodulates AM Signals
 */
  BOOLEAN
Demodulate_AM( Transceiver_t *TRx, uint32_t sound_buf_len )
{
  // Average value of AM amplitude
  static double am_ave[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  static uint32_t
    // Count of ADC samples to go into one sound sample
    decimate_cnt[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    // Index to the samples buffer for the sound card
    snd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to hermes2's ADC samples buffer
    ddc_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to i and q buffers
    iqd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Saves current bandwidth value
    bandwidth[MAX_RECEIVERS]    = { 0, 0, 0, 0 };

  // Demodulated Signal (Audio)
  double demod_signal;

  double
    sig_ampl, // AM Radio Signal Amplitude
    ifreq;    // Instantaneous frequency of IF signal

  // Initialize on new Transceiver object and window
  if( TRx->new_receiver )
  {
    bandwidth[TRx->index] = 0;
    TRx->new_receiver = 0;
    return( True );
  }

  // Prepare LP filters on B/W change
  if( TRx->demodulator_init || (bandwidth[TRx->index] != TRx->rx_bandwidth) )
  {
    // Allocate local filter buffers
    Alloc_Demod_Buffers( TRx, sound_buf_len );

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    double cutoff;
    bandwidth[TRx->index] = TRx->rx_bandwidth;
    cutoff  = (double)bandwidth[TRx->index];
    cutoff /= SND_DSP_RATE * 2.0;

    // Initialize Demodulator I filter
    TRx->demod_filter_data_i.cutoff = cutoff;
    TRx->demod_filter_data_i.ripple = RX_AM_FILTER_RIPPLE;
    TRx->demod_filter_data_i.npoles = RX_AM_FILTER_POLES;
    TRx->demod_filter_data_i.type = FILTER_LOWPASS;
    TRx->demod_filter_data_i.ring_idx = 0;
    TRx->demod_filter_data_i.samples_buffer = TRx->demod_id;
    TRx->demod_filter_data_i.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_i.init_filter = True;

    // Initialize Demodulator Q filter
    TRx->demod_filter_data_q.cutoff = cutoff;
    TRx->demod_filter_data_q.ripple = RX_AM_FILTER_RIPPLE;
    TRx->demod_filter_data_q.npoles = RX_AM_FILTER_POLES;
    TRx->demod_filter_data_q.type = FILTER_LOWPASS;
    TRx->demod_filter_data_q.ring_idx = 0;
    TRx->demod_filter_data_q.samples_buffer = TRx->demod_qd;
    TRx->demod_filter_data_q.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_q.init_filter = True;
    TRx->freq_offset_data.limit = SND_DSP_RATE;
    TRx->freq_offset_data.ave_offset = 0.0;
    TRx->freq_offset_data.count = 0;

    TRx->demodulator_init = False;

    return( True );
  } // if( Flag[DEMODULATOR_INIT] || )

  // Fill buffer from hermes2 device. buf_len is in snd frames
  sound_buf_len *= SND_NUM_CHANNELS;
  while( !TRx->demodulator_exit )
  {
    while( snd_buf_idx[TRx->index] < sound_buf_len )
    {
      /* Use hermes2_rc.sound_decimate samples to
       * produce one carrier amplitude sample */
      TRx->demod_id[iqd_buf_idx[TRx->index]] = 0.0;
      TRx->demod_qd[iqd_buf_idx[TRx->index]] = 0.0;
      while( decimate_cnt[TRx->index] < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( ddc_buf_idx[TRx->index] >= hermes2_rc.ddc_buffer_length )
        {
          ddc_buf_idx[TRx->index] = 0;
          if( !TRx->demodulator_exit ) return( True );
        }

        // Sum I/Q values as floats
        TRx->demod_id[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_i.samples_buffer[ddc_buf_idx[TRx->index]];
        TRx->demod_qd[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_q.samples_buffer[ddc_buf_idx[TRx->index]];
        ddc_buf_idx[TRx->index]++;

        decimate_cnt[TRx->index]++;
      } // while( decimate_cnt < hermes2_rc.sound_decimate )
      decimate_cnt[TRx->index] = 0;

      // Will correct frequency offset only if enabled
      if( TRx->afc_enable_status )
      {
        // Measure and compensate frequency offset
        ifreq = Calculate_Frequency_Offset( TRx,
            TRx->demod_id[iqd_buf_idx[TRx->index]],
            TRx->demod_qd[iqd_buf_idx[TRx->index]] );

        // Frequency deviation is averaged over 1 sec
        /* Summate freq offset and count
         * total length (number of samples) */
        TRx->freq_offset_data.ave_offset += ifreq;

        // Gives a rate of 1 AFC per second
        TRx->freq_offset_data.count += 1;

        // Correct center frequency at limit intervals (default 1 sec)
        if( TRx->freq_offset_data.count >= TRx->freq_offset_data.limit )
        {
          // Normalize frequency offset
          TRx->freq_offset_data.norm_offset =
            TRx->freq_offset_data.ave_offset / (double)TRx->freq_offset_data.count;
          TRx->freq_offset_data.count = 0;

          // Post to semaphore that AFC data is ready
          int sval;
          sem_getvalue( &(TRx->afc_semaphore), &sval );
          if( !sval ) sem_post( &(TRx->afc_semaphore) );
        }
      } // if( Flag[ENABLE_AFC] )

      iqd_buf_idx[TRx->index]++;
      snd_buf_idx[TRx->index] += SND_NUM_CHANNELS;
    } // While( snd_buf_idx < sound_buf_len )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    // Demodulate filtered I/Q buffers
    DSP_Filter( &(TRx->demod_filter_data_i) );
    DSP_Filter( &(TRx->demod_filter_data_q) );
    for( uint32_t idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
    {
      // Amplitude of AM radio signal
      sig_ampl = hypot(
          TRx->demod_id[iqd_buf_idx[TRx->index]],
          TRx->demod_qd[iqd_buf_idx[TRx->index]] );

      // Scale the measured amplitude
      sig_ampl /= (double)TRx->sound_decimate;

      // Remove average (DC) value
      am_ave[TRx->index]  = am_ave[TRx->index] * AM_AVE_WINDOW_MUL + sig_ampl;
      am_ave[TRx->index] /= AM_AVE_WINDOW_LENGTH;

      /* Fill sound buffer frames. Amplitude with average
       * subtracted must be mult. by 2 to scale properly */
      demod_signal = ( sig_ampl - am_ave[TRx->index] );

      // Process and save demodulated signal in startrx buffer
      Demodulated_Signal_Control( TRx, demod_signal, idx );

      iqd_buf_idx[TRx->index]++;
    } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHN )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    /* Advance demodulator and
     * sound ring buffers index */
    if( hermes2_rc.sound_pc_local ) PCSound_Control();

  } // while( !TRx->demodulator_exit )

  TRx->demodulator_exit = False;
  TRx->Demodulator = TRx->New_Demodulator;
  return( True );
} // Demodulate_AM()

//----------------------------------------------------------------------

#define RX_SSB_FILTER_NARROW   500

/* Demodulate_SSB()
 *
 * Demodulates SSB Signals
 */
  BOOLEAN
Demodulate_SSB( Transceiver_t *TRx, uint32_t sound_buf_len )
{
  static uint32_t
    // Count of ADC samples to go into one sound sample
    decimate_cnt[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    // Index to the samples buffer for the sound card
    snd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to hermes2's ADC samples buffer
    ddc_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to i and q buffers
    iqd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Saves current bandwidth value
    bandwidth[MAX_RECEIVERS]    = { 0, 0, 0, 0 },
    weaver[MAX_RECEIVERS]       = { 0, 0, 0, 0 };

  // Saves current modulation mode
  static uint8_t mode[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  // Demodulator filter data structs for above samples buffers
  static filter_data_t
    afc_filter_data_i[MAX_RECEIVERS],
    afc_filter_data_q[MAX_RECEIVERS];

    uint32_t idx;

    double
      cutoff,           // Filter cutoff frequency
      base_band = 0.0,  // SSB Radio Signal's base band
      ifreq;            // Instantaneous frequency of IF signal

    // Delta of Weaver method phase angle per
    // decimated IQ sample (of 48kHz normally)
    static double dphi = 0.0;

    // Weaver method phase angle for AFC and SSB demodulator
    static double phi_afc = 0.0, phi_wvr = 0.0;

    // Abort if zero weaver frequency selected
    if( !TRx->rx_weaver_frequency ) return( True );

    // Initialize on new Transceiver object and window
    if( TRx->new_receiver )
    {
      bandwidth[TRx->index] = 0;
      mode[TRx->index]      = 0;
      weaver[TRx->index]    = 0;
      TRx->new_receiver     = 0;
      return( True );
    }

    // Allocate AFC I/Q double buffers on first call
    if( (afc_id[TRx->index] == NULL) || (afc_qd[TRx->index] == NULL) )
    {
      size_t mem_req;
      mem_req = (size_t)sound_buf_len * sizeof(double);
      Mem_Alloc( (void **) &afc_id[TRx->index], mem_req );
      Mem_Alloc( (void **) &afc_qd[TRx->index], mem_req );

      /* LP Filter cutoff must be specified taking into
       * account the transition band as a fraction of Fc */
      cutoff  = AFC_BANDWIDTH;
      cutoff /= SND_DSP_RATE * 2.0;

      // Initialize Demodulator AFC filter
      afc_filter_data_i[TRx->index].cutoff = cutoff;
      afc_filter_data_i[TRx->index].ripple = AFC_FILTER_RIPPLE;
      afc_filter_data_i[TRx->index].npoles = AFC_FILTER_POLES;
      afc_filter_data_i[TRx->index].type = FILTER_LOWPASS;
      afc_filter_data_i[TRx->index].ring_idx = 0;
      afc_filter_data_i[TRx->index].samples_buffer = afc_id[TRx->index];
      afc_filter_data_i[TRx->index].samples_buffer_len = sound_buf_len;
      afc_filter_data_i[TRx->index].init_filter = True;

      // Initialize Demodulator AFC filter
      afc_filter_data_q[TRx->index].cutoff = cutoff;
      afc_filter_data_q[TRx->index].ripple = AFC_FILTER_RIPPLE;
      afc_filter_data_q[TRx->index].npoles = AFC_FILTER_POLES;
      afc_filter_data_q[TRx->index].type = FILTER_LOWPASS;
      afc_filter_data_q[TRx->index].ring_idx = 0;
      afc_filter_data_q[TRx->index].samples_buffer = afc_qd[TRx->index];
      afc_filter_data_q[TRx->index].samples_buffer_len = sound_buf_len;
      afc_filter_data_q[TRx->index].init_filter = True;

      return( True );
    } // if( (afc_id == NULL) || (afc_qd == NULL) )

    // Initialize on first call or change of params
    if( (mode[TRx->index]   != TRx->rx_modulation_mode)  ||
        (weaver[TRx->index] != TRx->rx_weaver_frequency) )
    {
      /* Calculate trigonometric tables. "Negative" Weaver
       * frequency is used for lower sideband demodulation */
      // Change in Phase angle of sampled IF signal
      switch( TRx->rx_modulation_mode )
      {
        case RX_MODE_USBW:  case RX_MODE_USBM:  case RX_MODE_USBN:
        case RX_MODE_SAMUW: case RX_MODE_SAMUM: case RX_MODE_SAMUN:
        case RX_MODE_RSID:  case RX_MODE_RFAXU: case RX_MODE_SSTVU:
        case RX_MODE_FTX:
          dphi = M_2PI * (double)TRx->rx_weaver_frequency / SND_DSP_RATE;
          break;

        case RX_MODE_LSBW:  case RX_MODE_LSBM:  case RX_MODE_LSBN:
        case RX_MODE_SAMLW: case RX_MODE_SAMLM: case RX_MODE_SAMLN:
        case RX_MODE_RFAXL: case RX_MODE_SSTVL:
          dphi = -M_2PI * (double)TRx->rx_weaver_frequency / SND_DSP_RATE;
          break;
      }

      mode[TRx->index]   = TRx->rx_modulation_mode;
      weaver[TRx->index] = TRx->rx_weaver_frequency;

      TRx->freq_offset_data.limit      = SND_DSP_RATE;
      TRx->freq_offset_data.ave_offset = 0.0;
      TRx->freq_offset_data.count      = 0;

      return( True );
    } // if( ( ((mode[TRx->index] != TRx->modulation_mode) || )

    // Prepare LP filters on B/W change
    if( TRx->demodulator_init || (bandwidth[TRx->index] != TRx->rx_bandwidth) )
    {
      // Allocate local filter buffers if needed
      Alloc_Demod_Buffers( TRx, sound_buf_len );

      /* LP Filter cutoff must be specified taking into
       * account the transition band as a fraction of Fc */
      bandwidth[TRx->index] = TRx->rx_bandwidth;
      cutoff  = (double)bandwidth[TRx->index];
      cutoff /= SND_DSP_RATE * 2.0;

      // Initialize Demodulator I filter
      TRx->demod_filter_data_i.cutoff = cutoff;
      TRx->demod_filter_data_i.ripple = RX_SSB_FILTER_RIPPLE;
      if( bandwidth[TRx->index] < RX_SSB_FILTER_NARROW )
        TRx->demod_filter_data_i.npoles = RX_SSBN_FILTER_POLES;
      else
        TRx->demod_filter_data_i.npoles = RX_SSBW_FILTER_POLES;
      TRx->demod_filter_data_i.type = FILTER_LOWPASS;
      TRx->demod_filter_data_i.ring_idx = 0;
      TRx->demod_filter_data_i.samples_buffer = TRx->demod_id;
      TRx->demod_filter_data_i.samples_buffer_len = sound_buf_len;
      TRx->demod_filter_data_i.init_filter = True;

      // Initialize Demodulator Q filter
      TRx->demod_filter_data_q.cutoff = cutoff;
      TRx->demod_filter_data_q.ripple = RX_SSB_FILTER_RIPPLE;
      if( bandwidth[TRx->index] < RX_SSB_FILTER_NARROW )
        TRx->demod_filter_data_q.npoles = RX_SSBN_FILTER_POLES;
      else
        TRx->demod_filter_data_q.npoles = RX_SSBW_FILTER_POLES;
      TRx->demod_filter_data_q.type = FILTER_LOWPASS;
      TRx->demod_filter_data_q.ring_idx = 0;
      TRx->demod_filter_data_q.samples_buffer = TRx->demod_qd;
      TRx->demod_filter_data_q.samples_buffer_len = sound_buf_len;
      TRx->demod_filter_data_q.init_filter = True;

      TRx->demodulator_init = False;

      return( True );
    } // if( Flag[DEMODULATOR_INIT] || )

    /* Fill buffer from hermes2 device. sound_buf_len is in frames */
    sound_buf_len *= SND_NUM_CHANNELS;
    while( !TRx->demodulator_exit )
    {
      while( snd_buf_idx[TRx->index] < sound_buf_len )
      {
        /* Use hermes2_rc.sound_decimate samples to
         * produce one carrier amplitude sample */
        TRx->demod_id[iqd_buf_idx[TRx->index]] = 0.0;
        TRx->demod_qd[iqd_buf_idx[TRx->index]] = 0.0;
        while( decimate_cnt[TRx->index] < TRx->sound_decimate )
        {
          // Wait for hermes2 buffer to refill
          if( ddc_buf_idx[TRx->index] >= hermes2_rc.ddc_buffer_length )
          {
            ddc_buf_idx[TRx->index] = 0;
            if( !TRx->demodulator_exit ) return( True );
          }

          // Sum I/Q values as floats
          TRx->demod_id[iqd_buf_idx[TRx->index]] +=
            TRx->roof_filter_i.samples_buffer[ddc_buf_idx[TRx->index]];
          TRx->demod_qd[iqd_buf_idx[TRx->index]] +=
            TRx->roof_filter_q.samples_buffer[ddc_buf_idx[TRx->index]];
          ddc_buf_idx[TRx->index]++;

          decimate_cnt[TRx->index]++;
        } // while( decimate_cnt < hermes2_rc.sound_decimate )
        decimate_cnt[TRx->index] = 0;

        iqd_buf_idx[TRx->index]++;
        snd_buf_idx[TRx->index] += SND_NUM_CHANNELS;
      } // while( snd_buf_idx < buf_len )
      iqd_buf_idx[TRx->index] = 0;
      snd_buf_idx[TRx->index] = 0;

      // Measure and compensate frequency offset
      if( TRx->afc_enable_status )
      {
        /* Translate frequency of I/Q sample stream by Weaver frequency.
         * This is to shift the carrier frequency of SYNC AM signals to 0 Hz */
        for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
        {
          afc_id[TRx->index][iqd_buf_idx[TRx->index]] =
            TRx->demod_id[iqd_buf_idx[TRx->index]] * sin( phi_afc ) -
            TRx->demod_qd[iqd_buf_idx[TRx->index]] * cos( phi_afc );
          afc_qd[TRx->index][iqd_buf_idx[TRx->index]] =
            TRx->demod_qd[iqd_buf_idx[TRx->index]] * sin( phi_afc ) +
            TRx->demod_id[iqd_buf_idx[TRx->index]] * cos( phi_afc );

          // Step and limit Weaver phase angle
          phi_afc += dphi;
          CLAMP_N2PI( phi_afc );

          iqd_buf_idx[TRx->index]++;
        } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
        iqd_buf_idx[TRx->index] = 0;

        // Low-pass filter AFC samples buffers
        DSP_Filter( &afc_filter_data_i[TRx->index] );
        DSP_Filter( &afc_filter_data_q[TRx->index] );

        // Find frequency offset
        for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
        {
          // Measured signal frequency offset
          ifreq = Calculate_Frequency_Offset( TRx,
              afc_id[TRx->index][iqd_buf_idx[TRx->index]],
              afc_qd[TRx->index][iqd_buf_idx[TRx->index]] );

          // Frequency deviation is averaged over 1 sec
          /* Summate freq offset and count
           * total length (number of samples) */
          TRx->freq_offset_data.ave_offset += ifreq;

          // Gives a rate of 1 AFC per second
          TRx->freq_offset_data.count += 1;

          iqd_buf_idx[TRx->index]++;
        } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
        iqd_buf_idx[TRx->index] = 0;

        // Correct center frequency at limit intervals (default 1 sec)
        if( TRx->freq_offset_data.count >= TRx->freq_offset_data.limit )
        {
          // Normalize frequency offset
          TRx->freq_offset_data.norm_offset =
            TRx->freq_offset_data.ave_offset /
            (double)TRx->freq_offset_data.count;
          TRx->freq_offset_data.count = 0;

          // Post to semaphore that AFC data is ready
          int sval;
          sem_getvalue( &(TRx->afc_semaphore), &sval );
          if( !sval ) sem_post( &(TRx->afc_semaphore) );
        } // if( freq_offset_data.count >= freq_offset_data.limit )
      } // if( TRx->afc_enable_status )

      // Filter I/Q buffers to set bnadwidth
      DSP_Filter( &(TRx->demod_filter_data_i) );
      DSP_Filter( &(TRx->demod_filter_data_q) );

      // These IQ data copies may be used by guest modes
      if( Flag[GUEST_DEMOD_IQ_DATA] )
      {
        memcpy( demod_id_cpy, TRx->demod_id, demod_iq_buf_siz );
        memcpy( demod_qd_cpy, TRx->demod_qd, demod_iq_buf_siz );
      }

      // Demodulate filtered I/Q buffers
      for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
      {
        // Apply Weaver SSB demodulator method to get base band
        base_band =
          TRx->demod_id[iqd_buf_idx[TRx->index]] * cos( phi_wvr ) +
          TRx->demod_qd[iqd_buf_idx[TRx->index]] * sin( phi_wvr );

        // Step and limit Weaver phase angle
        phi_wvr += dphi;
        CLAMP_N2PI( phi_wvr );

        // Scale the base band value
        base_band /= (double)TRx->sound_decimate;

        // Process and save demodulated signal in startrx buffer
        Demodulated_Signal_Control( TRx, base_band, idx );

        iqd_buf_idx[TRx->index]++;
      } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHN )
      iqd_buf_idx[TRx->index] = 0;
      snd_buf_idx[TRx->index] = 0;

      // Advance demodulator's and sound ring buffer's index
      if( hermes2_rc.sound_pc_local ) PCSound_Control();

    } // while( !TRx->demodulator_exit )

    mode[TRx->index]   = RX_MODE_ITEMS;
    weaver[TRx->index] = 0;

    TRx->demodulator_exit = False;
    TRx->Demodulator = TRx->New_Demodulator;
    return( True );
} // Demodulate_SSB()

//----------------------------------------------------------------------

/* Demodulate_CW()
 *
 * Demodulates CW Signals
 */
  BOOLEAN
Demodulate_CW( Transceiver_t *TRx, uint32_t sound_buf_len )
{
  static uint32_t
    // Count of ADC samples to go into one sound sample
    decimate_cnt[MAX_RECEIVERS] = { 0, 0, 0, 0 },
    // Index to the samples buffer for the sound card
    snd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to hermes2's ADC samples buffer
    ddc_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Index to i and q buffers
    iqd_buf_idx[MAX_RECEIVERS]  = { 0, 0, 0, 0 },
    // Saves current bandwidth and weaver frequency
    bandwidth[MAX_RECEIVERS]    = { 0, 0, 0, 0 },
    weaver[MAX_RECEIVERS]       = { 1000, 1000, 1000, 1000 };

  // Saves current modulation mode
  static uint8_t mode[MAX_RECEIVERS] = { 0, 0, 0, 0 };

  double
    base_band = 0.0, // CW Radio Signal's base band
    ifreq;           // Instantaneous frequency of IF signal

  // Delta of Weaver method phase angle per
  // decimated IQ sample (of 48kHz normally)
  static double dphi = 0.0;

  // Weaver method phase angle for SSB demodulator
  static double phi_wvr = 0.0;


  // Abort if zero weaver frequency selected
  if( !TRx->rx_weaver_frequency ) return( True );

  // Initialize on new Transceiver object and window
  if( TRx->new_receiver )
  {
    bandwidth[TRx->index] = 0;
    mode[TRx->index]      = 0;
    weaver[TRx->index]    = 0.0;
    TRx->new_receiver     = 0;
    return( True );
  }

  // Initialize on first call or change of mode
  if( (mode[TRx->index]   != TRx->rx_modulation_mode)  ||
      (weaver[TRx->index] != TRx->rx_weaver_frequency) )
  {
    /* Calculate trigonometric tables. "Negative" Weaver
     * frequency is used for lower sideband demodulation */
    switch( TRx->rx_modulation_mode )
    {
      case RX_MODE_CWUW:    case RX_MODE_CWUM:   case RX_MODE_CWUN:
      case RX_MODE_PSK31U:  case RX_MODE_WSPR:   case RX_MODE_32_1000:
      case RX_MODE_16_1000: case RX_MODE_16_500: case RX_MODE_8_500:
      case RX_MODE_8_250:   case RX_MODE_4_250:  case RX_MODE_4_125:
      case RX_MODE_HELLW:   case RX_MODE_HELLM:  case RX_MODE_HELLN:
      case RX_MODE_RTTYW:   case RX_MODE_RTTYM:  case RX_MODE_RTTYN:
        dphi = M_2PI * (double)TRx->rx_weaver_frequency / SND_DSP_RATE;
        break;

      case RX_MODE_CWLW: case RX_MODE_CWLM:
      case RX_MODE_CWLN: case RX_MODE_PSK31L:
        dphi = -M_2PI * (double)TRx->rx_weaver_frequency / SND_DSP_RATE;
        break;
    }

    mode[TRx->index]   = TRx->rx_modulation_mode;
    weaver[TRx->index] = TRx->rx_weaver_frequency;

    TRx->freq_offset_data.limit      = SND_DSP_RATE;
    TRx->freq_offset_data.ave_offset = 0.0;
    TRx->freq_offset_data.count      = 0;

    return( True );
  } // if( mode != hermes2_rc.modulation_mode )

  // Prepare LP filters on B/W change
  if( TRx->demodulator_init || (bandwidth[TRx->index] != TRx->rx_bandwidth) )
  {
    // Allocate local filter buffers
    Alloc_Demod_Buffers( TRx, sound_buf_len );

    /* LP Filter cutoff must be specified taking into
     * account the transition band as a fraction of Fc */
    double cutoff;
    bandwidth[TRx->index] = TRx->rx_bandwidth;
    cutoff  = (double)bandwidth[TRx->index];
    cutoff /= SND_DSP_RATE * 2.0;

    // Initialize Demodulator I filter
    TRx->demod_filter_data_i.cutoff = cutoff;
    TRx->demod_filter_data_i.ripple = RX_CW_FILTER_RIPPLE;
    TRx->demod_filter_data_i.npoles = RX_CW_FILTER_POLES;
    TRx->demod_filter_data_i.type   = FILTER_LOWPASS;
    TRx->demod_filter_data_i.ring_idx = 0;
    TRx->demod_filter_data_i.samples_buffer = TRx->demod_id;
    TRx->demod_filter_data_i.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_i.init_filter = True;

    // Initialize Demodulator Q filter
    TRx->demod_filter_data_q.cutoff = cutoff;
    TRx->demod_filter_data_q.ripple = RX_CW_FILTER_RIPPLE;
    TRx->demod_filter_data_q.npoles = RX_CW_FILTER_POLES;
    TRx->demod_filter_data_q.type = FILTER_LOWPASS;
    TRx->demod_filter_data_q.ring_idx = 0;
    TRx->demod_filter_data_q.samples_buffer = TRx->demod_qd;
    TRx->demod_filter_data_q.samples_buffer_len = sound_buf_len;
    TRx->demod_filter_data_q.init_filter = True;

    TRx->demodulator_init = False;

    return( True );
  } // if( Flag[DEMODULATOR_INIT] || )

  // Fill buffer from hpsdr device. buf_len is in frames
  sound_buf_len *= SND_NUM_CHANNELS;
  while( !TRx->demodulator_exit )
  {
    while( snd_buf_idx[TRx->index] < sound_buf_len )
    {
      /* Use hermes2_rc.sound_decimate samples to
       * produce one carrier amplitude sample */
      TRx->demod_id[iqd_buf_idx[TRx->index]] = 0.0;
      TRx->demod_qd[iqd_buf_idx[TRx->index]] = 0.0;
      while( decimate_cnt[TRx->index] < TRx->sound_decimate )
      {
        // Wait for hermes2 buffer to refill
        if( ddc_buf_idx[TRx->index] >= hermes2_rc.ddc_buffer_length )
        {
          ddc_buf_idx[TRx->index] = 0;
          if( !TRx->demodulator_exit ) return( True );
        }

        // Sum I/Q values as floats
        TRx->demod_id[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_i.samples_buffer[ddc_buf_idx[TRx->index]];
        TRx->demod_qd[iqd_buf_idx[TRx->index]] +=
          TRx->roof_filter_q.samples_buffer[ddc_buf_idx[TRx->index]];

        ddc_buf_idx[TRx->index]++;
        decimate_cnt[TRx->index]++;
      } // while( decimate_cnt < hermes2_rc.sound_decimate )
      decimate_cnt[TRx->index] = 0;

      iqd_buf_idx[TRx->index]++;
      snd_buf_idx[TRx->index] += SND_NUM_CHANNELS;
    } // while( snd_buf_idx < buf_len )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    // Demodulate filtered I/Q buffers
    DSP_Filter( &(TRx->demod_filter_data_i) );
    DSP_Filter( &(TRx->demod_filter_data_q) );

    // These IQ data copies may be used by guest modes
    if( Flag[GUEST_DEMOD_WVR_DATA] )
    {
      memcpy( demod_id_cpy, TRx->demod_id, demod_iq_buf_siz );
      memcpy( demod_qd_cpy, TRx->demod_qd, demod_iq_buf_siz );
    }

    for( uint32_t idx = 0; idx < sound_buf_len; idx += SND_NUM_CHANNELS )
    {
      // Apply Weaver Demodulation method for CW detection
      base_band =
        TRx->demod_id[iqd_buf_idx[TRx->index]] * cos( phi_wvr ) +
        TRx->demod_qd[iqd_buf_idx[TRx->index]] * sin( phi_wvr );

      // Step and limit Weaver phase angle
      phi_wvr += dphi;
      CLAMP_N2PI( phi_wvr );

      // Scale the base-band value
      base_band /= (double)TRx->sound_decimate;

      // Process and save demodulated signal in startrx buffer
      Demodulated_Signal_Control( TRx, base_band, idx );

      // Will correct frequency offset only if enabled
      if( TRx->afc_enable_status )
      {
        // Measure and compensate frequency offset
        ifreq = Calculate_Frequency_Offset( TRx,
            TRx->demod_id[iqd_buf_idx[TRx->index]],
            TRx->demod_qd[iqd_buf_idx[TRx->index]] );

        // Frequency deviation is averaged over 1 sec
        /* Summate freq offset and count
         * total length (number of samples) */
        TRx->freq_offset_data.ave_offset += ifreq;

        // Gives a rate of 1 AFC per second
        TRx->freq_offset_data.count += 1;

        // Correct center frequency at limit intervals (default 1 sec)
        if( TRx->freq_offset_data.count >= TRx->freq_offset_data.limit )
        {
          // Normalize frequency offset
          TRx->freq_offset_data.norm_offset =
            TRx->freq_offset_data.ave_offset /
            (double)TRx->freq_offset_data.count;
          TRx->freq_offset_data.count = 0;

          // Post to semaphore that AFC data is ready
          int sval;
          sem_getvalue( &(TRx->afc_semaphore), &sval );
          if( !sval ) sem_post( &(TRx->afc_semaphore) );
        }
      } // if( Flag[ENABLE_AFC] )

      iqd_buf_idx[TRx->index]++;
    } // for( idx = 0; idx < sound_buf_len; idx += SND_NUM_CHN )
    iqd_buf_idx[TRx->index] = 0;
    snd_buf_idx[TRx->index] = 0;

    /* Advance demodulator and
     * sound ring buffers index */
    if( hermes2_rc.sound_pc_local ) PCSound_Control();

  } // while( !TRx->demodulator_exit )

  TRx->demodulator_exit = False;
  TRx->Demodulator = TRx->New_Demodulator;
  return( True );
}  // Demodulate_CW()

//----------------------------------------------------------------------

// Free resources
  void
Free_Demod_Buffers( Transceiver_t *TRx )
{
  Mem_Free( (void **) &(TRx->demod_id) );
  Mem_Free( (void **) &(TRx->demod_qd) );
  Mem_Free( (void **) &afc_id[TRx->index] );
  Mem_Free( (void **) &afc_qd[TRx->index] );
}

//----------------------------------------------------------------------

// Free resources
  void
Free_Demod_Buf_Copies( void )
{
  // Allocate IQ data copy buffers
  Mem_Free( (void **) &demod_id_cpy );
  Mem_Free( (void **) &demod_qd_cpy );
} // Free_Demod_Buf_Copies()

//----------------------------------------------------------------------

