/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "callbacks.h"
#include "bookmarks.h"
#include "callback_func.h"
#include "demodulate.h"
#include "display.h"
#include "interface.h"
#include "process.h"
#include "sound.h"
#include "spectrum.h"
#include "../common/common.h"
#include "../common/guest_utils.h"
#include "../common/hermes2_rc.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/pc_to_hw.h"
#include "../hpsdr/settings.h"
#include "../rsid/rsid.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

static GtkWidget *destroy_widget = NULL;

//----------------------------------------------------------------------

/*
 * All the callback functions of the HPSDR client "hermes2"
 */
  void
on_hermes2_window_destroy(
    GObject       *object,
    gpointer      user_data )
{
  /* Free Hermes2 objects */
  if( hermes2_gui.window_builder != NULL )
    g_object_unref( G_OBJECT(hermes2_gui.window_builder) );

  // Wait for GTK to complete its tasks
  while( g_main_context_iteration(NULL, FALSE) );
  gtk_main_quit();
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_window_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  Quit_Dialog();
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_hermes2_quit_button_clicked(
    GtkButton   *button,
    gpointer    user_data )
{
  Quit_Dialog();
}

//----------------------------------------------------------------------

  void
on_hermes2_quit_dialog_response(
    GtkDialog       *dialog,
    gint             response_id,
    gpointer         user_data )
{
  Quit_Dialog_Response( dialog, response_id );
}

//----------------------------------------------------------------------

  void
on_hermes2_quit_dialog_destroy(
    GObject       *object,
    gpointer       user_data )
{
  hermes2_gui.quit_dialog = NULL;
}

//----------------------------------------------------------------------

  void
on_adc_addrx_button_clicked(
    GtkButton   *button,
    gpointer     user_data )
{
  // Stop the Radio stream
  if( Indices.Num_of_TRXs > 0 )
  {
    Transceiver[Indices.TRx_Index]->receive_active = False;
    Device_Start_Stop( STOP_RADIO_STREAM );
  }

  // Copy Transceiver settings if an additional TCVR window has been created
  ADC_Add_Transceiver();
  if( Indices.Num_of_TRXs > 1 )
  {
    Copy_TCVR_Settings( Indices.TRx_Index - 1, Indices.TRx_Index );
    Restore_TCVR_GUI();
  }
}

//----------------------------------------------------------------------

  gboolean
on_tcvr_window_key_press(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  if( ((GdkEventKey *)event)->keyval == ' ' )
  {
    // Get the window's title and find its index number
    const gchar *title = gtk_window_get_title( GTK_WINDOW(widget) );
    title = strstr( title, "TRX" );
    Indices.TRx_Index = (uint8_t)atoi( title + 3 );
    GtkWidget *button =
      Builder_Get_Object( Transceiver[Indices.TRx_Index]->builder, "mox_togglebutton" );
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) )
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), FALSE );
    else
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), TRUE );
  }

  return( TRUE );
}

//----------------------------------------------------------------------

  gboolean
on_tcvr_window_enter_notify(
    GtkWidget *widget,
    GdkEvent  *event,
    gpointer   user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(widget) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  return( TRUE );
}

//----------------------------------------------------------------------

  gboolean
on_tcvr_window_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  // Do not delete last Transceiver window
  if( hermes2_rc.num_of_receivers == 1 )
  {
    Message_Dialog( _("The last Transceiver Window must not be closed") );
    return( TRUE );
  }

  // Cleanup Transceiver state before destroying window
  Transceiver_Window_Delete( widget );
  destroy_widget = widget;
  Message_Dialog(
      _("NOTE: It is necessary to stop all Receive\n"\
        "activity when closing a Transceiver window.\n"\
        "Close Transceiver Window?") );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_tcvr_window_destroy(
    GObject       *object,
    gpointer      user_data )
{
  destroy_widget = NULL;
  Transceiver_Window_Destroy( object );
}

//----------------------------------------------------------------------

  gboolean
on_freq_scroll(
    GtkWidget       *widget,
    const GdkEvent  *event,
    gpointer         user_data )
{
  gboolean ret = Frequency_Scroll_Event( widget, event );
  return( ret );
}

//----------------------------------------------------------------------

  gboolean
on_freq_button_press(
    GtkWidget       *widget,
    const GdkEventButton *event,
    gpointer         user_data )
{
  gboolean ret = Frequency_Button_Press_Event( widget, event );
  return( ret );
}

//----------------------------------------------------------------------

  void
on_to_tx_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer     user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Transceiver[Indices.TRx_Index]->link_tx_rx_freq = True;
  else
    Transceiver[Indices.TRx_Index]->link_tx_rx_freq = False;
}

//----------------------------------------------------------------------

  void
on_afc_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  uint8_t idx = (uint8_t)atoi( title + 3 );
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  Afc_Checkbutton_Toggled( idx, active );
}

//----------------------------------------------------------------------

  void
on_rx_modes_menu_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Modes_Menu_Button_Clicked( RX_FLAG );
}

//----------------------------------------------------------------------

  void
on_tx_modes_menu_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Modes_Menu_Button_Clicked( TX_FLAG );
}

//----------------------------------------------------------------------

  void
on_guest_modes_menu_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Guest_Modes_Menu_Button_Clicked();
}

//----------------------------------------------------------------------

  void
on_rx_bands_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  Rx_Bands_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_tx_bands_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  if( !ddv->transmit_on )
    Tx_Bands_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_tx_power_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  /* The value of the range slider is 0-5 in steps of 0.1 but we add 0.5
   * to allow the power control function to raise the power to 5.0 Watts */
  hermes2_rc.tx_power_limit = (double)gtk_range_get_value( range ) + 0.5;
}

//----------------------------------------------------------------------

  void
on_mic_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );

  // Set Mic Level
  gdouble value = gtk_range_get_value( range );
  Transceiver[Indices.TRx_Index]->mic_level = (uint8_t)value;
  hermes2_rc.capture_level = (int)value;

  // Set Capture Level. Failure is not considered fatal
  char err_mesg[MESG_STRING_SIZE];
  if( !Set_Capture_Level(hermes2_rc.capture_level, err_mesg) )
    Message_Dialog( err_mesg );
}

//----------------------------------------------------------------------

  void
on_tx_pa_enable_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  PA_Enable_Button_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_mic_comp_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)) )
    Transceiver[Indices.TRx_Index]->mic_compress = True;
  else
    Transceiver[Indices.TRx_Index]->mic_compress = False;
}

//----------------------------------------------------------------------

  void
on_mic_lpf_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)) )
    Transceiver[Indices.TRx_Index]->mic_filter = True;
  else
    Transceiver[Indices.TRx_Index]->mic_filter = False;
}

//----------------------------------------------------------------------

  void
on_tune_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( Flag[GUEST_RECEIVING] || Flag[GUEST_TRANSMITTING] )
    return;
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  Tune_Button_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_mox_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer     user_data )
{
  if( Flag[GUEST_RECEIVING] || Flag[GUEST_TRANSMITTING] )
    return;

  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  MOX_Button_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_rx_weaver_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  Rx_Weaver_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_tx_weaver_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  Tx_Weaver_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_rx_bw_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  Rx_Bandwidth_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_tx_bw_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  Tx_Bandwidth_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_sample_rate_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  Rate_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_fft_bw_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  FFT_Bandwidth_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_fft_rate_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  FFT_Rate_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  void
on_volume_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  gdouble vol = gtk_range_get_value( range );
  vol /= 100.0;
  vol *= vol;
  Transceiver[Indices.TRx_Index]->sound_volume = vol;
}

//----------------------------------------------------------------------

  void
on_agc_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  /* Range is 10-100 and the formula below was derived
   * by trial and error to give a good compromise in
   * ADAGC attack/decay characteristics */
  gdouble val = gtk_range_get_value( range );
  Transceiver[Indices.TRx_Index]->adagc_decay = 1.0 - val / ADAGC_DIVISOR;
}

//----------------------------------------------------------------------

  void
on_squelch_hscale_value_changed(
    GtkRange        *range,
    gpointer         user_data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  if( TRx->squelch_on_status )
  {
    TRx->squelch_value =
      SQUELCH_RANGE - (uint16_t)( gtk_range_get_value(range) );
    TRx->squelch_value *= SQUELCH_THRESHOLD;
  }
  else
    TRx->squelch_value = SQUELCH_RANGE * SQUELCH_THRESHOLD;
}

//----------------------------------------------------------------------

  void
on_squelch_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  Squelch_Checkbutton_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_adc_antenna1_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
    if( !ddv->transmit_on ) Adc_Change_Antenna( 1 );
  }
}

//----------------------------------------------------------------------

  void
on_adc_antenna2_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    const discovered_device_t *ddv = &Device[hermes2_rc.device_index];
    if( !ddv->transmit_on ) Adc_Change_Antenna( 2 );
  }
}

//----------------------------------------------------------------------

  void
on_tx_rx_duplex_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  /* NOTE: Hermes Lite 2 is always in Duplex. Disabling duplex will
   * internally divert all frequency settings to the transmitter! */
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    ddv->duplex = True;
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] |= BIT_2; // Set bit [2] = Duplex
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] |= BIT_2; // Set bit [2] = Duplex
  }
  else
  {
    ddv->duplex = False;
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][4] &= ~BIT_2; // Clear bit [2] = Simplex
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][4] &= ~BIT_2; // Clear bit [2] = Simplex
  }
}

//----------------------------------------------------------------------

  void
on_lna_gain_auto_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( gtk_toggle_button_get_active(togglebutton) )
    ddv->lna_gain_auto = True;
  else
    ddv->lna_gain_auto = False;
}

//----------------------------------------------------------------------

  void
on_lna_gain_hw_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  if( gtk_toggle_button_get_active(togglebutton) )
    ddv->lna_gain_hw = True;
  else
    ddv->lna_gain_hw = False;
}

//----------------------------------------------------------------------

  void
on_lna_gain_value_changed(
    GtkSpinButton *spinbutton,
    gpointer       user_data )
{
  int8_t gain = (int8_t)gtk_spin_button_get_value_as_int( spinbutton );
  Hpsdr_Set_LNA_Gain( gain );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_button_clicked(
    GtkButton   *button,
    gpointer     user_data )
{
  Bookmarks_Button_Clicked( user_data );
}

//----------------------------------------------------------------------

  void
on_rx_start_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  // Get the Transceiver window's title
  const gchar *title = gtk_window_get_title( GTK_WINDOW((Transceiver_t *)user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  StartRx_Button_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_hermes2_error_dialog_response(
    GtkDialog       *dialog,
    gint             response_id,
    gpointer         user_data )
{
  if( response_id == GTK_RESPONSE_CLOSE )
  {
    // Free any digimode semaphore blocks
    if( Flag[GUEST_RECEIVING] )
    {
      Flag[GUEST_RECEIVE_MODE] = False;
      int sval;
      sem_getvalue( &digimode_semaphore, &sval );
      if( !sval ) sem_post( &digimode_semaphore );
    }

    // Save GUI state for restoring windows
    Hermes2_Stop();
    Get_GUI_Geometry();
    Save_Hermes2_Config( FROM_HERMES );
    if( hermes2_gui.window != NULL )
      gtk_widget_destroy( hermes2_gui.window );
    else gtk_main_quit();
  }
  else if( response_id == GTK_RESPONSE_OK )
    gtk_widget_destroy( hermes2_gui.error_dialog );
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_error_dialog_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_hermes2_error_dialog_destroy(
    GObject       *object,
    gpointer      user_data )
{
  hermes2_gui.error_dialog = NULL;
}

//----------------------------------------------------------------------

  void
on_discovery_dialog_response(
    GtkDialog       *dialog,
    gint             response_id,
    gpointer         user_data )
{
  Discovery_Dialog_Response( dialog, response_id );
}

//----------------------------------------------------------------------

  gboolean
on_discovery_dialog_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  gtk_widget_destroy( hermes2_gui.discovery_dialog );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_discovery_dialog_destroy(
    GObject       *object,
    gpointer      user_data )
{
  hermes2_gui.discovery_dialog = NULL;
}

//----------------------------------------------------------------------

  void
on_hermes2_message_dialog_response(
    GtkDialog       *dialog,
    gint             response_id,
    gpointer         user_data )
{
  gtk_widget_destroy( hermes2_gui.message_dialog );

  // Quit hermes2
  if( response_id == GTK_RESPONSE_CLOSE )
  {
    if( !Flag[HERMES2_CREATE_CONFIG] )
    {
      Get_GUI_Geometry();
      Save_Hermes2_Config( FROM_HERMES );
    }

    if( hermes2_gui.window != NULL )
      gtk_widget_destroy( hermes2_gui.window );
    else
      gtk_main_quit();
  }
  else if( response_id == GTK_RESPONSE_OK )
  {
    // Save Configuration if activated
    if( Flag[HERMES2_CREATE_CONFIG] )
    {
      Flag[HERMES2_CREATE_CONFIG] = False;
      Hermes2_Stop();
      Save_Default_Config();
      return;
    }

    /* Destroy the widget that opened the message dialog
     * Currently this is only an open Transceiver window */
    if( Indices.Num_of_TRXs == 1 ) return;
    if( destroy_widget != NULL )
    {
      gtk_widget_destroy( destroy_widget );
      destroy_widget = NULL;
    }
  }
}

//----------------------------------------------------------------------

  void
on_hermes2_message_dialog_destroy(
    GObject       *object,
    gpointer      user_data )
{
  hermes2_gui.message_dialog = NULL;
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_bmk_window_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  Save_Bookmarks_File();
  return( FALSE );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_window_destroy(
    GObject       *object,
    gpointer       user_data )
{
  hermes2_bmk_store = NULL;
  hermes2_bmk_treeview = NULL;
  Indices.Bookmarks_TRx_Idx = -1;
  g_object_unref( hermes2_gui.bmk_window_builder );
  hermes2_gui.bmk_window = NULL;
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_treeview_cursor_changed(
    GtkTreeView     *treeview,
    gpointer         user_data )
{
  Bookmarks_Cursor_Changed( treeview );
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_bmk_treeview_button_press(
    GtkWidget       *widget,
    const GdkEventButton *event,
    gpointer         user_data )
{
  return( Treeview_Button_Press( event) );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_treeview_row_expanded(
    GtkTreeView *tree_view,
    GtkTreeIter *iter,
    GtkTreePath *path,
    gpointer     user_data )
{
  static GtkTreePath *last_path = NULL;

  // Collapse a previously expanded parent row
  if( (last_path != NULL) && gtk_tree_path_compare(last_path, path) != 0 )
  {
    gtk_tree_view_collapse_row( tree_view, last_path );
    gtk_tree_path_free( last_path );
  }
  last_path = gtk_tree_path_copy( path );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_down_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Select_Treeview_Row( TREEVIEW_DOWN );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_up_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Select_Treeview_Row( TREEVIEW_UP );
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_delete_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Delete_Bookmarks_Row();
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_new_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  New_Bookmarks_Row();
}

//----------------------------------------------------------------------

  void
on_hermes2_bmk_save_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Flag[HERMES2_SAVE_BOOKMARKS] = True;
  Save_Bookmarks_File();
}

//----------------------------------------------------------------------

  void
on_wideband_spectrum_radiobutton_toggled(
    GtkRadioButton   *radiobutton,
    gpointer          user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(radiobutton) );
  ADC_Wideband_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_receiver_spectrum_radiobutton_toggled(
    GtkRadioButton *radiobutton,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(radiobutton) );
  Spectrum_Button_Toggled( active );
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_spectrum_drawingarea_draw(
    GtkWidget   *widget,
    cairo_t     *cr,
    gpointer    user_data )
{
  Hermes2_Spectrum_Draw( widget, cr, user_data );
  return( TRUE );
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_spectrum_drawingarea_button_press(
    GtkWidget       *widget,
    const GdkEventButton  *event,
    gpointer         user_data )
{
  Hermes2_Tune_to_Monitor( event->x, user_data );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_hermes2_spectrum_drawingarea_realize(
    GtkWidget       *widget,
    gpointer         user_data )
{
  // Get the window's title and find its index number
  const gchar *title = gtk_window_get_title( GTK_WINDOW(user_data ) );
  title = strstr( title, "TRX" );
  Indices.TRx_Index = (uint8_t)atoi( title + 3 );

  // Allocate size to waterfall pixbuff
  Hermes2_Spectrum_Size_Alloc( widget );
}

//----------------------------------------------------------------------

  void
on_hermes2_spectrum_drawingarea_size_allocate(
    GtkWidget       *widget,
    GdkRectangle    *allocation,
    gpointer         user_data )
{
  hermes2_rc.spectrum_height = (uint16_t)allocation->height;
  hermes2_rc.spectrum_width  = (uint16_t)allocation->width;
}

//----------------------------------------------------------------------

  void
on_rx_modes_menuitem_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data )
{
  Modes_Menu_Item_Activate( menuitem, RX_FLAG );
}

//----------------------------------------------------------------------

  void
on_tx_modes_menuitem_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data )
{
  Modes_Menu_Item_Activate( menuitem, TX_FLAG );
}

//----------------------------------------------------------------------

  void
on_guest_modes_menuitem_activate(
    GtkMenuItem     *menuitem,
    gpointer         user_data )
{
  Guest_Modes_Menu_Item_Activate( menuitem );
}

//----------------------------------------------------------------------

  void
on_device_combobox_changed(
    GtkComboBox     *combobox,
    gpointer         user_data )
{
  Device_Combobox_Changed( combobox );
}

//----------------------------------------------------------------------

  gboolean
on_pointer_cross(
    GtkWidget      *widget,
    const GdkEventButton *event,
    gpointer        data)
{
  Pointer_Cross_Event( event, data );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_rx_zero_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Transceiver[Indices.TRx_Index]->rx_zero_right = True;
  else
    Transceiver[Indices.TRx_Index]->rx_zero_right = False;
}

//----------------------------------------------------------------------

  void
on_tx_zero_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Transceiver[Indices.TRx_Index]->tx_zero_right = True;
  else
    Transceiver[Indices.TRx_Index]->tx_zero_right = False;
}

//----------------------------------------------------------------------

  void
on_tx_freq_lock_togglebutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  if( gtk_toggle_button_get_active(togglebutton) )
    Transceiver[Indices.TRx_Index]->tx_freq_lock = True;
  else
    Transceiver[Indices.TRx_Index]->tx_freq_lock = False;
}

//----------------------------------------------------------------------

  void
on_hermes2_config_button_clicked(
    GtkButton       *button,
    gpointer         user_data )
{
  Config_Dialog();
}

//----------------------------------------------------------------------

  void
on_hermes2_config_dialog_response(
    GtkDialog       *dialog,
    gint             response_id,
    gpointer         user_data )
{
  if( response_id == GTK_RESPONSE_APPLY )
  {
    Flag[HERMES2_RCCONFIG_SETUP] = False;
    Create_Hermes2_Config();
  }

  gtk_widget_destroy( hermes2_gui.config_dialog );
}

//----------------------------------------------------------------------

  gboolean
on_hermes2_config_dialog_delete(
    GtkWidget       *widget,
    GdkEvent        *event,
    gpointer         user_data )
{
  gtk_widget_destroy( hermes2_gui.config_dialog );
  return( TRUE );
}

//----------------------------------------------------------------------

  void
on_hermes2_config_dialog_destroy(
    GObject       *object,
    gpointer      user_data )
{
  hermes2_gui.config_dialog = NULL;
  g_object_unref( hermes2_gui.config_dialog_builder );
  hermes2_gui.config_dialog_builder = NULL;
}

//----------------------------------------------------------------------

  void
on_mute_on_tx_checkbutton_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  if( gtk_toggle_button_get_active(togglebutton) )
    TRx->mute_receiver = True;
  else
    TRx->mute_receiver = False;
}

//----------------------------------------------------------------------

  void
on_guest_save_dialog_destroy(
    GObject  *object,
    gpointer  user_data)
{
  hermes2_gui.save_dialog = NULL;
}

//------------------------------------------------------------------------

  void
on_guest_cancel_save_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  gtk_widget_destroy( hermes2_gui.save_dialog );
}

//------------------------------------------------------------------------

  void
on_guest_save_ok_clicked(
    GtkButton *button,
    gpointer   user_data)
{
  Save_QSO_Record( &qso_record );
  gtk_widget_destroy( hermes2_gui.save_dialog );
}

//------------------------------------------------------------------------

  void
on_tx_rsid_enable_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  if( gtk_toggle_button_get_active(togglebutton) )
  {
    TRx->tx_rsid_enable = True;
    RsId_Init();
  }
  else
  {
    // Free RSID objects if RxId not enabled
    TRx->tx_rsid_enable = False;
    RsId.Free();
  }
}

//----------------------------------------------------------------------

  void
on_rx_rsid_enable_toggled(
    GtkToggleButton *togglebutton,
    gpointer         user_data )
{
  gboolean active = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton) );
  RxID_Checkbutton_Toggled( active );
}

//----------------------------------------------------------------------

  void
on_rsid_apply_button_clicked(
    GtkButton   *button,
    gpointer     user_data )
{
  if( Transceiver[Indices.TRx_Index]->rx_rsid_enable )
    RsId.Setup_TRx();
}

//----------------------------------------------------------------------

  gboolean
on_rsid_apply_button_press_event(
    GtkWidget *widget,
    const GdkEventButton  *event,
    gpointer   user_data )
{
  if( event->button == 3 ) Rsid_Modes_Menu_Button_Clicked();
  return(FALSE);
}

//----------------------------------------------------------------------

  void
on_rsid_select_activate(
    GtkMenuItem *menuitem,
    gpointer     user_data )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  Strlcpy( TRx->RSID_select, gtk_menu_item_get_label(menuitem), sizeof(TRx->RSID_select) );

  // Display RSID listening status
  g_idle_add( RSID_Status, NULL );
}

//----------------------------------------------------------------------

