/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "callback_func.h"
#include "bookmarks.h"
#include "demodulate.h"
#include "display.h"
#include "interface.h"
#include "modulate.h"
#include "sound.h"
#include "spectrum.h"
#include "process.h"
#include "../common/common.h"
#include "../common/filters.h"
#include "../common/cfft.h"
#include "../common/hermes2_rc.h"
#include "../common/shared.h"
#include "../common/utils.h"
#include "../ftx/constants.h"
#include "../ftx/decode_ftx.h"
#include "../ftx/interface.h"
#include "../ftx/shared.h"
#include "../hpsdr/discovery.h"
#include "../hpsdr/hw_to_pc.h"
#include "../hpsdr/hpsdr_init.h"
#include "../hpsdr/pc_to_hw.h"
#include "../hpsdr/settings.h"
#include "../morse/interface.h"
#include "../morse/shared.h"
#include "../psk31/interface.h"
#include "../psk31/shared.h"
#include "../rtty/interface.h"
#include "../rtty/shared.h"
#include "../rsid/rsid.h"
#include "../sstv/interface.h"
#include "../sstv/shared.h"
#include "../olivia/interface.h"
#include "../olivia/operation.h"
#include "../olivia/shared.h"
#include "../hell/interface.h"
#include "../hell/shared.h"
#include "../time/interface.h"
#include "../time/time_signal.h"
#include "../wefax/shared.h"
#include "../wefax/interface.h"
#include "../wsprd/interface.h"
#include "../wsprd/shared.h"
#include <gtk/gtk.h>
#include <semaphore.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>

//----------------------------------------------------------------------

#define RX_MODES_MENU_IDS \
  "rx_modes_menu", \
  NULL

//----------------------------------------------------------------------

#define RSID_MODES_MENU_IDS \
  "rsid_modes_menu", \
  NULL

//----------------------------------------------------------------------

#define TX_MODES_MENU_IDS \
  "tx_modes_menu", \
  NULL

//----------------------------------------------------------------------

#define HERMES2_STATIONS_WINDOW_IDS \
  "hermes2_bmk_window", \
  "hermes2_bmk_treeview", \
  NULL

//----------------------------------------------------------------------

#define QSO_MODES_MENU_IDS \
  "guest_modes_menu", \
  NULL

//----------------------------------------------------------------------

// Guest QSO Modes menu items
#define QSO_MODE_MENU_ITEMS \
  _("FTX"), \
  _("HELL"), \
  _("OLIVIA"), \
  _("PSK31"), \
  _("RTTY"), \
  _("SSTV"), \
  _("MORSE"), \
  _("WEFAX"), \
  _("TIME"), \
  _("WSPR")

//----------------------------------------------------------------------

// Guest QSO Modes menu items
enum QSO_MODES
{
  FTX,
  HELLSCHREIBER,
  OLIVIA,
  PSK31,
  RTTY,
  SSTV,
  MORSE_DECODER,
  WEFAX_DECODER,
  TIME_STATIONS,
  WSPR,
  QSO_MODE_ITEMS
};

//----------------------------------------------------------------------

// Tx frequency deviation for FM modes
#define NBFM1_DEVIATION     5000
#define NBFM2_DEVIATION     2500

//----------------------------------------------------------------------

// For the Wideband spectrum display
#define WIDEBAND_CENTER_FREQ    19200000
#define WIDEBAND_BANDWIDTH      38400000

//----i------------------------------------------------------------------

// Tx Microphone LPF Cut-off
#define NBFM1_MIC_LPF   3800
#define NBFM2_MIC_LPF   3400
#define AMW_MIC_LPF     3800
#define AMN_MIC_LPF     3400
#define SSBW_MIC_LPF    3400
#define SSBM_MIC_LPF    3200
#define SSBN_MIC_LPF    3000

//----------------------------------------------------------------------

// Radio Time transmitters frequencies
#define MSF60_FREQUENCY     60000
#define RBU66_FREQUENCY     66667
#define DCF77_FREQUENCY     77500
#define ALL162_FREQUENCY    162000

//----------------------------------------------------------------------

/* Scale values needed in demodulators, to bring
 * sample values sent to sound card in the range
 * of a 'short int' data type */
#define NBFM1_RX_SCALE   40.0
#define NBFM2_RX_SCALE   80.0
#define AM_RX_SCALE      4.0
#define SSB_RX_SCALE     2.34
#define CW_RX_SCALE      2.34

//----------------------------------------------------------------------

// HERMES2 DSP sampling rates
enum SAMPLE_RATE
{
  HERMES2_RATE_48 = 0,
  HERMES2_RATE_96,
  HERMES2_RATE_192,
  HERMES2_RATE_348
};

//----------------------------------------------------------------------

// Scale factors to avoid overflow in Complex FFT summations
#define SPECTRUM_FFT_SCALE   64.0
#define WIDEBAND_FFT_SCALE   32.0

// IFFT direction
#define FFT_FORWARD    1
#define FFT_REVERSE    0

//----------------------------------------------------------------------

// Minimum desired FFT/Spectrum bandwidth - kHz
#define FFT_MIN_BANDWIDTH   12

//----------------------------------------------------------------------

/* StartRx_Button_Toggled()
 *
 * Handles on_startrx_togglebutton_toggled() C/B
 */
  void
StartRx_Button_Toggled( gboolean active )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  TRx->Demodulator   = TRx->New_Demodulator;


  // Start Receiver
  if( active )
  {
    // Create a thread for PC Sound playback
    if( hermes2_rc.sound_pc_local && !Flag[HERMES2_PCSOUND_SETUP] )
    {
      if( Open_PCSound() )
      {
        // Create a thread to play back demodulation buffer
        if( !Pthread_Create(
              &TRx->sound_thread, NULL, Sound_Write_Async, NULL,
              _("Failed to create Sound Playback thread")) )
          return;
      } // if( Open_PCSound() )

      else return;
    } // if( hermes2_rc.sound_pc_local )

    // (Re)-initialize ADAGC parameters
    TRx->adagc_scale = 0.01;

    // Start DDC Receive stream if not already started
    if( !DDC_Start(TRx, START_RADIO_STREAM) ) return;
    TRx->receive_active = True;

    // (Re)set the AFC check button
    if( (TRx->rx_modulation_mode == RX_MODE_SAMUW) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMUM) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMUN) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLW) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLM) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLN) )
    {
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->afc_checkbutton), TRUE );
    }

    /* Get the Transceiver window's title and add
     * TRX/ADC number to Bookmarks window title */
    if( hermes2_gui.bmk_window )
    {
      const gchar *title = gtk_window_get_title( GTK_WINDOW(TRx->tcvr_window) );
      title = strstr( title, "TRX" );
      gchar str[32];
      snprintf( str, sizeof(str), _("Hermes2 %s Bookmarks"), title );
      gtk_window_set_title( GTK_WINDOW(hermes2_gui.bmk_window), str );
      Indices.Bookmarks_TRx_Idx = (int8_t)Indices.TRx_Index;
    }

  } // if( active )
  else if( !Flag[GUEST_RECEIVING] && !Flag[GUEST_TRANSMITTING] )
  {
    // Unset the AFC check button
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->afc_checkbutton), FALSE );

    // Stop the Radio stream
    TRx->receive_active = False;
    Device_Start_Stop( STOP_RADIO_STREAM );

    // Disable local PC sound output
    Close_PCSound();

  } // if( gtk_toggle_button_get_active() ...

} // StartRx_Button_Toggled()

//----------------------------------------------------------------------

/* Spectrum_Button_Toggled()
 *
 * Handles the on_receiver_spectrum_radiobutton_toggled CB
 */
  void
Spectrum_Button_Toggled( gboolean active )
{
  if( !Flag[HERMES2_RCCONFIG_SETUP] ) return;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];


  if( !active && TRx->spectrum_data.receiver_spectrum )
  {
    // Stop the Spectrum / Radio stream if no other TRx is active
    TRx->spectrum_data.spectrum_active   = False;
    TRx->spectrum_data.receiver_spectrum = False;

    // Wait for spectrum thread if running
    if( TRx->spectrum_data.spectrum_running )
      pthread_join( TRx->spectrum_data.spectrum_thread, NULL );

    if( !Device_Start_Stop(STOP_SPEC_STREAM) )
    {
      ddv->rcve_thread_run = False;
      Close_Data_IO_Socket();
      Error_Dialog( _("ADC_Wideband(): Device_Start_Stop() failed"), SHOW_OK );
    }

    return;
  } // if( !active )

  // Abort if Wideband Spectrum Monitor is open
  if( TRx->spectrum_data.wideband_spectrum )
    return;

  // Wait for spectrum thread if running
  if( TRx->spectrum_data.spectrum_running )
    pthread_join( TRx->spectrum_data.spectrum_thread, NULL );

  // Initialize FFT and spectrum displays
  // Scale factor to keep FFT in range
  TRx->spectrum_data.fft_scale = SPECTRUM_FFT_SCALE;
  Initialize_FFT( &(TRx->spectrum_data), SPECTRUM_WIDTH, SPECTRUM_WIDTH, FFT_FORWARD );

  // Create Spectrum Monitor if null
  if( TRx->spectrum_data.spectrum_drawingarea == NULL )
    TRx->spectrum_data.spectrum_drawingarea =
      Builder_Get_Object( TRx->builder, "hermes2_spectrum_drawingarea" );
  Queue_Draw( TRx->spectrum_data.spectrum_drawingarea );

  // Create a thread to do FFT's and display signal spectrum
  TRx->spectrum_data.spectrum_active   = True;
  TRx->spectrum_data.receiver_spectrum = True;
  TRx->spectrum_data.wideband_spectrum = False;
  TRx->spectrum_data.trx_index = TRx->index;

  if( !Pthread_Create(
      &(TRx->spectrum_data.spectrum_thread), NULL,
      Paint_Waterfall, (void *)&(TRx->spectrum_data),
      _("Failed to create Waterfall Paint thread")) )
  {
    TRx->spectrum_data.spectrum_active   = False;
    TRx->spectrum_data.receiver_spectrum = False;
    return;
  }

  // Start DDC Receive stream
  if( !DDC_Start(TRx, START_SPEC_STREAM) )
  {
    TRx->spectrum_data.spectrum_active   = False;
    TRx->spectrum_data.receiver_spectrum = False;
    return;
  }

} // Spectrum_Button_Toggled()

//----------------------------------------------------------------------

/* ADC_Wideband_Toggled()
 *
 * Handles the wideband_togglebutton_toggled() callback
 */
  void
ADC_Wideband_Toggled( gboolean active )
{
  if( !Flag[HERMES2_RCCONFIG_SETUP] ) return;

  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  Transceiver_t *TRx       = Transceiver[Indices.TRx_Index];
  static uint32_t fft_bw;

  if( !active && TRx->spectrum_data.wideband_spectrum )
  {
    // Restore original data
    TRx->spectrum_data.center_freq   = TRx->rx_frequency + (uint32_t)TRx->rx_weaver_offset;
    TRx->spectrum_data.fft_bandwidth = fft_bw;

    if( TRx->spectrum_data.fft_bandwidth > hermes2_rc.ddc_sample_rate )
      TRx->spectrum_data.fft_bandwidth = hermes2_rc.ddc_sample_rate;

    /* Restore original Filter settings.
     * Prepare C&C buffer for sending Filter settings to HL2 */
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][2]  = ddv->filter_settings;
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][2]  = ddv->filter_settings;

    // Stop the wideband data transfer
    TRx->spectrum_data.spectrum_active   = False;
    TRx->spectrum_data.wideband_spectrum = False;

    // Wait for spectrum thread if running
    if( TRx->spectrum_data.spectrum_running )
      pthread_join( TRx->spectrum_data.spectrum_thread, NULL );

    if( !Device_Start_Stop(STOP_BANDSCOPE) )
    {
      ddv->rcve_thread_run = False;
      Close_Data_IO_Socket();
      Error_Dialog( _("ADC_Wideband(): Device_Start_Stop() failed"), SHOW_OK );
    }

    return;
  } // if( !active )

  // Wait for Spectrum thread to finish
  if( TRx->spectrum_data.spectrum_running )
    pthread_join( TRx->spectrum_data.spectrum_thread, NULL );

  // Save and initialize Spectrum data
  fft_bw = TRx->spectrum_data.fft_bandwidth;
  TRx->spectrum_data.fft_scale     = WIDEBAND_FFT_SCALE;
  TRx->spectrum_data.center_freq   = WIDEBAND_CENTER_FREQ;
  TRx->spectrum_data.fft_bandwidth = WIDEBAND_BANDWIDTH;
  Initialize_FFT( &(TRx->spectrum_data), SPECTRUM_WIDTH, REAL_FFT_INPUT_LENGTH, FFT_FORWARD );

  // Create Spectrum Monitor if null
  if( TRx->spectrum_data.spectrum_drawingarea == NULL )
    TRx->spectrum_data.spectrum_drawingarea =
      Builder_Get_Object( TRx->builder, "hermes2_spectrum_drawingarea" );
  Queue_Draw( TRx->spectrum_data.spectrum_drawingarea );

  // Create a thread to do FFT's and display signal spectrum
  TRx->spectrum_data.spectrum_active   = True;
  TRx->spectrum_data.wideband_spectrum = True;
  TRx->spectrum_data.receiver_spectrum = False;
  TRx->spectrum_data.trx_index         = TRx->index;

  // Initialize SDR device if not already
  if( !Hpsdr_Initialize() ) return;

  // Create a thread to do FFT's and display signal spectrum
  if( !Pthread_Create(
      &(TRx->spectrum_data.spectrum_thread), NULL,
      Paint_Waterfall, (void *)&(TRx->spectrum_data),
      _("Failed to create Waterfall Paint thread")) )
  {
    TRx->spectrum_data.spectrum_active   = False;
    TRx->spectrum_data.wideband_spectrum = False;
    return;
  }

  // Activate Bandscope stream
  if( !Device_Start_Stop(START_BANDSCOPE) ) return;

  // Prepare C&C buffer for sending Filter settings to HL2
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][2] = 0; // Disable all filters
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][2] = 0; // Disable all filters
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;


} // ADC_Wideband_Toggled()

//----------------------------------------------------------------------

/* PA_Enable_Button_Toggled()
 *
 * Handles on_tx_pa_enable_toggled() C/B
 */
  void
PA_Enable_Button_Toggled( gboolean active )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Set C&C C0 byte for On-Board PA On / Off (addr=0x09, C2[3])
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x09;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  // Enable or disable Power Amplifier
  if( active )
  {
    // Set the Transmitter frequency
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );

    // Enable Power Amplifier
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x09][2] |= BIT_3; // Set C2[3]
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x09][2] |= BIT_3; // Set C2[3]
    ddv->tx_pa_enable = True;
  } // if( gtk_toggle_button_get_active() ...
  else
  {
    // Disable Power Amplifier
    Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x09][2] &= ~BIT_3; // Clear C2[3]
    Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x09][2] &= ~BIT_3; // Clear C2[3]
    ddv->tx_pa_enable = False;
    ddv->transmit_on  = False;

    // Disable MOX and TUNE button
    GtkWidget *button;
    button = Builder_Get_Object(TRx->builder, "mox_togglebutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), FALSE );
    button = Builder_Get_Object(TRx->builder, "tune_togglebutton" );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), FALSE );
  } // if( gtk_toggle_button_get_active() ...

} // PA_Enable_Button_Toggled()

//----------------------------------------------------------------------

/* Tune_Button_Toggled()
 *
 * Handles the on_tune_togglebutton_toggled() callback
 */
  void
Tune_Button_Toggled( gboolean active )
{
  // Save current settings
  static void ( *modulator )( void *data );
  static double power;
  discovered_device_t *ddv = &Device[hermes2_rc.device_index];

  // Save Modulator and CW mode and then disable both
  if( active )
  {
    // Fill the USB frame I/Q buffer with a "DC" signal of max amplitude
    uint16_t buf_idx1 = 20, buf_idx2 = 532;
    for( uint8_t idx = 0; idx < CCD_BUFFER_IQ_PAIRS; idx++ )
    {
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx1++] = 0x4F;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx1++] = 0xFF;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx1++] = 0x4F;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx1++] = 0xFF;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx2++] = 0x4F;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx2++] = 0xFF;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx2++] = 0x4F;
      Cmnd_Ctrl_Audio_IQ_Data.buffer[buf_idx2++] = 0xFF;
      buf_idx1 += 4;
      buf_idx2 += 4;
    }

    modulator = Modulator;
    Modulator = NULL;
    power     = hermes2_rc.tx_power_limit;
    hermes2_rc.tx_power_limit = hermes2_rc.tx_tune_power;

    // Enable MOX to transmit Tune power
    ddv->tx_pa_tune = True;
    MOX_Control( MOX_ON );
  }
  else // Restore Modulator and CW Mode
  {
    // Clear the USB frame IQ buffer and DUC
    Clear_DUC();

    // Restore Settings
    hermes2_rc.tx_power_limit = power;
    ddv->tx_pa_tune = False;
    MOX_Control( MOX_OFF );
    Modulator = modulator;
  }

} // Tune_Button_Toggled()

//----------------------------------------------------------------------

/* MOX_Button_Toggled()
 *
 * Handles the on_mox_togglebutton_toggled() callback
 */
  void
MOX_Button_Toggled( gboolean active )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  if( TRx == NULL ) return;

  discovered_device_t *ddv = &Device[hermes2_rc.device_index];
  const GtkWidget *image   = Builder_Get_Object( TRx->builder, "tx_on_image" );
  static double lna_gain   = 0.0;
  static uint8_t filter_settings = 0;

  if( active && TRx->receive_active )
  {
    // Abort if Tx is disabled
    if( !ddv->tx_pa_enable ) return;

    // Mute Receiver if enabled
    if( TRx->mute_receiver )
      TRx->receiver_muted = True;
    else
      TRx->receiver_muted = False;

    // Allocate ring buffer
    audio_ring_buff.buff_size = 3 * hermes2_rc.ddc_buffer_length;
    size_t mreq = 3 * (size_t)hermes2_rc.ddc_buffer_length * sizeof( int16_t );
    memset( audio_ring_buff.buff_left,  0, mreq );
    memset( audio_ring_buff.buff_right, 0, mreq );

    // Unset LNA Gain Auto check button
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(hermes2_gui.lna_gain_auto), FALSE );

    // Save LNA gain and Set value to minimum
    lna_gain = gtk_spin_button_get_value( GTK_SPIN_BUTTON(hermes2_gui.lna_gain_spinbutton) );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(hermes2_gui.lna_gain_spinbutton), -12.0 );

    // Enable Hardware Managed LNA Gain
    if( ddv->lna_gain_hw )
    {
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0E][3]  = BIT_7 | BIT_6;  // Set LNA Gain to -12db
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0E][3]  = BIT_7 | BIT_6;  // Set LNA Gain to -12db
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0E;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0E;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
    }

    // (Re)set TX frequency
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );

    // Bypass 3.0MHz HPF on transmit. (Apparently done by hardware?)
    filter_settings       = ddv->filter_settings;
    ddv->filter_settings &= ~HP_FILTER_3MHZ;

    // If Speech modes (SSB, AM, FM), enable Sound (Mic) Capture
    if( TRx->tx_modulation_mode <= TX_MODE_NBFM2 )
    {
      // Open Sound Capture (Mic), abort on error
      if( !Open_Capture() )
      {
        Error_Dialog( _("Failed to open ALSA Sound Capture device"), HIDE_OK );
        return;
      }
    }

    // Enable Transmit mode (MOX On)
    ddv->transmit_on = True;
    ddv->pa_power_up = RESET;
    gtk_image_set_from_icon_name( GTK_IMAGE(image), "gtk-no", GTK_ICON_SIZE_BUTTON );

    // Wait for the RF delay (Wait for relays etc)
    usleep( hermes2_rc.rf_delay * 1000 );

    // Send RSID tones if enabled
    if( TRx->tx_rsid_enable  &&
        !Flag[RSIDTX_ACTIVE] &&
        (TRx->RSID_Mode != MODE_NULL) )
    {
      // Initialize RsId if needed
      RsId_Init();
      Flag[RSIDTX_ACTIVE] = True;
      RsId.Send( TRx->RSID_Mode );
    }

  } // if( active) )
  else if( ddv->transmit_on )
  {
    // Disable PTT and TX indicator
    ddv->transmit_on = False;

    // Clear Tx On icon
    gtk_image_set_from_icon_name( GTK_IMAGE(image), "gtk-close", GTK_ICON_SIZE_BUTTON );

    // Disable Sound (Mic) Capture, if enabled
    if( Flag[HERMES2_CAPTURE_SETUP] ) Close_Capture();
    if( Flag[HERMES2_MIXER_SETUP] )   Close_Mixer();

    // Un-mute Receiver
    TRx->receiver_muted = False;

    // Set ADC0 Attenuator Auto check button
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(hermes2_gui.lna_gain_auto), TRUE );

    // Disable Hardware Managed LNA Gain
    if( ddv->lna_gain_hw )
    {
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0E][3] = BIT_6;    // Disable Hardware LNA Gain
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0E][3] = BIT_6;    // Disable Hardware LNA Gain
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0E;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0E;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
      Send_Cmnd_Ctrl_Audio_IQ_Data( "MOX_Button_Toggled" );
    }

    // Reset LNA gain value
    usleep( 50000 );
    gtk_spin_button_set_value( GTK_SPIN_BUTTON(hermes2_gui.lna_gain_spinbutton), lna_gain );

    // Return original fliter settings
    ddv->filter_settings = filter_settings;
    Clear_DUC();

  } // else of if( active )

  // Prepare C&C buffer for sending Filter settings to HL2
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x00][2] = ddv->filter_settings;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x00][2] = ddv->filter_settings;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x00;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  // Clear Receive buffers
  if( demod_iq_buf_siz )
  {
    memset( TRx->demod_id, 0, demod_iq_buf_siz );
    memset( TRx->demod_qd, 0, demod_iq_buf_siz );
  }

} // MOX_Button_Toggled()

//----------------------------------------------------------------------

/* MOX_Control()
 *
 * Controls DUC samples transmission and MOX (Manual On Xmitter)
 */
  void
MOX_Control( BOOLEAN active )
{
  static BOOLEAN mox_on = False;

  if( active ) // Turn MOX On
  {
    if( !mox_on ) // If not already On, then turn MOX On
    {
      mox_on = True;
      MOX_Button_Toggled( True );
    }
  }
  else if( mox_on ) // If MOX ON, then turn MOX Off
  {
    Flag[HERMES2_SEND_DUC_PACKET] = False;
    Modulator = NULL;
    mox_on = False;
    MOX_Button_Toggled( False );
  }

} // MOX_Control()

//------------------------------------------------------------------------

/* Quit_Dialog_Response()
 *
 * Handles the user's response to the Quit dialog
 */
  void
Quit_Dialog_Response( GtkDialog *dialog, gint response_id )
{
  // Yes, quit Hermes2
  if( response_id == GTK_RESPONSE_CLOSE )
  {
    // Save current Hermes2 config state
    Get_GUI_Geometry();
    Save_Hermes2_Config( FROM_HERMES );

    /* Stop all activity and wait for threads to terminate before
     * destroying Transceiver objects and freeing resources */
    Guest_Quit_Activate( hermes2_gui.guest_window );
    Flag[HERMES2_QUIT] = True;
    Hermes2_Stop();

    // Destroy Transceiver Windows
    for( uint8_t idx = 0; idx < Indices.Num_of_TRXs; idx++ )
    {
      if( Transceiver[idx]->tcvr_window != NULL )
        gtk_widget_destroy( Transceiver[idx]->tcvr_window );
      while( g_main_context_iteration(NULL, FALSE) );
    }

    // Destroy Main Hermes2 Window
    if( hermes2_gui.window != NULL )
      gtk_widget_destroy( hermes2_gui.window );
    else gtk_main_quit();

  } // if( response_id == GTK_RESPONSE_CLOSE )
  else if( response_id == GTK_RESPONSE_CANCEL )
  {
    gtk_widget_destroy( hermes2_gui.quit_dialog );
    Flag[GUEST_QUIT] = False;
  }

} // Quit_Dialog_Response()

//------------------------------------------------------------------------

/* Time_Receive_Togglebutton_Toggled()
 *
 * Handles the on_time_receive_togglebutton_toggled() CB
 */
  gboolean
Time_Receive_Togglebutton_Toggled( gpointer togglebutton )
{
  Transceiver_t *TRx = Transceiver[Indices.Time_TRx_Idx];

  if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(togglebutton)) )
  {
    /* Normally it would be GUEST_RECEIVING
     * true but the time decoder doesn't use the
     * demodulator semaphore for data transfer */
    //Flag[GUEST_TRANSMITTING] = True;

    // Start DDC Receive stream
    if( !DDC_Start(TRx, START_RADIO_STREAM) )
      return( FALSE );
    TRx->timedec_active = True;
  } // if( gtk_toggle_button_get_active() ...
  else
  {
    /* Normally it would be GUEST_RECEIVING
     * false but the time decoder doesn't use the
     * demodulator semaphore for data transfer */
    //Flag[GUEST_TRANSMITTING] = False;

    // Stop hermes2 and cleanup. Stop the Radio stream
    Device_Start_Stop( STOP_RADIO_STREAM );
  } // if( gtk_toggle_button_get_active() ...

  return( FALSE );
} // Time_Receive_Togglebutton_Toggled()

//----------------------------------------------------------------------

/* Afc_Checkbutton_Toggled()
 *
 * Handles the on_afc_checkbutton_toggled() callback
 */
  void
Afc_Checkbutton_Toggled( uint8_t idx, gboolean active )
{
  Transceiver_t *TRx = Transceiver[idx];

  if( active )
  {
    // Initialize AFC semaphore
    if( !Init_Semaphore( &(TRx->afc_semaphore), True) )
      return;
    TRx->afc_enable_status = True;

    // Create a thread to do the AFC
    if( !Pthread_Create(
        &(TRx->afc_thread), NULL, Correct_Frequency_Offset, (void *)TRx,
        _("Failed to create AFC thread")) )
    {
      TRx->afc_enable_status = False;
      return;
    }
  }
  else
  {
    /* Post to semaphore to unblock the
     * thread to exit and free semaphore */
    TRx->afc_enable_status = False;
    int sval;
    sem_getvalue( &(TRx->afc_semaphore), &sval );
    if( !sval ) sem_post( &(TRx->afc_semaphore) );
  }

} // Afc_Checkbutton_Toggled()

//----------------------------------------------------------------------

/* Device_Combobox_Changed()
 *
 * Handles the on_device_combobox_changed() CB
 */
  void
Device_Combobox_Changed( GtkComboBox *combobox )
{
  gchar *txt = NULL;
  uint8_t idx;
  const discovered_device_t *ddv = &Device[hermes2_rc.device_index];


  // Stop hermes2 if running
  if(ddv->rcve_thread_run )
    for( idx = 0; idx < Indices.Num_of_TRXs; idx++ )
    {
      Hermes2_Stop();
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
            Transceiver[idx]->startrx_togglebutton), FALSE );
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(
            Transceiver[idx]->spectrum_off_radiobutton), TRUE );
    }

  // Get HPSDR device type
  txt = gtk_combo_box_text_get_active_text( GTK_COMBO_BOX_TEXT(combobox) );
  if( txt != NULL )
  {
    for( idx = 0; idx < MAX_DEVICES; idx++ )
    {
      if( strcmp(Device[idx].device_name, txt) == 0 )
      {
        hermes2_rc.device_index = idx;
        break;
      }
    }
  }
  g_free( txt );

  // Append new sample rates
  Append_Sample_Rates( hermes2_gui.sample_rate_combobox );

  for( idx = 0; idx < Indices.Num_of_TRXs; idx++ )
  {
    // Redo modulation mode setup
    Rx_Modulation_Mode_Changed( idx, Transceiver[idx]->rx_modulation_mode );
  }

} // Device_Combobox_Changed()

//----------------------------------------------------------------------

/* Rx_Bands_Combobox_Changed()
 *
 * Handles the bands_combobox_changed() CB
 */
  void
Rx_Bands_Combobox_Changed( GtkComboBox *combobox )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get the index of the active combobox entry
  gint idx = gtk_combo_box_get_active( combobox );

  // Set DDC center frequency and mode
  pointer_cross_label = NULL;
  TRx->rx_frequency = hermes2_rc.rx_bands[idx];
  TRx->rx_bands_idx = idx;

  // Remain in Guest Modulation mode if it is specified
  if( TRx->guest_modulation_mode < RX_MODE_ITEMS )
    TRx->rx_modulation_mode = TRx->guest_modulation_mode;
  else
    TRx->rx_modulation_mode = hermes2_rc.rx_modes[idx];

  // Match Tx Band if linked
  if( !TRx->tx_freq_lock && TRx->link_tx_rx_freq )
  {
    GtkWidget *combo = Builder_Get_Object( TRx->builder, "tx_bands_combobox" );
    gtk_combo_box_set_active( GTK_COMBO_BOX(combo), idx );
  }

  // Set Center frequency and Mode
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );
  Rx_Modulation_Mode_Changed( Indices.TRx_Index, TRx->rx_modulation_mode );

} // Rx_Bands_Combobox_Changed()

//----------------------------------------------------------------------

/* Tx_Bands_Combobox_Changed()
 *
 * Handles the bands_combobox_changed() CB
 */
  void
Tx_Bands_Combobox_Changed( GtkComboBox *combobox )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get the index of the active combobox entry
  gint idx = gtk_combo_box_get_active( combobox );

  // Set DDC center frequency and mode
  pointer_cross_label = NULL;

  TRx->tx_frequency = hermes2_rc.tx_bands[idx];
  TRx->tx_bands_idx = idx;

  // Match Rx Mode and Frequency if linked
  if( !TRx->tx_freq_lock && TRx->link_tx_rx_freq )
  {
    TRx->tx_modulation_mode = TRx->rx_modulation_mode;
    if( TRx->tx_modulation_mode >= TX_MODE_ITEMS )
      TRx->tx_modulation_mode = TX_MODE_ITEMS - 1;

    /* This is to match Tx frequency to Rx frequency
     * if saved Band frequency is different */
    TRx->tx_frequency = TRx->rx_frequency;
  }
  else
  {
    // Use per Band saved Mode
    if( !(TRx->guest_modulation_mode < RX_MODE_ITEMS) )
      TRx->tx_modulation_mode = hermes2_rc.tx_modes[idx];
    else
    {
      // Use Guest Mode
      TRx->tx_modulation_mode = TRx->guest_modulation_mode;
      if( TRx->tx_modulation_mode >= TX_MODE_ITEMS )
        TRx->tx_modulation_mode = TX_MODE_ITEMS - 1;
    }
  }

  // Set Center frequency and Mode
  Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
  Tx_Modulation_Mode_Changed( Indices.TRx_Index, TRx->tx_modulation_mode );

} // Tx_Bands_Combobox_Changed()

//----------------------------------------------------------------------

/* Time_Station_Combobox_Changed()
 *
 * Handles the on_time_station_combobox_changed() CB
 */
  void
Time_Station_Combobox_Changed( GtkComboBox *combobox )
{
  uint8_t modulation_mode;
  uint32_t center_freq, station_idx;
  Transceiver_t *TRx = Transceiver[Indices.Time_TRx_Idx];

  // Get the selected station
  Flag[TIME_NEW_STATION] = True;
  station_idx = (uint32_t)gtk_combo_box_get_active( combobox );
  switch( station_idx )
  {
    case MSF60:
      Receive_Time_Station = Receive_MSF60;
      center_freq = MSF60_FREQUENCY;
      modulation_mode  = RX_MODE_CWUN;
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "dcf77_frame") );
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "rbu66_frame") );
      gtk_widget_show( Builder_Get_Object(time_window_builder, "msf60_frame") );
      gtk_window_set_title( GTK_WINDOW(time_window), "MSF 60kHz" );
      break;

    case RBU66:
      Receive_Time_Station = Receive_RBU66;
      center_freq = RBU66_FREQUENCY;
      modulation_mode  = RX_MODE_CWUW;
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "dcf77_frame") );
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "msf60_frame") );
      gtk_widget_show( Builder_Get_Object(time_window_builder, "rbu66_frame") );
      gtk_window_set_title( GTK_WINDOW(time_window), "RBU 66.667kHz" );
      break;

    case DCF77:
      Receive_Time_Station = Receive_DCF77;
      center_freq = DCF77_FREQUENCY;
      modulation_mode  = RX_MODE_CWUN;
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "rbu66_frame") );
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "msf60_frame") );
      gtk_widget_show( Builder_Get_Object(time_window_builder, "dcf77_frame") );
      gtk_label_set_text(
          GTK_LABEL(Builder_Get_Object( time_window_builder, "dcf77_label")), "DCF77" );
      gtk_window_set_title( GTK_WINDOW(time_window), "DCF77 77.5kHz" );
      break;

    case ALL162:
      Receive_Time_Station = Receive_ALL162;
      center_freq = ALL162_FREQUENCY;
      modulation_mode  = RX_MODE_CWUN;
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "rbu66_frame") );
      gtk_widget_hide( Builder_Get_Object(time_window_builder, "msf60_frame") );
      gtk_widget_show( Builder_Get_Object(time_window_builder, "dcf77_frame") );
      gtk_label_set_text(
          GTK_LABEL(Builder_Get_Object( time_window_builder, "dcf77_label")), "TDF162" );
      gtk_window_set_title( GTK_WINDOW(time_window), "Allouis 162kHz" );
      break;

    default:
      Receive_Time_Station = NULL;
      return;
  }
  gtk_window_resize( GTK_WINDOW(time_window), 10, 10 );

  // Set up center frequency on change
  if( TRx->rx_frequency != center_freq )
  {
    // Set the Rx Band combobox first
    Set_Rx_Band( TRx, center_freq );
    TRx->rx_frequency = center_freq;
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );
  }

  // Set up new mode on change
  if( TRx->rx_modulation_mode != modulation_mode )
    Rx_Modulation_Mode_Changed( Indices.Time_TRx_Idx, modulation_mode );

} // Time_Station_Combobox_Changed()

//----------------------------------------------------------------------

/* Clear_PCSound_Buf()
 *
 * Clears the sound buffers on parameter changes
 */
  static void
Clear_PCSound_Buf( void )
{
  size_t siz = 2 * SOUND_OP_BUF_SIZE * sizeof( short );

  if( Flag[HERMES2_INITIALIZED] && Flag[HERMES2_PCSOUND_SETUP] )
    for( uint32_t idx = 0; idx < NUM_SOUND_RING_BUFFS; idx++ )
      memset( (void *)sound_ring_buf[idx], 0, siz );

} // Clear_PCSound_Buf()

//----------------------------------------------------------------------

/* FFT_Bandwidth_Changed
 *
 * Handles on_fft_bw_combobox_changed() CB
 */
  void
FFT_Bandwidth_Changed( GtkComboBox *combobox )
{
  gchar *txt;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Abort if wideband spectrum is running
  if( Flag[HERMES2_OPERATION_PENDING] || TRx->spectrum_data.wideband_spectrum )
    return;

  // Get bandwidth in kHz and convert to Hz
  txt = gtk_combo_box_text_get_active_text( GTK_COMBO_BOX_TEXT(combobox) );
  if( txt != NULL )
  {
    // Go to numerals
    uint8_t offset = 0;
    while( txt[offset] != ' ' ) offset++;
    uint32_t bw = (uint32_t)atoi( &txt[offset] );
    bw *= 1000;
    if( bw == 0 ) bw = 1000;
    g_free( txt );

    // Init LPF filter if sample rate is specified
    TRx->spectrum_data.fft_bandwidth = bw;
    if( hermes2_rc.ddc_sample_rate ) Init_Roofing_Filter( TRx );
  }

  // Get the active text
  TRx->fft_bw_idx = gtk_combo_box_get_active( combobox );

} // FFT_Bandwidth_Changed()

//----------------------------------------------------------------------

/* Rx_Weaver_Combobox_Changed()
 *
 * Handles the rx_weaver_combobox_changed callback
 */
  void
Rx_Weaver_Combobox_Changed( GtkComboBox *combobox )
{
  // Available Weaver method phasing frequencies
  const char *weaver_frequency[] = { RX_WEAVER_FREQS };
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get selected Weaver frequency
  gint idx = gtk_combo_box_get_active( combobox );
  if( idx <= 0 ) return;

  // Go to numerals
  uint8_t off = 0;
  size_t len  = strlen( weaver_frequency[idx] );
  while( weaver_frequency[idx][off] != ' ' ) off++;
  off++;

  // If it is in kHz, multiply by 1000
  double freq = atof( &weaver_frequency[idx][off] );
  if( weaver_frequency[idx][len-3] == 'k' ) freq *= 1000.0;

  // Set Weaver offset frequency.
  // For Non-CW modes, Weaver frequency is 1/2 of BFO frequency
  switch( TRx->rx_modulation_mode )
  {
    case RX_MODE_USBW:  case RX_MODE_USBM:  case RX_MODE_USBN:
    case RX_MODE_SAMUW: case RX_MODE_SAMUM: case RX_MODE_SAMUN:
    case RX_MODE_RSID:  case RX_MODE_RFAXU: case RX_MODE_SSTVU:
    case RX_MODE_FTX:
      TRx->rx_weaver_frequency = (uint16_t)freq / 2;
      TRx->rx_weaver_offset    =  (int16_t)freq / 2;
      break;

    case RX_MODE_LSBW:  case RX_MODE_LSBM:  case RX_MODE_LSBN:
    case RX_MODE_SAMLW: case RX_MODE_SAMLM: case RX_MODE_SAMLN:
    case RX_MODE_RFAXL: case RX_MODE_SSTVL:
      TRx->rx_weaver_frequency = (uint16_t)freq / 2;
      TRx->rx_weaver_offset    = -(int16_t)freq / 2;
      break;

    default:
      TRx->rx_weaver_offset = 0;
      TRx->rx_weaver_frequency = (uint16_t)freq;
  }

  // Set center frequency
  Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

} // Rx_Weaver_Combobox_Changed()

//----------------------------------------------------------------------

/* Tx_Weaver_Combobox_Changed()
 *
 * Handles the rx_weaver_combobox_changed callback
 */
  void
Tx_Weaver_Combobox_Changed( GtkComboBox *combobox )
{
  // Available Weaver method phasing frequencies
  const char *weaver_frequency[] = { TX_WEAVER_FREQS };
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Get selected Weaver frequency
  gint idx = gtk_combo_box_get_active( combobox );
  if( idx < 0 ) return;
  size_t len  = strlen( weaver_frequency[idx] );

  // Go to numerals
  uint8_t offset = 0;
  while( weaver_frequency[idx][offset] != ' ' ) offset++;
  offset++;

  // For SSB modes, Weaver frequency is 1/2 of BFO frequency
  double freq = atof( &weaver_frequency[idx][offset] ) / 2.0;

  // If it is in kHz, multiply by 1000
  if( weaver_frequency[idx][len-3] == 'k' ) freq *= 1000.0;
  TRx->tx_weaver_frequency = (uint16_t)freq;

  // Set Weaver offset frequency
  switch( TRx->tx_modulation_mode )
  {
    case TX_MODE_USBW: case TX_MODE_USBM:
    case TX_MODE_USBN: case TX_MODE_FTX:
      TRx->tx_weaver_offset = (int16_t)TRx->tx_weaver_frequency;
      break;

    case TX_MODE_LSBW: case TX_MODE_LSBM: case TX_MODE_LSBN:
      TRx->tx_weaver_offset = -(int16_t)TRx->tx_weaver_frequency;
      break;

    default: TRx->tx_weaver_offset = 0;
  }

  // Set center frequency
  Hermes2_Set_Center_Frequency( TRx, TX_FLAG );

} // Tx_Weaver_Combobox_Changed()

//----------------------------------------------------------------------

/* Set_FFT_Bandwidth_Combobox()
 *
 * Appends entries of available FFT bandwidths to the
 * FFT Bandwidth Combo Box according to sample rate
 */
  static void
Set_FFT_Bandwidth_Combobox( Transceiver_t *TRx )
{
  gchar txt[32];
  uint32_t rate  = FFT_MIN_BANDWIDTH;
  gint bw_idx = TRx->fft_bw_idx;
  int8_t cnt = -1; // Count of combobox entries

  Flag[HERMES2_OPERATION_PENDING] = True;

  // Clear combo box tree model
  GtkTreeModel *model = gtk_combo_box_get_model( GTK_COMBO_BOX(TRx->fft_bw_combobox) );
  gtk_list_store_clear( GTK_LIST_STORE(model) );

  /* Enter a sequence of FFT Band Widths starting from the ADC
   * sample rate down to a minimum of rate/FFT_MIN_BANDWIDTH kHz */
  while( rate <= hermes2_rc.ddc_sample_rate / 1000 )
  {
    snprintf( txt, sizeof(txt), "BW %3u kHz", rate );
    gtk_combo_box_text_append_text( GTK_COMBO_BOX_TEXT(TRx->fft_bw_combobox), txt );
    rate *= 2;
    cnt++;
  }

  // Limit range of combobox index
  if( bw_idx > cnt ) bw_idx = cnt;
  TRx->fft_bw_idx = bw_idx;

  // Set same FFT bandwidth if available
  Flag[HERMES2_OPERATION_PENDING] = False;
  gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->fft_bw_combobox), bw_idx );

} // Set_FFT_Bandwidth_Combobox()

//----------------------------------------------------------------------

/* Rx_Bandwidth_Combobox_Changed()
 *
 * Handles the rx_bw_combobox_changed callback
 */
  void
Rx_Bandwidth_Combobox_Changed( GtkComboBox *combobox )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Available demodulator bandwidths
  const char *rx_bw[] = { RX_BAND_WIDTHS };
  gint idx;

  // Get selected demodulator bandwidth
  idx = gtk_combo_box_get_active( combobox );
  if( idx < 0 ) return;

  // Go to numerals
  uint8_t offset = 0;
  while( rx_bw[idx][offset] != ' ' ) offset++;
  double bndw = atof( &rx_bw[idx][offset] );
  size_t len  = strlen( rx_bw[idx] );

  // If it is in kB, multiply by 1000
  if( rx_bw[idx][len-3] == 'k' ) bndw *= 1000.0;
  TRx->rx_bandwidth = (uint16_t)bndw;
  Set_FFT_Bandwidth_Combobox( TRx );

} // Rx_Bandwidth_Combobox_Changed()

//----------------------------------------------------------------------

/* Tx_Bandwidth_Combobox_Changed()
 *
 * Handles the tx_bw_combobox_changed callback
 */
  void
Tx_Bandwidth_Combobox_Changed( GtkComboBox *combobox )
{
  // Available demodulator bandwidths
  const char *tx_bw[] = { TX_BAND_WIDTHS };
  gint idx;

  // Get selected demodulator bandwidth
  idx = gtk_combo_box_get_active( combobox );
  if( idx < 0 ) return;

  // Go to numerals
  uint8_t offset = 0;
  while( tx_bw[idx][offset] != ' ' ) offset++;
  double bndw = atof( &tx_bw[idx][offset] );

  // If it is in kB, multiply by 1000
  size_t len  = strlen( tx_bw[idx] );
  if( tx_bw[idx][len-3] == 'k' ) bndw *= 1000.0;
  Transceiver[Indices.TRx_Index]->tx_bandwidth = (uint16_t)bndw;

} // Tx_Bandwidth_Combobox_Changed()

//----------------------------------------------------------------------

/* Rate_Combobox_Changed()
 *
 * Handles the on_rate_combobox_changed() CB (new sample rate )
 */
  void
Rate_Combobox_Changed( GtkComboBox *combobox )
{
  if( !Flag[HERMES2_INITIALIZED] ) return;

  gchar *txt = NULL;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Clear play back buffer to avoid stutter etc
  if( hermes2_rc.sound_pc_local ) Clear_PCSound_Buf();

  // Get sample rate from combobox
  txt = gtk_combo_box_text_get_active_text( GTK_COMBO_BOX_TEXT(combobox) );
  if( txt != NULL )
  {
    // Go to numerals
    uint8_t offset = 0;
    while( txt[offset] != ' ' ) offset++;
    hermes2_rc.ddc_sample_rate    = (uint32_t)( atoi(&txt[offset]) * 1000 );
    hermes2_rc.ddc_mic_rate_ratio = (uint8_t)( hermes2_rc.ddc_sample_rate / MIC_SAMPLE_RATE );
  }
  g_free( txt );

  // Set the hermes2 ADC sample rate
  Hpsdr_Set_Sample_Rate( TRx );

  // Setup FFT Bandwidth combo box
  Set_FFT_Bandwidth_Combobox( TRx );

  // Save Sample Rate Index
  hermes2_rc.sample_rate_idx = gtk_combo_box_get_active( GTK_COMBO_BOX(combobox) );

  // Set up main filter data and init filter
  if( hermes2_rc.ddc_sample_rate ) Init_Roofing_Filter( TRx );

} // Rate_Combobox_Changed()

//----------------------------------------------------------------------

/* Modes_Menu_Item_Activate()
 *
 * Handles the on_modes_menuitem_activate callback
 */
  void
Modes_Menu_Item_Activate( GtkMenuItem *menuitem, BOOLEAN rx_flag )
{
  uint8_t mode, num_modes;
  gchar **menu_item;
  gchar *rx_items[] = { RX_MODE_MENU_ITEMS };
  gchar *tx_items[] = { TX_MODE_MENU_ITEMS };

  // Make a list of menu items
  if( rx_flag )
  {
    menu_item = rx_items;
    num_modes = RX_MODE_ITEMS;
  }
  else
  {
    menu_item = tx_items;
    num_modes = TX_MODE_ITEMS;
  }

  // Identify the menu item
  const gchar *label = gtk_menu_item_get_label( menuitem );
  for( mode = 0; mode < num_modes; mode++ )
  {
    if( strcmp(label, menu_item[mode]) == 0 )
      break;
  }

  // Set up new modulation mode
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  if( rx_flag && (mode < num_modes) )
  {
    // Save the mode in the Bands modes list
    if( TRx != NULL )
      hermes2_rc.rx_modes[TRx->rx_bands_idx] = mode;
    else
      hermes2_rc.rx_modes[0] = mode;
    Rx_Modulation_Mode_Changed( Indices.TRx_Index, mode );
  }
  else if( !rx_flag && (mode < num_modes) )
  {
    // Save the mode in the Bands modes list
    hermes2_rc.tx_modes[TRx->tx_bands_idx] = mode;
    Tx_Modulation_Mode_Changed( Indices.TRx_Index, mode );
  }

} // Modes_Menu_Item_Activate()

//----------------------------------------------------------------------

/* Guest_Modes_Menu_Item_Activate()
 *
 * Handles the on_guest_modes_menuitem_activate callback
 */
  void
Guest_Modes_Menu_Item_Activate( GtkMenuItem *menuitem )
{
  uint8_t mode;
  const gchar *qso_items[] = { QSO_MODE_MENU_ITEMS };
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Abort if already a digital mode is active
  if( Flag[GUEST_RECEIVING] || Flag[GUEST_TRANSMITTING] )
    return;

  // Identify the menu item
  const gchar *label = gtk_menu_item_get_label( menuitem );
  for( mode = 0; mode < QSO_MODE_ITEMS; mode++ )
  {
    if( strcmp(label, qso_items[mode]) == 0 )
      break;
  }

  // Select action to take
  TRx->guest_modulation_mode = RX_MODE_ITEMS;
  switch( mode )
  {
    case FTX:
      if( ftx_gui.window == NULL )
        hermes2_gui.guest_window = Create_Ftx_Window( &ftx_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_FTX;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_FTX );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_FTX );
      break;

    case HELLSCHREIBER:
      if( hell_gui.window == NULL )
        hermes2_gui.guest_window = Create_Hell_Window( &hell_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_HELLM;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_HELLM );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_GUEST );
      break;

    case PSK31:
      if( psk31_gui.window == NULL )
        hermes2_gui.guest_window = Create_Psk31_Window( &psk31_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_PSK31U;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_PSK31U );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_GUEST );
      break;

    case RTTY:
      if( rtty_gui.window == NULL )
        hermes2_gui.guest_window = Create_Rtty_Window( &rtty_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_RTTYN;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_RTTYN );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_GUEST );
      break;

    case SSTV:
      if( sstv_gui.window == NULL )
        hermes2_gui.guest_window = Create_Sstv_Window( &sstv_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_SSTVU;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_SSTVU );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_SSTVU );
      break;

    case OLIVIA:
      if( olivia_gui.window == NULL )
        hermes2_gui.guest_window = Create_Olivia_Window( &olivia_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      TRx->guest_modulation_mode = RX_MODE_8_250;
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_8_250 );
      break;

    case MORSE_DECODER:
      if( morse_gui.window == NULL )
        hermes2_gui.guest_window = Create_Morse_Window( &morse_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_CWUN );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_RX_ONLY );
      break;

    case WEFAX_DECODER:
      if( wefax_gui.window == NULL )
        hermes2_gui.guest_window = Create_Wefax_Window( &wefax_gui.window_builder );
      if( hermes2_gui.guest_window == NULL ) break;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_RFAXU );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_RX_ONLY );
      break;

    case TIME_STATIONS:
      Time_Tx_Activate();
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_RX_ONLY );
      break;

    case WSPR:
      if( wspr_gui.window == NULL )
        hermes2_gui.guest_window = Create_Wspr_Window( &wspr_gui.builder );
      if( hermes2_gui.guest_window == NULL ) break;
      Rx_Modulation_Mode_Changed( Indices.TRx_Index, RX_MODE_WSPR );
      Tx_Modulation_Mode_Changed( Indices.TRx_Index, TX_MODE_RX_ONLY );
      break;

  } // switch( mode )
} // Guest_Modes_Menu_Item_Activate()

//----------------------------------------------------------------------

/* Rx_Modulation_Mode_Changed()
 *
 * Sets up the Receiver for the selected modulation mode
 */
  void
Rx_Modulation_Mode_Changed( uint8_t trx_index, uint8_t mode )
{
  gint
    rx_bw_idx  = 0,  // Demodulator bandwidth combobox index
    weaver_idx = 0;  // Weaver frequency combobox index

  // Remembers previous state of an index
  static gint xidx[MAX_RECEIVERS] = { -1, -1, -1, -1 } ;

  BOOLEAN
    offset,  // If Weaver offset of center freq is needed
    half;    // If Weaver frequency must be halved

  // Can happen when using Config dialog
  Transceiver_t *TRx;
  if( Transceiver[trx_index] ) TRx = Transceiver[trx_index];
  else return;

  // Clear play back buffer to avoid stutter etc
  if( hermes2_rc.sound_pc_local ) Clear_PCSound_Buf();

  // Get selected mode and display
  TRx->rx_modulation_mode = mode;
  Hermes2_Display_Mode( TRx, mode, RX_FLAG );

  // Set the relevant parameters for each demodulator mode
  offset = True;
  half   = True;
  TRx->rx_cw_modes = False;
  switch( TRx->rx_modulation_mode )
  {
    case RX_MODE_NBFM1:
      rx_bw_idx = RX_BW_25k;
      offset    = False;
      break;

    case RX_MODE_NBFM2:
      rx_bw_idx = RX_BW_12k;
      offset    = False;
      break;

    case RX_MODE_AMW:
      rx_bw_idx = RX_BW_8k;
      offset    = False;
      break;

    case RX_MODE_AMN:
      rx_bw_idx = RX_BW_4500;
      offset    = False;
      break;

  } // switch( hermes2_rc.modulation_mode )

  // Set the relevant parameters for each demodulator mode
  TRx->demodulator_init = True;
  switch( TRx->rx_modulation_mode )
  {
    case RX_MODE_NBFM1:
      TRx->demod_scale     = NBFM1_RX_SCALE;
      TRx->New_Demodulator = Demodulate_FM;
      break;

    case RX_MODE_NBFM2:
      TRx->demod_scale     = NBFM2_RX_SCALE;
      TRx->New_Demodulator = Demodulate_FM;
      break;

    case RX_MODE_AMW: case RX_MODE_AMN:
      TRx->demod_scale     = AM_RX_SCALE;
      TRx->New_Demodulator = Demodulate_AM;
      break;

    case RX_MODE_USBW:  case RX_MODE_USBM:  case RX_MODE_USBN:
    case RX_MODE_LSBW:  case RX_MODE_LSBM:  case RX_MODE_LSBN:
    case RX_MODE_SAMUW: case RX_MODE_SAMLW: case RX_MODE_RFAXU:
    case RX_MODE_SAMUM: case RX_MODE_SAMLM: case RX_MODE_RFAXL:
    case RX_MODE_SAMUN: case RX_MODE_SAMLN: case RX_MODE_RSID:
    case RX_MODE_SSTVU: case RX_MODE_SSTVL: case RX_MODE_FTX:
      TRx->demod_scale     = SSB_RX_SCALE;
      TRx->New_Demodulator = Demodulate_SSB;
      break;

    case RX_MODE_CWUW:   case RX_MODE_CWLW:   case RX_MODE_CWUM:
    case RX_MODE_CWLM:   case RX_MODE_CWUN:   case RX_MODE_CWLN:
    case RX_MODE_RTTYW:  case RX_MODE_RTTYM:  case RX_MODE_RTTYN:
    case RX_MODE_HELLW:  case RX_MODE_HELLM:  case RX_MODE_HELLN:
    case RX_MODE_PSK31U: case RX_MODE_PSK31L: case RX_MODE_WSPR:
    case RX_MODE_4_125:  case RX_MODE_4_250:  case RX_MODE_8_250:
    case RX_MODE_8_500:  case RX_MODE_16_500: case RX_MODE_16_1000:
    case RX_MODE_32_1000:
      half = False;
      TRx->demod_scale = CW_RX_SCALE;
      TRx->rx_cw_modes = True;
      TRx->New_Demodulator = Demodulate_CW;
      break;
  } // switch( hermes2_rc.modulation_mode )
  TRx->demodulator_exit = True;

  // Set the relevant parameters for each demodulator mode
  switch( TRx->rx_modulation_mode )
  {
    case RX_MODE_USBW: case RX_MODE_LSBW:
      weaver_idx = RX_WEAVER_3200;
      rx_bw_idx  = RX_BW_2800;
      break;

    case RX_MODE_SAMUW: case RX_MODE_SAMLW:
      weaver_idx = RX_WEAVER_5500;
      rx_bw_idx  = RX_BW_4800;
      break;

    case RX_MODE_USBM:  case RX_MODE_LSBM:
      weaver_idx = RX_WEAVER_3200;
      rx_bw_idx  = RX_BW_2600;
      break;

    case RX_MODE_SAMUM: case RX_MODE_SAMLM:
      weaver_idx = RX_WEAVER_4200;
      rx_bw_idx  = RX_BW_3600;
      break;

    case RX_MODE_USBN: case RX_MODE_LSBN:
      weaver_idx = RX_WEAVER_2800;
      rx_bw_idx  = RX_BW_2200;
      break;

    case RX_MODE_SAMUN: case RX_MODE_SAMLN:
      weaver_idx = RX_WEAVER_3000;
      rx_bw_idx  = RX_BW_2400;
      break;

    case RX_MODE_CWUW: case RX_MODE_CWLW:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_700;
      break;

    case RX_MODE_CWUM: case RX_MODE_CWLM:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_400;
      break;

    case RX_MODE_CWUN: case RX_MODE_CWLN:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_200;
      break;

    case RX_MODE_RFAXU: case RX_MODE_RFAXL:
      weaver_idx = RX_WEAVER_3800;
      rx_bw_idx  = RX_BW_2400;
      break;

    case RX_MODE_SSTVU: case RX_MODE_SSTVL:
      weaver_idx = RX_WEAVER_4000;
      rx_bw_idx  = RX_BW_2200;
      break;

    case RX_MODE_PSK31U: case RX_MODE_PSK31L:
      weaver_idx = RX_WEAVER_500;
      rx_bw_idx  = RX_BW_200;
      break;

    case RX_MODE_FTX:
      weaver_idx = RX_WEAVER_3200;
      rx_bw_idx  = RX_BW_3000;
      break;

    case RX_MODE_HELLW:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_400;
      break;

    case RX_MODE_HELLM:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_250;
      break;

    case RX_MODE_HELLN:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_100;
      break;

    case RX_MODE_RTTYW:
      weaver_idx = RX_WEAVER_1200;
      rx_bw_idx  = RX_BW_1000;
      break;

    case RX_MODE_RTTYM:
      weaver_idx = RX_WEAVER_800;
      rx_bw_idx  = RX_BW_600;
      break;

    case RX_MODE_RTTYN:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_250;
      break;

    case RX_MODE_4_125:
      weaver_idx = RX_WEAVER_562;
      rx_bw_idx  = RX_BW_150;
      break;

    case RX_MODE_4_250: case RX_MODE_8_250:
      weaver_idx = RX_WEAVER_625;
      rx_bw_idx  = RX_BW_300;
      break;

    case RX_MODE_8_500: case RX_MODE_16_500:
      weaver_idx = RX_WEAVER_750;
      rx_bw_idx  = RX_BW_600;
      break;

    case RX_MODE_16_1000: case RX_MODE_32_1000:
      weaver_idx = RX_WEAVER_1000;
      rx_bw_idx  = RX_BW_1200;
      break;

    case RX_MODE_WSPR:
      weaver_idx = RX_WEAVER_600;
      rx_bw_idx  = RX_BW_300;
      break;

    case RX_MODE_RSID:
      weaver_idx = RX_WEAVER_5000;
      rx_bw_idx  = RX_BW_4500;
      break;

  } // switch( hermes2_rc.modulation_mode )

  // Set the parameters to the relevant combo boxes
  gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->rx_bw_combobox), rx_bw_idx );

  // Hide Weaver offset combobox if not needed
  if( offset )
  {
    gtk_widget_show( TRx->rx_weaver_combobox );
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->rx_weaver_combobox), weaver_idx );
  }
  else
  {
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->rx_weaver_combobox), RX_WEAVER_NONE );
    gtk_widget_hide( TRx->rx_weaver_combobox );
  }

  // (Re)set the AFC check button
  if( Device[hermes2_rc.device_index].rcve_thread_run )
  {
    if( (TRx->rx_modulation_mode == RX_MODE_SAMUW) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMUM) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMUN) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLW) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLM) ||
        (TRx->rx_modulation_mode == RX_MODE_SAMLN) )
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->afc_checkbutton), TRUE );
    else
      gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(TRx->afc_checkbutton), FALSE );
  }

  /* Calculate weaver offset if weaver_idx
   * doesn't change, as going from USBW to LSBW */
  if( (xidx[TRx->index] == weaver_idx) && offset )
  {
    const char *weaver_frequency[] = { RX_WEAVER_FREQS };
    size_t len = strlen( weaver_frequency[weaver_idx] );

    // Go to numerals after space
    uint8_t off = 0;
    while( weaver_frequency[weaver_idx][off] != ' ' )
      off++;

    // For Non-CW modes, Weaver frequency is 1/2 of BFO frequency
    double freq = atof( &weaver_frequency[weaver_idx][off] );
    if( half ) freq /= 2.0;

    // If it is in kB, multiply by 1000
    if( weaver_frequency[weaver_idx][len - 3] == 'k' )
      freq *= 1000.0;

    // Set Weaver offset frequency
    switch( TRx->rx_modulation_mode )
    {
      case RX_MODE_USBW:  case RX_MODE_USBM:  case RX_MODE_USBN:
      case RX_MODE_SAMUW: case RX_MODE_SAMUM: case RX_MODE_SAMUN:
      case RX_MODE_RSID:  case RX_MODE_RFAXU: case RX_MODE_SSTVU:
      case RX_MODE_FTX:
        TRx->rx_weaver_offset = (int16_t)freq;
        break;

      case RX_MODE_LSBW:  case RX_MODE_LSBM:  case RX_MODE_LSBN:
      case RX_MODE_SAMLW: case RX_MODE_SAMLM: case RX_MODE_SAMLN:
      case RX_MODE_RFAXL: case RX_MODE_SSTVL:
        TRx->rx_weaver_offset = -(int16_t)freq;
        break;
    }

  } // if( (xidx == weaver_idx) && offset )

  xidx[TRx->index] = weaver_idx;
  if( !offset ) TRx->rx_weaver_offset = 0;

  // Set center freq to account for SSB freq offset if relevant
  if( Flag[HERMES2_RCCONFIG_SETUP] )
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

  // Set up Transmitter mode to match if Tx is linked
  if( TRx->link_tx_rx_freq )
  {
    if( mode >= RX_MODE_PSK31U )
      mode = TX_MODE_GUEST;
    else if( (mode >= RX_MODE_CWUW) )
      mode = TX_MODE_CW_INTERN;
    else if( (mode == RX_MODE_SAMUW) || (mode == RX_MODE_SAMUM) || (mode == RX_MODE_SAMUN) )
      mode = TX_MODE_USBM;
    else if( (mode == RX_MODE_SAMLW) || (mode == RX_MODE_SAMLM) || (mode == RX_MODE_SAMLN) )
      mode = TX_MODE_LSBM;

    // Other Rx modes have the same value with Tx modes
    Tx_Modulation_Mode_Changed( Indices.TRx_Index, mode );
  }

} // Rx_Modulation_Mode_Changed()

//----------------------------------------------------------------------

/* Tx_Modulation_Mode_Changed()
 *
 * Sets up the Transmitter for the selected modulation mode
 */
  void
Tx_Modulation_Mode_Changed( uint8_t trx_index, uint8_t mode )
{
  gint
    tx_bw_idx  = 0,  // Modulator bandwidth combobox index
    weaver_idx = 0;  // Weaver frequency combobox index

  // Remembers previous state of an index
  static gint xidx = -1;

  // If Weaver offset or modulator bandwidth is needed
  BOOLEAN offset = False, modbw = False;

  // Can happen when using Config dialog
  Transceiver_t *TRx;
  if( Transceiver[trx_index] ) TRx = Transceiver[trx_index];
  else return;

  // Get selected mode and display
  TRx->tx_modulation_mode = mode;
  TRx->tx_cw_modes        = False;
  Hermes2_Display_Mode( TRx, mode, TX_FLAG );

  // Hide mode-related widgets to start with
  gtk_widget_hide( TRx->tx_weaver_combobox );
  gtk_widget_hide( TRx->tx_bw_combobox );
  gtk_widget_hide( TRx->mic_hscale );
  gtk_widget_hide( TRx->mic_label );
  gtk_widget_hide( TRx->mic_check );

  // Disable CWX to start with. It should be disabled in any other mode but CWX.
  Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0F][1] &= 0xFE;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0F][1] &= 0xFE;
  Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
  Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
  Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;

  /* Set the relevant parameters for each modulator
   * mode and show widgets relevant to the mode */
  switch( TRx->tx_modulation_mode )
  {
    case TX_MODE_NBFM1:
      TRx->tx_fm_deviation      = NBFM1_DEVIATION;
      hermes2_rc.mic_lpf_cutoff = NBFM1_MIC_LPF;
      break;

    case TX_MODE_NBFM2:
      TRx->tx_fm_deviation      = NBFM2_DEVIATION;
      hermes2_rc.mic_lpf_cutoff = NBFM2_MIC_LPF;
      break;

    case TX_MODE_AMW:
      hermes2_rc.mic_lpf_cutoff = AMW_MIC_LPF;
      break;

    case TX_MODE_AMN:
      hermes2_rc.mic_lpf_cutoff = AMN_MIC_LPF;
      break;

    case TX_MODE_USBW: case TX_MODE_LSBW:
      weaver_idx = TX_WEAVER_3200;
      tx_bw_idx  = TX_BW_2800;
      hermes2_rc.mic_lpf_cutoff = SSBW_MIC_LPF;
      break;

    case TX_MODE_USBM: case TX_MODE_LSBM:
      weaver_idx = TX_WEAVER_3200;
      tx_bw_idx  = TX_BW_2600;
      hermes2_rc.mic_lpf_cutoff = SSBM_MIC_LPF;
      break;

    case TX_MODE_USBN: case TX_MODE_LSBN:
      weaver_idx = TX_WEAVER_2800;
      tx_bw_idx  = TX_BW_2200;
      hermes2_rc.mic_lpf_cutoff = SSBN_MIC_LPF;
      break;

    case TX_MODE_CW_EXTERN:
      // Enable CWX. It should be disabled in any other mode but CWX.
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0F][1] |= 0x01;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0F][1] |= 0x01;
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
      TRx->tx_cw_modes = True;
      break;

    case TX_MODE_CW_INTERN:
      // Disable CWX. It should be disabled in any other mode but CWX.
      Cmnd_Ctrl_Audio_IQ_Data.Cn1[0x0F][1] &= 0xFE;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2[0x0F][1] &= 0xFE;
      Cmnd_Ctrl_Audio_IQ_Data.Cn1_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
      Cmnd_Ctrl_Audio_IQ_Data.Cn2_address[Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx] = 0x0F;
      Cmnd_Ctrl_Audio_IQ_Data.addr_stack_idx++;
      TRx->tx_cw_modes = True;
      break;

    case TX_MODE_16_1000: case TX_MODE_32_1000:
      weaver_idx = TX_WEAVER_2000;
      tx_bw_idx  = TX_BW_1200;
      break;

    case TX_MODE_8_500: case TX_MODE_16_500:
      weaver_idx = TX_WEAVER_1500;
      tx_bw_idx  = TX_BW_600;
      break;

    case TX_MODE_4_250: case TX_MODE_8_250:
      weaver_idx = TX_WEAVER_1250;
      tx_bw_idx  = TX_BW_300;
      break;

    case TX_MODE_4_125:
      weaver_idx = TX_WEAVER_1125;
      tx_bw_idx  = TX_BW_200;
      break;

     case TX_MODE_FTX:
      weaver_idx = TX_WEAVER_3200;
      tx_bw_idx  = TX_BW_3000;
      break;

 } // switch( TRx->tx_modulation_mode )

  // Set the relevant parameters for each modulator mode
  switch( TRx->tx_modulation_mode )
  {
    case TX_MODE_NBFM1: case TX_MODE_NBFM2:
      modbw  = False;
      offset = False;
      Modulator = Modulate_FM;
      gtk_widget_show( TRx->mic_hscale );
      gtk_widget_show( TRx->mic_label );
      gtk_widget_show( TRx->mic_check );
      break;

    case TX_MODE_AMW: case TX_MODE_AMN:
      modbw     = False;
      offset    = False;
      Modulator = Modulate_AM;
      gtk_widget_show( TRx->mic_hscale );
      gtk_widget_show( TRx->mic_label );
      gtk_widget_show( TRx->mic_check );
      break;

    case TX_MODE_USBW: case TX_MODE_LSBW:
    case TX_MODE_USBM: case TX_MODE_LSBM:
    case TX_MODE_USBN: case TX_MODE_LSBN:
      modbw     = True;
      offset    = True;
      Modulator = Modulate_SSB;
      gtk_widget_show( TRx->mic_hscale );
      gtk_widget_show( TRx->mic_label );
      gtk_widget_show( TRx->tx_bw_combobox );
      gtk_widget_show( TRx->tx_weaver_combobox );
      gtk_widget_show( TRx->mic_check );
      break;

    case TX_MODE_CW_EXTERN:
      modbw     = False;
      offset    = False;
      Modulator = Modulate_CW;
      break;

    case TX_MODE_CW_INTERN:
      modbw     = False;
      offset    = False;
      Modulator = NULL;
      break;

    case TX_MODE_32_1000: case TX_MODE_16_1000:
    case TX_MODE_16_500:  case TX_MODE_8_500:
    case TX_MODE_8_250:   case TX_MODE_4_250:
    case TX_MODE_4_125:
      modbw     = True;
      offset    = True;
      Modulator = Modulate_Olivia;
      gtk_widget_show( TRx->tx_bw_combobox );
      gtk_widget_show( TRx->tx_weaver_combobox );
      break;

    case TX_MODE_FTX:
      modbw     = True;
      offset    = True;
      Modulator = NULL;
      gtk_widget_show( TRx->tx_bw_combobox );
      gtk_widget_show( TRx->tx_weaver_combobox );
      break;

    case TX_MODE_GUEST:
      modbw     = False;
      offset    = False;
      Modulator = NULL;
      break;

  } // switch( TRx->tx_modulation_mode )

  // Disable TXID when not needed
  switch( TRx->tx_modulation_mode )
  {
    case TX_MODE_NBFM1:     case TX_MODE_NBFM2:
    case TX_MODE_AMW:       case TX_MODE_AMN:
    case TX_MODE_USBW:      case TX_MODE_LSBW:
    case TX_MODE_USBM:      case TX_MODE_LSBM:
    case TX_MODE_USBN:      case TX_MODE_LSBN:
    case TX_MODE_CW_EXTERN: case TX_MODE_CW_INTERN:
    case TX_MODE_FTX:
    TRx->RSID_Mode = MODE_NULL;
    break;
  } // switch( TRx->tx_modulation_mode )

  // Set the parameters to the relevant combo boxes/
  gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->tx_weaver_combobox), TX_WEAVER_NONE );
  if( modbw )
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->tx_bw_combobox), tx_bw_idx );
  if( offset )
    gtk_combo_box_set_active( GTK_COMBO_BOX(TRx->tx_weaver_combobox), weaver_idx );

  /* Calculate weaver offset if weaver_idx
   * doesn't change, as going from USBW to LSBW */
  if( (xidx == weaver_idx) && offset )
  {
    const char *weaver_frequency[] = { TX_WEAVER_FREQS };
    size_t len = strlen( weaver_frequency[weaver_idx] );

    //  Weaver frequency is 1/2 of CFO frequency
    // Go to numerals
    uint8_t off = 0;
    while( weaver_frequency[weaver_idx][off] != ' ' )
      off++;
    double freq = atof( &weaver_frequency[weaver_idx][off] ) / 2.0;

    // If it is in kB, multiply by 1000
    if( weaver_frequency[weaver_idx][len-3] == 'k' )
      freq *= 1000.0;

    // Set Weaver offset frequency
    switch( TRx->tx_modulation_mode )
    {
      case TX_MODE_USBW: case TX_MODE_USBM:
      case TX_MODE_USBN: case TX_MODE_FTX:
        TRx->tx_weaver_offset = (int16_t)freq;
        break;

      case TX_MODE_LSBW: case TX_MODE_LSBM: case TX_MODE_LSBN:
        TRx->tx_weaver_offset = -(int16_t)freq;
        break;
    }

  } // if( (xidx == weaver_idx) && offset )

  // Set center freq to account for SSB freq offset if relevant
  if( Flag[HERMES2_RCCONFIG_SETUP] )
    Hermes2_Set_Center_Frequency( TRx, TX_FLAG );

  xidx = weaver_idx;
  if( !offset ) TRx->tx_weaver_offset = 0;

} // Tx_Modulation_Mode_Changed()

//----------------------------------------------------------------------

/* Frequency_Scroll_Event()
 *
 * Handles scroll events on the Frequency spin dial
 */
  gboolean
Frequency_Scroll_Event( GtkWidget *widget, const GdkEvent *event )
{
  const gchar *txt;
  int32_t  data;
  GtkLabel *label;
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];
  BOOLEAN rx_flag = False;

  // Determine Rx or Tx label
  const gchar *name = gtk_widget_get_name( widget );
  if( strstr(name, "rx") )
    rx_flag = True;
  else if( strstr(name, "tx") )
    rx_flag = False;

  // If Lock, abort if not changing Receiver frequency
  if( TRx->tx_freq_lock && !rx_flag )
    return( FALSE );

  // Get the label from the event box widget
  if( GTK_IS_BIN(widget) )
    label = GTK_LABEL( gtk_bin_get_child(GTK_BIN(widget)) );
  else return( FALSE );

  // Get text from label and convert to number
  txt  = gtk_label_get_text( label );
  data = (int32_t)atoi( txt );

  // Change value of label data according to scroll direction
  if( event->scroll.direction == GDK_SCROLL_UP )
    data++;
  else if( (event->scroll.direction == GDK_SCROLL_DOWN) )
    data--;

  // Display frequency and send to transceiver
  Display_Frequency_Spin_Dial( TRx, label, data );

  return( TRUE );
} // Frequency_Scroll_Event()

//----------------------------------------------------------------------

/* Frequency_Button_Press_Event()
 *
 * Handles button press events on the Frequency spin dial
 */
  gboolean
Frequency_Button_Press_Event( GtkWidget *widget, const GdkEventButton *event )
{
  const gchar *txt;
  int32_t  data;
  GtkLabel *label;
  BOOLEAN rx_flag = False;
  const Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  // Determine Rx or Tx label
  const gchar *name = gtk_widget_get_name( widget );
  if( strstr(name, "rx") )
    rx_flag = True;
  else if( strstr(name, "tx") )
    rx_flag = False;

  // If Lock, abort if not changing Receiver frequency
  if( TRx->tx_freq_lock && !rx_flag )
    return( FALSE );

  // Get the label from the event box widget
  if( GTK_IS_BIN(widget) )
    label = GTK_LABEL( gtk_bin_get_child(GTK_BIN(widget)) );
  else return( FALSE );

  // Get text from label and convert to number
  txt  = gtk_label_get_text( label );
  data = (int32_t)atoi( txt );

  /* Change value of label data
   * according to button pressed */
  if( event->button == 1 )
    data++;
  else if( event->button == 2 )
    data = 0;
  else if( (event->button == 3) )
    data--;

  // Display frequency and send to transceiver
  Display_Frequency_Spin_Dial( Transceiver[Indices.TRx_Index], label, data );

  return( TRUE );
} // Frequency_Button_Press_Event()

//----------------------------------------------------------------------

/* Bookmarks_Button_Clicked()
 *
 * Handles the on_bookmarks_button_clicked() callback
 */
  void
Bookmarks_Button_Clicked( gpointer user_data )
{
  // Create Bookmarks Window
  if( hermes2_gui.bmk_window == NULL )
  {
    gchar *object_ids[] = { HERMES2_STATIONS_WINDOW_IDS };
    Gtk_Builder(
        &hermes2_gui.bmk_window_builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    hermes2_gui.bmk_window =
      Builder_Get_Object( hermes2_gui.bmk_window_builder, "hermes2_bmk_window" );

    Set_Window_Geometry(
        hermes2_gui.bmk_window,
        hermes2_rc.bookmarks_window_x,
        hermes2_rc.bookmarks_window_y,
        hermes2_rc.bookmarks_window_width,
        hermes2_rc.bookmarks_window_height );

    gtk_widget_show( hermes2_gui.bmk_window );
    List_Bookmarks();

  } // if( (hermes2_bmk_window == NULL) && Num_of_TRXs )

  /* Get the Transceiver window's title and add
   * TRX/ADC naumber to Bookmarks window title */
  gchar str[32];
  const gchar *title;
  if( user_data )
  {
    title = gtk_window_get_title( GTK_WINDOW(user_data) );
    Indices.Bookmarks_TRx_Idx = (int8_t)Indices.TRx_Index;
  }
  else
  {
    title = gtk_window_get_title( GTK_WINDOW(Transceiver[0]->tcvr_window) );
    Indices.Bookmarks_TRx_Idx = 0;
  }

  title = strstr( title, "TRX" );
  snprintf( str, sizeof(str), _("Hermes2 %s Bookmarks"), title );
  gtk_window_set_title( GTK_WINDOW(hermes2_gui.bmk_window), str );

} // Bookmarks_Button_Clicked()

//----------------------------------------------------------------------

/* Pointer_Cross_Event()
 *
 * Handles the on_pointer_cross_event() callback
 */
  void
Pointer_Cross_Event( const GdkEventButton *event, gpointer data )
{
  gchar new_txt[HERMES2_MARKUP_SIZE];
  pointer_cross_label = GTK_LABEL( data );

  // Get the value of the scrolled/pointed-to Frequency label
  uint32_t value = (uint32_t)atoi( gtk_label_get_text(data) );

  if( event->type == GDK_ENTER_NOTIFY )
  {
    snprintf( new_txt, sizeof(new_txt), POINTER_ENTER_MARKUP, value );
    gtk_label_set_markup( pointer_cross_label, new_txt );
  }
  else if( event->type == GDK_LEAVE_NOTIFY )
  {
    snprintf( new_txt, sizeof(new_txt), POINTER_LEAVE_MARKUP, value );
    gtk_label_set_markup( pointer_cross_label, new_txt );
    pointer_cross_label = NULL;
  }

} // Pointer_Cross_Event()

//----------------------------------------------------------------------

/* Modes_Menu_Button_Clicked()
 *
 * Handles the on_*x_modes_menu_button_clicked() callback
 */
  void
Modes_Menu_Button_Clicked( BOOLEAN rx_flag )
{
  GtkBuilder *builder;
  GtkWidget  *modes_menu;

  if( rx_flag )
  {
    gchar *object_ids[] = { RX_MODES_MENU_IDS };
    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    modes_menu = Builder_Get_Object( builder, "rx_modes_menu");
  }
  else
  {
    gchar *object_ids[] = { TX_MODES_MENU_IDS };
    Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
    modes_menu = Builder_Get_Object( builder, "tx_modes_menu");
  }
  gtk_menu_popup_at_pointer( GTK_MENU(modes_menu), NULL );
  g_object_unref( builder );

} // Modes_Menu_Button_Clicked()

//----------------------------------------------------------------------

/* Guest_Modes_Menu_Button_Clicked()
 *
 * Handles the on_guest_modes_menu_button_clicked() callback
 */
  void
Guest_Modes_Menu_Button_Clicked( void )
{
  GtkBuilder *builder;
  GtkWidget *modes_menu;

  gchar *object_ids[] = { QSO_MODES_MENU_IDS };
  Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
  modes_menu = Builder_Get_Object( builder, "guest_modes_menu");
  gtk_menu_popup_at_pointer( GTK_MENU(modes_menu), NULL );
  g_object_unref( builder );

} // Guest_Modes_Menu_Button_Clicked()

//----------------------------------------------------------------------

/* Rsid_Modes_Menu_Button_Clicked()
 *
 * Handles the  callback
 */
  void
Rsid_Modes_Menu_Button_Clicked( void )
{
  GtkBuilder *builder;
  GtkWidget *modes_menu;

  gchar *object_ids[] = { RSID_MODES_MENU_IDS };
  Gtk_Builder( &builder, (gchar *)hermes2_rc.hermes2_glade, object_ids );
  modes_menu = Builder_Get_Object( builder, "rsid_modes_menu");
  gtk_menu_popup_at_pointer( GTK_MENU(modes_menu), NULL );
  g_object_unref( builder );

} // Rsid_Modes_Menu_Button_Clicked()

//----------------------------------------------------------------------

/* FFT_Rate_Combobox_Changed()
 *
 * Handles the on_fft_rate_combobox_changed callback
 */
  void
FFT_Rate_Combobox_Changed( GtkComboBox *combobox )
{
  gchar *txt = gtk_combo_box_text_get_active_text( GTK_COMBO_BOX_TEXT(combobox) );
  if( txt == NULL ) return;

  // Go to numerals
  uint8_t offset = 0;
  while( (txt[offset] != ' ') && (txt[offset] != '\0') ) offset++;
  Transceiver[Indices.TRx_Index]->spectrum_data.fft_frame_rate = (uint8_t)atoi( &txt[offset] );
  g_free( txt );

} // FFT_Rate_Combobox_Changed()

//----------------------------------------------------------------------

/* Squelch_Checkbutton_Toggled()
 *
 * Handles the on_squelch_checkbutton_toggled callback
 */
  void
Squelch_Checkbutton_Toggled( gboolean active )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  if( active )
  {
    TRx->squelch_on_status = True;
    GtkRange *range     = GTK_RANGE( Builder_Get_Object(TRx->builder, "squelch_hscale") );
    TRx->squelch_value  = SQUELCH_RANGE - (uint16_t)( gtk_range_get_value(range) );
    TRx->squelch_value *= SQUELCH_THRESHOLD;
  }
  else
  {
    TRx->squelch_on_status = False;
    TRx->squelch_value = SQUELCH_RANGE * SQUELCH_THRESHOLD;
  }

} // Squelch_Checkbutton_Toggled()

//----------------------------------------------------------------------


/* RxID_Checkbutton_Toggled()
 *
 * Handles the on_rx_rsid_enable_toggled callback
 */
  void
RxID_Checkbutton_Toggled( gboolean active )
{
  Transceiver_t *TRx = Transceiver[Indices.TRx_Index];

  if( active )
  {
    // Initialize RSID semaphore
    if( !Init_Semaphore(&rsid_semaphore, True) )
      return;

    // Create a thread to run RSID function
    TRx->rx_rsid_enable = True;
    RsId_Init();
  }
  else
  {
    // Post to RSID semaphore to exit RSID Receive
    TRx->rx_rsid_enable = False;
    int sval;
    sem_getvalue( &rsid_semaphore, &sval );
    if( !sval ) sem_post( &rsid_semaphore );
    RsId.Free();
  }

  // Display RSID listening status
  g_idle_add( RSID_Status, NULL );

} // RxID_Checkbutton_Toggled()

//----------------------------------------------------------------------

/* Discovery_Dialog_Response()
 *
 * Handles the on_discovery_dialog_response callback
 */
  void
Discovery_Dialog_Response( GtkDialog *dialog, gint response_id )
{
  switch( response_id )
  {
    case GTK_RESPONSE_YES:
      // Re-do device discovery
      g_idle_add( Do_Discovery, NULL );
      break;

    case GTK_RESPONSE_NO:
      Hermes2_Stop();
      if( hermes2_gui.window != NULL )
        gtk_widget_destroy( hermes2_gui.window);
      else
        gtk_main_quit();
      break;

    case GTK_RESPONSE_CLOSE:
      gtk_widget_destroy( hermes2_gui.discovery_dialog );
      if( hermes2_gui.window == NULL ) gtk_main_quit();
      break;
  }

} // Discovery_Dialog_Response()

//----------------------------------------------------------------------

