/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "bookmarks.h"
#include "callback_func.h"
#include "display.h"
#include "interface.h"
#include "../common/common.h"
#include "../common/shared.h"
#include "../common/transceiver.h"
#include "../common/utils.h"
#include "../hpsdr/settings.h"
#include <gtk/gtk.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//----------------------------------------------------------------------

#define HERMES2_BMK_LINE_LENGTH    37  // Bookmarks file, line length
#define HERMES2_BMK_NAME_WIDTH     13  // Bookmarks file, width in char of Name
#define HERMES2_BMK_FREQ_WIDTH      9  // Bookmarks file, width in char of Freq
#define HERMES2_BMK_MODE_WIDTH     11  // Bookmarks file, width in char of Mode

#define HERMES2_BOOKMARKS_FILE "/.hermes2/hermes2/bookmarks"

//----------------------------------------------------------------------

// Tree list store and treeview for hermes2 bookmarks window
GtkTreeStore *hermes2_bmk_store    = NULL;
GtkTreeView  *hermes2_bmk_treeview = NULL;

//----------------------------------------------------------------------

// Bookmarks tree columns
enum
{
  HERMES2_BMK_NAME_COL = 0,
  HERMES2_BMK_FREQ_COL,
  HERMES2_BMK_MODE_COL,
  HERMES2_BMK_TREE_NUM_COLS
};

//----------------------------------------------------------------------

/* cell_edited_callback()
 *
 * Text cell edited callback
 */
  static void
cell_edited_callback(
    GtkCellRendererText *cell,
    gchar               *path,
    gchar               *new_text,
    gpointer             user_data )
{
  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  guint column;

  // Set edited text into the appropriate column
  column = GPOINTER_TO_UINT( g_object_get_data(G_OBJECT(cell), "column") );
  selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(user_data) );
  gtk_tree_selection_get_selected( selection, &model, &iter );
  gtk_tree_store_set( GTK_TREE_STORE(model), &iter, column, new_text, -1 );

  Flag[HERMES2_SAVE_BOOKMARKS] = True;

} // cell_edited_callback()

//----------------------------------------------------------------------

/* Create_Tree_Store()
 *
 * Create store needed for the treeview
 */
  static void
Create_Tree_Store( void )
{
  GtkTreeModel *model;

  // Bookmarks tree column names
  char *tree_col_name[HERMES2_BMK_TREE_NUM_COLS] =
  {
    _("BOOKMARK   "),
    _("  FREQ-Hz"),
    _("     MODE")
  };

  // Create list stores only if needed
  if( hermes2_bmk_store != NULL ) return;

  // Create bookmarks tree store
  hermes2_bmk_store = gtk_tree_store_new(
      HERMES2_BMK_TREE_NUM_COLS,
      G_TYPE_STRING,
      G_TYPE_STRING,
      G_TYPE_STRING );

  /* Create cell renderers for Bookmark Name, Frequency and Mode */
  for( uint8_t idx = 0; idx < HERMES2_BMK_TREE_NUM_COLS; idx++ )
  {
    GtkTreeViewColumn *column;
    GtkCellRenderer   *renderer;

    // Cell render type "text"
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer), "editable", TRUE, NULL );
    g_object_set_data( G_OBJECT(renderer), "column", GUINT_TO_POINTER(idx) );
    g_signal_connect( renderer, "edited",
        (GCallback)cell_edited_callback, hermes2_bmk_treeview );

    // For the Name, alignment should be zero, others 1.0
    if( idx == 0 )
      gtk_cell_renderer_set_alignment( renderer, 0.0, 0.0 );
    else
      gtk_cell_renderer_set_alignment( renderer, 1.0, 0.0 );

    /* Create a column */
    column = gtk_tree_view_column_new_with_attributes(
        tree_col_name[idx], renderer, "text", idx, NULL );

    /* Add the column to the view */
    gtk_tree_view_append_column( GTK_TREE_VIEW(hermes2_bmk_treeview), column );
  }

  // Create a model to insert tree store
  Flag[HERMES2_BLOCK_GTK_SIGNALS] = True;
  model = GTK_TREE_MODEL( hermes2_bmk_store );
  gtk_tree_view_set_model( hermes2_bmk_treeview, model );

  // Set the selection mode to single
  GtkTreeSelection *selection = gtk_tree_view_get_selection( hermes2_bmk_treeview );
  gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

  // Destroy model automatically with view
  g_object_unref( model );
  Flag[HERMES2_BLOCK_GTK_SIGNALS] = False;

} // Create_Tree_Store()

//----------------------------------------------------------------------

/* List_Bookmarks()
 *
 * Reads bookmarks data from file and lists in tree view
 */
  void
List_Bookmarks( void )
{
  FILE *fp = NULL;     // Bookmarks file pointer
  GtkTreeIter iter1;   // Treeview parent iteration
  GtkTreeIter iter2;   // Treeview child iteration

  // Buffers one line of bookmarks data
  char line_buf[READ_LINE_BUF_SIZE + 1];
  char bookmarks_fpath[FILE_NAME_SIZE];

  // Create treeview pointer
  hermes2_bmk_treeview = GTK_TREE_VIEW(
      Builder_Get_Object( hermes2_gui.bmk_window_builder, "hermes2_bmk_treeview") );

  // Create the tree store
  Create_Tree_Store();

  // Setup file path to hermes2rc in working dir
  Strlcpy( bookmarks_fpath, getenv("HOME"), sizeof(bookmarks_fpath) );
  Strlcat( bookmarks_fpath, HERMES2_BOOKMARKS_FILE, sizeof(bookmarks_fpath) );

  // Open bookmarks file
  if( Open_File(&fp, bookmarks_fpath, "r") == FAILURE )
  {
    puts( bookmarks_fpath );
    perror( "!! hermes2: Failed to open Bookmarks File" );
    Error_Dialog( _("Failed to open Bookmarks File in ~/hermes2"), SHOW_OK );
    return;
  }

  // Clear all tree view
  gtk_tree_store_clear( hermes2_bmk_store );

  // Get new row if available
  gtk_tree_model_get_iter_first( GTK_TREE_MODEL(hermes2_bmk_store), &iter1 );

  // Fill bookmarks tree
  while( True )
  {
    int eof;              // End of File flag
    uint32_t start, end;  // Start and end character of strings in treeview

    // Read a line and separate entries
    eof = Read_Line( line_buf, fp, _("Bookmarks File") );
    if( eof == ERROR )
    {
      Error_Dialog( _("Error reading Bookmarks File"), SHOW_OK );
      return;
    }

    // Finish if [END] key received
    if( strcmp(line_buf, "[END]") == 0 ) break;

    // If line is the name of a group of bookmarks
    if( line_buf[0] == '[' )
    {
      // Look for closing ']'
      uint8_t idx = 1;
      while( idx++ < HERMES2_BMK_LINE_LENGTH )
        if( line_buf[idx] == ']' ) break;

      // Abort if ']' not found, abort
      if( idx == HERMES2_BMK_LINE_LENGTH )
      {
        Error_Dialog( _("Error reading Bookmarks File"), SHOW_OK );
        return;
      }

      // Append a row and enter Bookmark Group name
      line_buf[idx] = '\0';
      gtk_tree_store_append( hermes2_bmk_store, &iter1, NULL );
      gtk_tree_store_set( hermes2_bmk_store, &iter1, HERMES2_BMK_NAME_COL, &line_buf[1], -1 );
      continue;
    }

    /* Separate entries in the line by string
     * terminator and set data to tree store */
    gtk_tree_store_append( hermes2_bmk_store, &iter2, &iter1 );

    // Bookmark Name
    start = 0;
    end = HERMES2_BMK_NAME_WIDTH;
    line_buf[end] = '\0';
    gtk_tree_store_set( hermes2_bmk_store,
        &iter2, HERMES2_BMK_NAME_COL, &line_buf[start], -1 );

    // Frequency - Hz
    end += HERMES2_BMK_FREQ_WIDTH + 1;
    line_buf[end] = '\0';
    start += HERMES2_BMK_NAME_WIDTH + 1;
    gtk_tree_store_set( hermes2_bmk_store,
        &iter2, HERMES2_BMK_FREQ_COL, &line_buf[start], -1 );

    // Modulation Mode
    end += HERMES2_BMK_MODE_WIDTH + 1;
    line_buf[end] = '\0';
    start += HERMES2_BMK_FREQ_WIDTH + 1;
    gtk_tree_store_set( hermes2_bmk_store,
        &iter2, HERMES2_BMK_MODE_COL, &line_buf[start], -1 );

    // Get new row if available
    gtk_tree_model_iter_next( GTK_TREE_MODEL(hermes2_bmk_store), &iter2 );

  } // while( True )

  Flag[HERMES2_SAVE_BOOKMARKS] = False;
  Close_File( &fp );

} // List_Bookmarks()

//----------------------------------------------------------------------

/* Save_Bookmarks_File()
 *
 * Saves the data in bookmarks treeview to a given filename
 */
  void
Save_Bookmarks_File( void )
{
  FILE *fp = NULL;           // Bookmarks file pointer
  GtkTreeIter iter1, iter2;  // Iterations to bookmarks treeview

  // Buffer for one file line
  char file_line[HERMES2_BMK_LINE_LENGTH];
  gboolean ret = FALSE;  // Return value of functions
  char *string;          // String created from parameters
  uint32_t idx;
  char bookmarks_fpath[FILE_NAME_SIZE];
  int cnt;

  // Do not save if treeview is not valid or save flag not set
  if( (hermes2_bmk_treeview == NULL) || !Flag[HERMES2_SAVE_BOOKMARKS] )
    return;

  // Setup file path to hermes2rc in working dir
  Strlcpy( bookmarks_fpath, getenv("HOME"), sizeof(bookmarks_fpath) );
  Strlcat( bookmarks_fpath, HERMES2_BOOKMARKS_FILE, sizeof(bookmarks_fpath) );

  // Open bookmarks file for writing
  if( Open_File(&fp, bookmarks_fpath, "w") == FAILURE )
  {
    Error_Dialog( _("Failed to open Bookmarks File in ~/hermes2"), SHOW_OK );
    return;
  }

  // Print headers of file
  cnt = fprintf( fp, "# BOOKMARK NAME       FREQUENCY MODE\n" );
  if( cnt < 0 )
  {
    Fprintf_Error( fp, bookmarks_fpath );
    return;
  }

  // Bookmarks treeview model
  GtkTreeModel *model = GTK_TREE_MODEL( hermes2_bmk_store );

  // Get the first parent's iteration
  ret = gtk_tree_model_get_iter_first( model, &iter1 );
  if( !ret )
  {
    Close_File( &fp );
    Error_Dialog( _("gtk_tree_model_get_iter_first() failed"), SHOW_OK );
    return;
  }

  // Go through the tree store rows and columns and save data
  gboolean next_child = True;
  BOOLEAN next_parent = True;
  while( ret )
  {
    size_t len;
    uint8_t i;

    // Clear file line
    for( i = 0; i < HERMES2_BMK_LINE_LENGTH; i++ )
      file_line[i] = ' ';

    // Get the Bookmark Group nameFILE *
    if( next_parent )
    {
      gtk_tree_model_get( model, &iter1, HERMES2_BMK_NAME_COL, &string, -1 );
      len = strlen( string );
      if( len > HERMES2_BMK_LINE_LENGTH )
        len = HERMES2_BMK_LINE_LENGTH;

      // Remove leading spaces and print to bookmarks file
      i = 0;
      while( (i < len) && (string[i] == ' ') )
      {
        i++;
        len--;
      }
      cnt = fprintf( fp, "[%s]\n", &string[i] );
      g_free( string );
      if( cnt < 0 )
      {
        Fprintf_Error( fp, bookmarks_fpath );
        return;
      }

      // Get the first child node
      next_child = gtk_tree_model_iter_children( model, &iter2, &iter1 );
      if( !next_child )
      {
        Close_File( &fp );
        Error_Dialog( _("gtk_tree_model_iter_children() failed"), SHOW_OK );
        return;
      }

      next_parent = False;
      continue;

    } // if( next_parent )

    if( next_child )
    {
      size_t limit;

      // Get the Bookmark's name
      gtk_tree_model_get( model, &iter2, HERMES2_BMK_NAME_COL, &string, -1 );
      len = strlen( string );
      limit = HERMES2_BMK_NAME_WIDTH;
      if( len > limit ) len = limit;

      // Remove leading spaces and print to bookmarks file
      i = 0;
      while( (i < len) && (string[i] == ' ') )
      {
        i++;
        len--;
      }
      Strlcpy( file_line, &string[i], len + 1 );
      file_line[len] = ' ';
      idx = HERMES2_BMK_NAME_WIDTH + 1;
      g_free( string );

      // Get the Bookmark's frequency
      gtk_tree_model_get( model, &iter2, HERMES2_BMK_FREQ_COL, &string, -1 );
      uint32_t value = (uint32_t)atoi( string );
      g_free( string );

      // Format value before writing to file
      snprintf( &file_line[idx], HERMES2_BMK_FREQ_WIDTH + 1,
                "%*u", HERMES2_BMK_FREQ_WIDTH, value );
      idx += HERMES2_BMK_FREQ_WIDTH;
      file_line[idx] = ' ';
      idx++;

      // Get and write the station Mode
      gtk_tree_model_get( model, &iter2, HERMES2_BMK_MODE_COL, &string, -1 );
      len = strlen( string );

      // Remove leading spaces and print
      i = 0;
      while( (i < len) && (string[i] == ' ') ) i++;
      snprintf( &file_line[idx], HERMES2_BMK_MODE_WIDTH + 1, "%s", &string[i] );
      g_free( string );

      // Write line to bookmarks file
      file_line[HERMES2_BMK_LINE_LENGTH - 1] = '\0';
      cnt = fprintf( fp, "%s\n", file_line );
      if( cnt < 0 )
      {
        Fprintf_Error( fp, bookmarks_fpath );
        return;
      }

      // Get next child node and continue if valid
      next_child = gtk_tree_model_iter_next( model, &iter2 );
      if( next_child ) continue;

    } // if( next_child )

    ret = gtk_tree_model_iter_next( model, &iter1 );
    next_parent = True;

  } // while( ret )

  // Write the end of file marker
  cnt = fprintf( fp, "[END]\n#" );
  if( cnt < 0 )
  {
    Fprintf_Error( fp, bookmarks_fpath );
    return;
  }

  Flag[HERMES2_SAVE_BOOKMARKS] = False;
  Close_File( &fp );

  return;
} // Save_Bookmarks_File()

//----------------------------------------------------------------------

/* Select_Treeview_Row()
 *
 * Selects a bookmarks treeview row on user click
 * on up or down buttons in bookmarks window
 */
  void
Select_Treeview_Row( uint8_t direction )
{
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;

  // Create a selection object and find selected node
  selection = gtk_tree_view_get_selection( hermes2_bmk_treeview );
  if( !gtk_tree_selection_get_selected(selection, &model, &iter) )
    return;

  // Move up or down in tree view
  gboolean ret = FALSE;
  if( direction == TREEVIEW_DOWN )
    ret = gtk_tree_model_iter_next( model, &iter );
  else if( direction == TREEVIEW_UP )
    ret = gtk_tree_model_iter_previous( model, &iter );

  // Select new node, if one is found
  if( ret )
  {
    // Simulate a button press on treeview
    gtk_tree_selection_select_iter( selection, &iter );
    Bookmarks_Cursor_Changed( hermes2_bmk_treeview );
  }

} // Select_Treeview_Row()

//----------------------------------------------------------------------

/* Bookmarks_Cursor_Changed()
 *
 * Handles cursor changed event on bookmarks treeview.
 * Reads the selected row and sets up receiver center
 * frequency and modulation mode if they are changed
 */
  void
Bookmarks_Cursor_Changed( GtkTreeView *treeview )
{
  // Bookmarks treeview selection objects
  GtkTreeSelection *selection;
  GtkTreeModel     *model;
  GtkTreeIter       iter;
  gchar *name;            // Bookmark name
  gchar *value;           // Value from treeview strings
  int idx, len;
  const char *modulation_modes[] = { RX_MODE_MENU_ITEMS };
  gchar title[256];
  uint32_t center_freq;
  static int modulation_mode = -1;

  // Abort if Bookmarks List is not attached to a Transceiver
  if( Indices.Bookmarks_TRx_Idx < 0 ) return;
  Transceiver_t *TRx = Transceiver[Indices.Bookmarks_TRx_Idx];

  // Get the selected row when ready
  if( Flag[HERMES2_BLOCK_GTK_SIGNALS] ) return;
  selection = gtk_tree_view_get_selection( treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter );

  // Ignore parent nodes
  gint depth = gtk_tree_store_iter_depth( GTK_TREE_STORE(model), &iter );
  if( depth < 1 ) return;

  // Get the station name
  gtk_tree_model_get( model, &iter, HERMES2_BMK_NAME_COL, &name, -1 );

  // Remove trailing spaces
  len = (int)strlen( name ) - 1;
  for( idx = len; idx >= 0; idx-- )
    if( name[idx] != ' ' ) break;
  name[idx + 1] = '\0';

  // Set the title of the main window. Only one ADC (adc 0) in HL2.
  snprintf( title, sizeof(title), "Hermes2 TRX%d/ADC%d : %s", TRx->index, 0, name );
  gtk_window_set_title( GTK_WINDOW(TRx->tcvr_window), title );
  g_free( name );

  // Get the RX frequency
  gtk_tree_model_get( model, &iter, HERMES2_BMK_FREQ_COL, &value, -1 );
  if( strstr(value, "--") ) return;
  center_freq = (uint32_t)atoi( value );
  g_free( value );

  // Get the station Mode
  gtk_tree_model_get( model, &iter, HERMES2_BMK_MODE_COL, &value, -1 );
  if( strstr(value, "--") ) return;

  // Bypass any leading spaces
  gchar *tmp = value;
  while( tmp[0] == ' ' ) tmp++;

  // Match tree value to modes array
  for( idx = 0; idx < RX_MODE_ITEMS; idx++ )
  {
    if( strcmp(modulation_modes[idx], tmp) == 0 )
      break;
  }
  g_free( value );

  // Show error message once if mode not recognized
  if( (idx == RX_MODE_ITEMS) && (modulation_mode != idx) )
    Error_Dialog( _("Unknown Modulation Mode in Bookmarks"), SHOW_OK );
  modulation_mode = idx;

  // Set up new frequency on change
  if( TRx->rx_frequency != center_freq )
  {
    // Set the Rx Band combobox first
    Set_Rx_Band( TRx, center_freq );

    // Enter the new Receiver frequency
    TRx->rx_frequency = center_freq;
    Update_Spin_Dial( TRx, RX_FLAG );
    Hermes2_Set_Center_Frequency( TRx, RX_FLAG );

    // Link Tx frequency to Rx
    if( TRx->link_tx_rx_freq && !TRx->tx_freq_lock )
    {
      TRx->tx_frequency = center_freq;
      Update_Spin_Dial( TRx, TX_FLAG );
      Hermes2_Set_Center_Frequency( TRx, TX_FLAG );
    }
  }

  // Set up new mode if valid
  if( modulation_mode != RX_MODE_ITEMS )
  {
    Indices.TRx_Index = (uint8_t)Indices.Bookmarks_TRx_Idx;
    Rx_Modulation_Mode_Changed(
        (uint8_t)Indices.Bookmarks_TRx_Idx, (uint8_t)modulation_mode );
  }

} // Bookmarks_Cursor_Changed()

//----------------------------------------------------------------------

/* Treeview_Button_Press()
 *
 * Handles the bookmarks treeview button press callback
 */
  gboolean
Treeview_Button_Press( const GdkEventButton *event )
{
  // Enter frequency and mode data into treeview row
  if( event->button == 2 )
  {
    // Bookmarks treeview selection objects
    GtkTreeSelection *selection;
    GtkTreeModel     *model;
    GtkTreeIter       iter;
    const char *modulation_modes[RX_MODE_ITEMS] = { RX_MODE_MENU_ITEMS };

    // Data from bookmarks treeview
    gchar freq[HERMES2_BMK_FREQ_WIDTH + 1];
    gchar mode[HERMES2_BMK_MODE_WIDTH + 1];


    // Abort if Bookmarks List is not attached to a Transceiver
    if( Indices.Bookmarks_TRx_Idx < 0 ) return( FALSE );
    const Transceiver_t *TRx = Transceiver[Indices.Bookmarks_TRx_Idx];

    // Get the selected row, abort if none
    selection = gtk_tree_view_get_selection( hermes2_bmk_treeview );
    if( !gtk_tree_selection_get_selected(selection, &model, &iter) )
      return( FALSE );

    // Get the depth of selected row
    gint depth = gtk_tree_store_iter_depth( GTK_TREE_STORE(model), &iter );

    // Ignore parent nodes
    if( depth == 0 ) return( FALSE );

    // Create data to insert to treeview
    snprintf( freq, sizeof(freq), "%*u",
        HERMES2_BMK_FREQ_WIDTH, TRx->rx_frequency );
    snprintf( mode, sizeof(mode), "%*s",
        HERMES2_BMK_MODE_WIDTH, modulation_modes[TRx->rx_modulation_mode] );

    // Set data to tree store
    gtk_tree_store_set(
        hermes2_bmk_store,    &iter,
        HERMES2_BMK_FREQ_COL, freq,
        HERMES2_BMK_MODE_COL, mode,
        -1 );

    return( TRUE );
  } // if( event->button == 2 )

  return( FALSE );
} // Treeview_Button_Press()

//----------------------------------------------------------------------

/* New_Bookmarks_Row()
 *
 * Adds a new row to the bookmarks treeview
 */
  void
New_Bookmarks_Row( void )
{
  // Bookmarks treeview selection objects
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter, sibling;
  uint8_t ncols, idx;
  gint depth;

  if( hermes2_bmk_treeview == NULL ) return;

  // Find selected row and add new after
  selection    = gtk_tree_view_get_selection( hermes2_bmk_treeview );
  gboolean ret = gtk_tree_selection_get_selected( selection, &model, &sibling );
  ncols        = (uint8_t)(gtk_tree_model_get_n_columns(model) );
  if( !ret ) // No rows in tree view
  {
    // Insert first parent and child row
    gtk_tree_store_insert( GTK_TREE_STORE(model), &iter, NULL, 0 );
    gtk_tree_store_insert( GTK_TREE_STORE(model), &sibling, &iter, 0 );

    // Prime columns of parent and child
    gtk_tree_store_set( GTK_TREE_STORE(model), &iter, 0, "---", -1 );
    for( idx = 0; idx < ncols; idx++ )
      gtk_tree_store_set( GTK_TREE_STORE(model), &sibling, idx, "---", -1 );

    return;
  }
  else
  {
    // If current node is a parent node, create a new parent with a default child
    depth = gtk_tree_store_iter_depth( GTK_TREE_STORE(model), &sibling );
    if( depth < 1 )
    {
      gtk_tree_store_insert_after( GTK_TREE_STORE(model), &iter, NULL, &sibling );
      gtk_tree_store_insert( GTK_TREE_STORE(model), &sibling, &iter, 0 );

      // Prime columns of parent and child
      gtk_tree_store_set( GTK_TREE_STORE(model), &iter, 0, "---", -1 );
      for( idx = 0; idx < ncols; idx++ )
        gtk_tree_store_set( GTK_TREE_STORE(model), &sibling, idx, "---", -1 );
    }
    else  // If sibling is a child, add a new default child
    {
      gtk_tree_store_insert_after( GTK_TREE_STORE(model), &iter, NULL, &sibling );

      // Prime columns of child
      for( idx = 0; idx < ncols; idx++ )
        gtk_tree_store_set( GTK_TREE_STORE(model), &iter, idx, "---", -1 );
    }
  } // if( !ret )

} // New_Bookmarks_Row()

//----------------------------------------------------------------------

/* Delete_Bookmarks_Row()
 *
 * Deletes a selected row in the bookmarks treeview
 */
  void
Delete_Bookmarks_Row( void )
{
  // Bookmarks treeview selection objects
  GtkTreeModel *model;
  GtkTreeIter   iter;
  GtkTreeSelection *selection;

  if( hermes2_bmk_treeview == NULL ) return;
  selection = gtk_tree_view_get_selection( hermes2_bmk_treeview );
  gtk_tree_selection_get_selected( selection, &model, &iter);
  gtk_tree_store_remove( GTK_TREE_STORE(model), &iter );

  Flag[HERMES2_SAVE_BOOKMARKS] = True;

} // Delete_Bookmarks_Row()

//----------------------------------------------------------------------

